; file: saha_boltzmann_frac.pro = return fractional Saha-Boltzmann population
; init: Sep  5 2015  Rob Rutten  Freiburg
; last: Sep  5 2015  Rob Rutten  Freiburg
; note: cut out from extline_chianti.pro
;       should work for any stage of first 30 elements

;===========================================================================
function saha_boltzmann_frac,$
  temp,eldens,nhtot,elemnr,ionstage,excerg,glow

;+
  ;; frac=extline_chianti($
  ;;    temp,eldens,nhtot,$    ; arrays(h), cgs, total H
  ;;    elemnr,ionstage,$      ; neutral=0
  ;;    excerg,glow            ; excerg in eV
  ;; returns float array frac(h)
;-

; common in WSA programs 
common common_atomdata_wsa,at_weight,at_abu,$
  at_ionerg1,at_ionerg2,at_ionerg3,at_sym
;RR ionerg values not used below 

if (n_params() lt 7) then begin 
  print,' ext=extline_chianti($'
  print,'   temp,eldens,nhtot,$    ; arrays(h), cgs, total H'
  print,'   elemnr,ionstage,$      ; neutral=0'
  print,'   excerg,glow            ; excerg in eV'
  print,' returns float array(h)'
  return,-1
endif

; checks
if (elemnr gt 30) then begin
  print,' #### ABORT extline_chianti: element not in Chianti (only 1-30)'
  return,-2
endif
if (ionstage eq elemnr) then return, 1.   ; naked ion has pf=1?
if (ionstage gt elemnr) then begin
  print,' #### ABORT extline_chianti: ionstage > elemnr'
  return,-3
endif

; physics constants
hcgs=6.62607D-27              ; Planck constant (erg s)
ccgs=2.99792458D10            ; velocity of light (cm/s)
kcgs=1.38062D-16              ; B;; oltzmann constant in erg/deg
kev=8.61734D-5                ; Boltzmann constant (eV/deg)
hckcgs=hcgs*ccgs/kcgs         ; hc/k in cgs
matom=1.6605D-24              ; atomic mass constant 
melectron=9.1094E-28          ; electron mass in gram
eelectron=4.803204E-10        ; electron charge in cgs

; get and check array sizes (height or z in 1-D atmosphere)  
ntemp=n_elements(temp)
neldens=n_elements(eldens)
nnhtot=n_elements(nhtot)
if (neldens ne ntemp) then print,' #### temp, eldens not same dim'
if (nnhtot ne ntemp) then print,' #### temp, nhtot not same dim'
natmos=max([ntemp,neldens,nnhtot])

; get _wsa atom parameters 
atomdata_wsa,elemnr,atomwgt,abund,ion1,ion2,ion3,elemsymbol
;; ionerg=[ion1,ion2] ; -wsa has only first three stages

; get ionization energy from Chianti file
read_ip,!xuvtop+'/ip/chianti.ip',ip,ref
ionerg=ip[elemnr-1,ionstage]
; convert ionization energy from cm^-1 into eV
ergpercmwav=1.98649E-16   ; erg per cm-1 wave number
ergperev=1.60217657E-12   ; erg per electronvolt
ionerg=ionerg*ergpercmwav/ergperev

; gas pressure (including all neutral elements in common_atomdata_wsa)
totabu=total(at_abu)
pgas=(totabu*nhtot+eldens)*kcgs*temp

; partition function (array if temp is array) from chianti tables 
parfun=partfunc_chianti(elemnr,ionstage,temp)

; if partfunc_chianti error (no file or small nr of levels) and ionstage<3
if (parfun[0] lt 0 and ionstage lt 3) then begin
;;  print,' ==== partfunc_chianti = '+ntostr(parfun)+'; switch to partfunc_rr'
  parfun0=partfunc_rr(temp,elemnr,0)
  parfun1=partfunc_rr(temp,elemnr,1)
  parfun2=partfunc_rr(temp,elemnr,2)
  if (ionstage eq 0) then parfun=parfun0 
  if (ionstage eq 1) then parfun=parfun1
endif
; abort at fatal error (no Chianti and no wsa)
if (parfun[0] lt 0) then begin 
  print,' ##### ABORT extline_chianti: no partition function for stage = '+$
    ntostr(ionstage)
  return, -3
endif

; Saha 
ionfrac=saha_one(elemnr,temp,eldens,ionstage)
ionfrac=reform(ionfrac)

; Boltzmann ratio
  bolrat=(glow/parfun)*exp(-excerg/(kev*temp))

;; extlinetot= $
;;   1.E8*!pi*eelectron^2/(melectron*ccgs) $          ; units total = cm^-1 AA
;;   *vacwavcm^2/ccgs $                               ; wavelength units
;;   *oscf*(glow/parfun)*ionfrac*abund*nhtot*blow $   ; lower level density
;;   *exp(-excerg/(kev*temp)) $                       ; Boltzmann
;;   *(1.-(bup/blow)*exp(-hckcgs/(vacwavcm*temp)))    ; stimulated emission


return,ionfrac*bolrat
end

; ============= main to test per IDLWAVE Hyper-c ========================

; put plot window besides emacs frame
window,xpos=0,ypos=380,xsi=630

; read FALC 
falcfile='/home/rutten/rr/web/rjr-edu/exercises/ssb/falc.dat'  
readcol,falcfile,height,tau5,colmass,temp,vmicro,nhtot,nprot,eldens,ptot,$
    pgas_ptot,rho,skipline=4

; get Halpha line parameters
read_line_params,6563,$
  linewav,elemnr,ionstage,excerg,$
  glow,gup,llow,lup,oscf,logc4

; compute, plot, and check 
frac=saha_boltzmann_frac(temp,eldens,nhtot,elemnr,ionstage,excerg,glow)
plot,height,frac,/ylog
;RR looks OK

end
