; file: saha_all.pro = evaluate Saha for all stages of an element
; init: Mar 28 2015  Rob Rutten  Deil
; last: Apr  2 2015  Rob Rutten  Deil

function saha_all,elemnr,temp,eldens,$
  ionmin=ionmin,ionmax=ionmax,verbose=verbose

;+
 ; NAME:
 ;   saha_all  
 ; PURPOSE:
 ;   compute LTE ionization fractions with Saha for many stages
 ; DESCRIPTION:
 ;   mostly uses Chianti data in parfunc_chianti.pro and saha_ratio.pro
 ; CALL:
 ;    saha=saha_all(elemnr,temp,eldens,ionmax=ionmax,verbose=verbose)
 ; INPUTS:
 ;   elemnr = element number (hydrogen=1)
 ;   temp = temperature [K] (scalar or array)
 ;   eldens = electrion density [cm^-3] (scalar or same-size array)
 ; OPTIONAL KEYWORD INPUTS:
 ;   ionmin = start at this ionization stage (default 0)
 ;   ionmax = up to this ionization stage (default all)
 ;   verbose 1/0: print intermediate results
 ; OUTPUTS:
 ;   single float array [ionmin:ionmax] for scalar temp and eldens
 ;   double foat array [ionmin:ionmax,0:natmos] for array temp and/or eldens
 ; OPTIONAL KEYWORD OUTPUT
 ;   ionmin = start at this ionization stage (default 0)
 ;   ionmax = stop at this ionization stage  (default nrelem)
 ;     NB: when using one or both be sure to bracket the dominant stage
 ;   verbose 1/0: print intermediate results (default 0)
 ; RESTRICTIONS:
 ;   needs SSW Chianti and other Rob Rutten ltelib programs
 ;   maybe some Chianti data availability issues   
 ; HISTORY:
 ;   Mar 29 2015 RR: start
 ;   Apr  2 2015 RR: accept arrays for temp and/or eldens
;-

; answer no-parameter query 
if (n_params(0) lt 3) then begin
  print,'xx=saha_all(elemnr,temp,eldens,$'
  print,'  ionmin=ionmin,ionmax=ionmax,verbose=verbose)'
  return,-1   ;RR return,-1 for a function
endif

; defaults for keywords
if (not keyword_set(ionmin)) then ionmin=0
if (not keyword_set(ionmax)) then ionmax=elemnr
if (not keyword_set(verbose)) then verbose=0

; array sizes
ntemp=n_elements(temp)
neldens=n_elements(eldens)
natmos=max([ntemp,neldens])  ;RR need to add [] took me ages...
nions=ionmax+1-ionmin   ; add neutral atom 
resultarr=fltarr(nions,natmos)

; various checks
if (elemnr gt 30) then begin
  print,' #### ABORT saha_all: element not in Chianti (only 1-30)'
  return,-2
endif
if (ionmax gt elemnr) then ionmax=elemnr
if (ntemp gt 1 and neldens gt 1 and ntemp ne neldens) then begin
  print,' #### ABORT saha_all: temp and eldens inequal array sizes' 
  return,-3
endif

; fix when only one of temp and eldens is an array
if (ntemp gt 1 and neldens eq 1) then begin
  eldens0=eldens
  eldens=fltarr(natmos)+eldens0 
endif
if (ntemp eq 1 and neldens gt 1) then begin
  temp0=temp
  temp=fltar(natmos)+temp0
endif

; set minimum value to avoid division by insecure small numbers
lowcut=1.E-20

; start huge loop over atmos entries (multiple temp and/or eldens) 
for iatmos=0,natmos-1 do begin

; get successive Saha ratios
ratarr=dblarr(nions)+lowcut
for iion=0,nions-2 do begin
  ratarr[iion]=1.D0*saha_ratio(elemnr,temp[iatmos],$
    eldens[iatmos],iion+ionmin)
;RR next also takes care of -3 returns "no partition function"
    if (ratarr[iion] lt lowcut) then ratarr[iion]=lowcut
endfor
if (verbose) then print,' ratarr = ',ntostr(ratarr,format='(G15.3)')

; start iteration loop including inorm definition 
; (needed to to avoid the fake non-valid low-ratio cutoffs in saha_ratio)
for iter=0,3 do begin

; first try is not peak population but hopefully is above fake cutoff
  if (iter eq 0) then begin
;;;    ratarr=finite(ratarr) ;RR @@@@@@@@
    peakrat=where(ratarr eq max(ratarr))  
    inorm=peakrat[0]
    if (inorm lt ionmax) then inorm=inorm+1   ; safer bet

; next try: Saha peak ion from the previous iteration
  endif else begin
    peaksaha=where(fracarr eq max(fracarr))
    inormnew=peaksaha[0]

; if no change in inorm then discard iteration redo
    if (inormnew eq inorm) then break else inorm=inormnew
  endelse
  if (verbose) then print,' inorm = '+ntostr(inorm)

; now follow Gray III (1.23) page 18 evaluating N_tot/N_inorm (1 in Gray)

; chain successivily multiplied inverse ratio products below inorm
  mularr=dblarr(nions)
  mularr[inorm]=1.  ; start value = N_inorm/N_inorm
  for iion=inorm-1,0,-1 do begin
    mularr[iion]=mularr[iion+1]*(1./ratarr[iion])
    if (mularr[iion] lt lowcut) then mularr[iion]=lowcut
  endfor

; chain successivily multiplied ratio products above inorm
  for iion=inorm+1,nions-1 do begin
    mularr[iion]=mularr[iion-1]*ratarr[iion-1]
    if (mularr[iion] lt lowcut) then mularr[iion]=lowcut
  endfor
  
  if (verbose) then print,' mularr = ',ntostr(mularr,format='(G15.3)')

; now evaluate all individual fractions similarly 

  fracarr=dblarr(nions)
  fracarr[inorm]=1./total(mularr)
  if (verbose) then print, ' fracarr[inorm] = '+ntostr(fracarr[inorm])  

; chain-evalute the fractions below i=inorm
  for iion=inorm-1,0,-1 do begin
    fracarr[iion]=fracarr[iion+1]*(1./ratarr[iion])
    if (fracarr[iion] lt lowcut) then fracarr[iion]=lowcut
  endfor

; chain-evalute the fractions above i=inorm 
  for iion=inorm+1,nions-1 do begin
    fracarr[iion]=fracarr[iion-1]*ratarr[iion-1]
    if (fracarr[iion] lt lowcut) then fracarr[iion]=lowcut
  endfor

  if (verbose) then print,' fracarr = ',ntostr(fracarr,format='(G15.3)')
  
endfor ; end of iteration loop for improving inorm

; fill this atmos
  resultarr[*,iatmos]=fracarr[*]

endfor ; end of iteration loop over multiple temp and or eldens array values

; done
return,resultarr
end


; =============== main for testing per IDLWAVE H-c ======================

elemnr=26 ; Fe
elemnr=56 ; Ba (error mssage test, not in Chianti)
elemnr=20 ; Ca
elemnr=14 ; Si
elemnr=1  ; H 

elemnr=14 ; Si
ionmin=0
ionmax=6
temp=[5000.,6000.,7000.] ; test array usage
temp=25000.
eldens=[1.E14,1.E13,1.E12] ; test array usage
eldens=1.E14  ; read off for temp value in fig:extLTE
verbose=0

sahafrac=saha_all(elemnr,temp,eldens,ionmin=ionmin,ionmax=ionmax,$
  verbose=verbose)
print,' ==== saha_all: ',ntostr(sahafrac,format='(G15.3)')

; check against saha_rr.pro (first three stages only)
atomdata_wsa,elemnr,atomwgt,abund,ion1,ion2,ion3,elemsymbol
u0=partfunc_rr(temp,elemnr,0)
u1=partfunc_rr(temp,elemnr,1)
u2=partfunc_rr(temp,elemnr,2)
saha_rr,temp,eldens,ion1,ion2,u0,u1,u2,n0_ntot,n1_ntot,n2_ntot
print,' ==== saha_rr:  ',ntostr([n0_ntot,n1_ntot,n2_ntot],format='(G15.3)')

end
