; file: partfunc_rr.pro = partition function using Gray table 6.2
; init: Sep 28 2013  Rob Rutten
; last: Oct  5 2013  Rob Rutten
; note: mostly from partfunc_nv.pro by Nikola Vitas


;+
function partfunc_rr,temp,elemnr,ionstage,$
  gray=gray,wittmann=wittmann,log=log,verbose=verbose

 ; PURPOSE:
 ;   return the partition function for ionization stage 0
 ;   (neutral), 1 (once ionized) or 2 (doubly ionized) for any element
 ;   as function of temperature.  The interpolation is linear in the
 ;   logarithmic tabulation in Aoppendix D2 in D.F. Gray, "The
 ;   observations and analysis of stellar photospheres", 3rd edition,
 ;   CUP, 2005.  For element stages not in this table the routine
 ;   reverts to the Wittmann-Sanchez-Asensio procedure
 ;   partfunc_wsa.pro.
 ;
 ; CATEGORY:
 ;   ATOMIC DATA
 ;
 ; CALLING SEQUENCE:
 ;   parfun=partfunc_rr(temp, elemnr, ionstage, $
 ;      gray=gray,wittmann=wittmann,log=log,verbose=verbose)
 ;
 ; INPUTS:
 ;   temp = temperature (K), single value or array
 ;   elemnr = element number, single value, hydrogen=1
 ;   ionstage = 0 for neutral, 1 for once-ionized, 2 for twice ionized
 ;
 ; KEYWORDS:
 ;   /log =  The result is log u(T). If the keyword is
 ;           not set, the result is u(T).
 ;   /verbose = add error messages, comparison plot
 ;   /gray = take Gray values even if outside range of validity
 ;   /wittmann = use Wittmann values for all
 ;
 ; COMMENT:
 ;   Use of the Gray table is restricted to the elements and
 ;   stages of ionisation listed in Gray (2005).
 ;   The table is in file partition_function.dat whose path must be specified
 ;
 ; EXAMPLE:
 ;   pf = partfunc_nv(6000, 7, 1)
 ;
 ; OUTPUTS:
 ;   partition function, number or array
 ;
 ; NOTE:
 ;   Gray: "The actual computation of partition function
 ;   can be somewhat laborious and requires a detailed
 ;   knowledge of the energy levels. For model photosphere
 ;   computation, interpolation within (this) table is easy
 ;   and convenient. In many publications. polynomials are
 ;   given, and while these are also convinient, they can
 ;   be misleading when used outside the temperature range
 ;   for which they were intended. (...) The entries (in the
 ;   Table) are accurate to $\approx 1\%$, and come from
 ;   several sources: Aller (1963), Evans (1966), Bolton
 ;   (1970), Irwin (1981), Cowley and Adelman (1983),
 ;   Sauval and Tatum (1984), Milone and Merlo (1998),
 ;   Halenka et al. (2001), and the NIST atomic data base.
 ;   Sauval and Tatum (1984) also give polynomials for
 ;   partition functions for 300 diatomic molecules."
 ;
 ; MODIFICATION HISTORY:
 ;   Jan 2006 Nikola Vitas: start
 ;   Nov 2007 Nikola Vitas: changed into function 
 ;   Sep 28 2013 RR: modified into partfunc_rr, added Wittmann call
;-

; no-parameter reply
if (n_params() lt 3) then begin ; N = nr required parameters
  print,' partfunc_rr,temp,elemnr,ionstage,log=log,verbose=verbose'
  return, -1
endif

; check keywords
if (n_elements(log) eq 0) then log=0
if (n_elements(verbose) eq 0) then verbose=0
if (n_elements(gray) eq 0) then gray=0
if (n_elements(wittmann) eq 0) then wittmann=0

; check ionstage
if (not (ionstage eq 0 or ionstage eq 1 or ionstage eq 2)) then begin 
  print,' ### partfunc_rr abort: ionstage = ',ntostr(ionstage),' out range'  
  return,-1
endif

; find temp array size
ntemp=n_elements(temp)

; read the Vitas-typed Gray table
tablefilename = 'rridl/ltelib/partfunc-gray.dat'
get_lun, u
openr, u, tablefilename
num = numlines(tablefilename)
graydata = fltarr(13, 247)
readf, u, graydata
free_lun, u

; first get Wittmann values 
partfunc_wsa,temp,elemnr,u1,u2,u3
if (ionstage eq 0) then wsaresult=u1
if (ionstage eq 1) then wsaresult=u2
if (ionstage eq 2) then wsaresult=u3
if (log eq 1) then wsaresult=alog10(wsaresult)
if (wittmann) then begin
  if (verbose) then print,'  partfunc_rr: Wittmann values only' 
  return,wsaresult
endif

; if elemnr/ionstage not in Gray table then use Wittmann 
identry=where(graydata[0,*] eq elemnr and graydata[1,*] eq ionstage)
if (identry eq -1) then begin
  if (verbose) then $
    print, ' elemnr/ionstage ',ntostr(elemnr),'/',ntostr(ionstage),$ 
    ' not in Gray, used Wittmann'
  return,wsaresult
endif

; interpolate in Gray table
theta=5040./temp
x=findgen(10)/5.+0.2
dgresult=interpol(graydata[2:11,identry], x, theta)
if (log eq 0) then dgresult=10.^dgresult
result=dgresult

; check Gray range of validity, replace by Wittmann outside that
if (not(gray)) then begin 

  if (ntemp eq 1) then begin
    if (theta lt 0.2 or theta gt 2.0) then begin
      result=wsaresult
      if (verbose) then print, ' ### Gray > Wittmann since temp out range'
    endif  
  endif

  if (ntemp gt 1) then begin
    if (min(theta) lt 0.2) then begin
      result[where(theta lt 0.2)]=wsaresult[where(theta lt 0.2)]
      if (verbose) then print,$
        '  ### Gray > Wittmann at too high temp values: ',$
        ntostr(temp[where(theta lt 0.2)])
    endif
    if (max(theta) gt 2.0) then begin 
      result[where(theta gt 2.0)]=wsaresult[where(theta gt 2.0)]
      if (verbose) then print,$
        '  ### Gray > Wittmann at too low temp values: ',$
        ntostr(temp[where(theta gt 2.0)])
    endif       
    if (verbose) then begin
      plotarr,result
      cgplot,dgresult,/overplot,/add,linestyle=2
      cgplot,wsaresult,/overplot,/add,linestyle=3
    endif
  endif
endif

return, result
end

; ------------- add main for testing per IDLWAVE H-c ----------------------

temp=1000
temp=[1000,3000,5000,10000,20000,100000]
for ielem=1,92 do begin 
  for ionstage=0,2 do $
    print,' elem nr = ',ntostr(ielem),' ionstage = ',ntostr(ionstage),$
          ' partfunc_rr = ', ntostr(partfunc_rr(temp,ielem,ionstage))
endfor
end


