; file: partfunc_nv.pro = partition function using Gray table 6.2
; last: Sep 24 2013  Rob Rutten, Deil
; site: ~/rr/idl/collib;  data in /home/rutten/rr/wrk/codes/gray-partfunc
; note: written by Nikola Vitas; found in my laptop Sep 15 2013 
;       needs XSSW for numlines.pro

;+
; NAME:
;   PARTITION_FUNCTION ;RR => partfunc_nv
;
; PURPOSE:
;   This procedure interpolates the partition function
;   table from Gray (The observations and analysis of
;   stellar photospheres, CUP, 2005) for a given
;   temperature T and for an element (Z) in stage of
;   ionisation R.
;
;   Gray: "The actual computation of partition function
;   can be somewhat laboriuos and requires a detailed
;   knowledge of the energy levels. For model photosphere
;   computation, interpolation within (this) Table is easy
;   and convinient. In many publications. polynomials are
;   given, and while these are also convinient, they can
;   be misleading when used outside the temperature range
;   for which they were intended. (...) The entries (in the
;   Table) are accurate to $\approx 1\%$, and come from
;   several sources: Aller (1963), Evans (1966), Bolton
;   (1970), Irwin (1981), Cowley and Adelman (1983),
;   Sauval and Tatum (1984), Milone and Merlo (1998),
;   Halenka et al. (2001), and NIST atomic data base.
;   Sauval and Tatum (1984) also give polynomials for
;   partition functions for 300 diatomic molecules."
;
;   The interpolation is linear in the logarithmic scale.
;
; CATEGORY:
;   ATOMIC DATA
;
; CALLING SEQUENCE:
;   PF = partfunc_nv(T, Z, R, log=log, verbose=verbose)
;
; INPUTS:
;   T =  temperature (K), number or array
;   Z =  atomic number, number
;   R =  ionisation stage (0 = neutral), number
;
; KEYWORDS:
;   /LOG =  The result is log u(T). If the keyword is
;           not set, the result is u(T).
;   /verbose = print error messages outside valid temperature range
;
; COMMENT:
;   Use of the table is restricted to the elements and
;   stages of ionisation listed in Gray (2005).
;   The table is in the file partition_function.dat whose path must be given
;
; EXAMPLE:
;   PF = partfunc_nv(1E4, 7, 1, /LOG)
;
; OUTPUTS:
;   PF: Value of partition number, number or array
;
; MODIFICATION HISTORY:
;   Jan 2006: Written by: Nikola Vitas (Jan2006)
;   Nov 2007: change into a function, NV 
;   Sep 15 2013 RR: use g_0 beyond theta=2; array indices () to []
;   Sep 17 2013 RR: changed name to partfunc_nv
;   Sep 24 2013 RR: addded /verbose
;-


FUNCTION partfunc_nv, T, Z, R, LOG=LOG, verbose=verbose

; no-parameter reply
if (n_params() lt 3) then begin    ; N = nr required parameters
  print,' partfunc_nv, T, Z, R, log=log, verbose=verbose'
  return, 0
endif

; check keywords
if (not keyword_set(verbose)) then verbose=0

FILENAME = 'rridl/ltelib/partfunc-gray.dat'
GET_LUN, U
OPENR, U, FILENAME
NUM = NUMLINES(FILENAME)
DATA = FLTARR(13, 247)
READF, U, DATA
FREE_LUN, U

; test validity element number and ion stage
ID = WHERE(DATA[0,*] EQ Z AND DATA[1,*] EQ R)
IF (ID EQ -1) THEN BEGIN
	PRINT, ' == partition_function: no element',Z,$
               ' or stage',R
	STOP
ENDIF

THETA = 5040./T

; define whether input temperature is single or array
sizetheta=size(theta)
if (sizetheta[0] gt 0) then ntheta=sizetheta[1] else ntheta=1
result=fltarr(ntheta)

; start loop over temp array; unrolled for upper individual limit test
for itheta=0,ntheta-1 do begin

; test for too low temperature
IF ((THETA[itheta] LT 0.2) and verbose) then begin
	PRINT, ' == partition_function: too high temp =',T[itheta]  ;RR
	THETA[itheta]=0.2  ;RR set cutoff
     ENDIF

if (theta[itheta] GT 2.0) then result[itheta]=data[12,ID] else begin
  X = FINDGEN(10)/5.+0.2
  RESULT[itheta] = INTERPOL(DATA[2:11,ID], X, THETA[itheta])
endelse

endfor ; end of loop over temperatures

if (ntheta eq 1) then result=result[0]  ;RR otherwise it is array[1] 

IF NOT(KEYWORD_SET(LOG)) THEN RESULT = 10.^RESULT

RETURN, RESULT
END
