; file: partfunc_chianti.pro
; init: Mar 28 2015  Rob Rutten  Deil
; last: Oct  2 2015  Rob Rutten  Deil

function partfunc_chianti,elemnr,ionstage,temp,$
  theory=theory,verbose=verbose

;+
 ; NAME:
 ;   partfunc_chianti
 ; PURPOSE:
 ;   compute partition function for a stage of an element in Chianti database 
 ; CALL:
 ;   U=partfunc_chianti(elemnr,ionstage,temp,theory=theory,verbose=verbose) 
 ; INPUTS:
 ;   elemnr = element number (hydrogen=1)
 ;   ionstage = ionization stage (neutral=0)
 ;   temp = temperature [K], single value or array
 ; OPTIONAL KEYWORD INPUTS:
 ;   theory 1/0: use theoretical instead of experimental excitation energies
 ;   verbose 1/0: print the addition per level; print less fatal errors too 
 ; OUTPUT:
 ;   partition function value (array if temp is array)
 ; RESTRICTIONS:
 ;   needs SSW package chianti
 ;   lacking data in the Chianti database
 ; HISTORY:
 ;   Mar 28 2015 RR: start
 ;   Oct  2 2015 RR: corrected zero energy entries 
;-

; answer no-parameter query like x=partfunc_chianti()
if (n_params() lt 3) then begin
  print,' U=partfunc_chianti(elemnr,ionstage,temp,$'
  print,'   theory=theory,verbose=verbose)'
  return, -1 
endif

; defaults for keywords
if (not keyword_set(theory)) then theory=0
if (not keyword_set(verbose)) then verbose=0

; checks
if (elemnr gt 30) then begin
  print,' #### ABORT partfunc_chianti: element not in Chianti (only 1-30)'
  return,-2
endif
if (ionstage eq elemnr) then return, 1.   ; naked ion has pf=1?
if (ionstage gt elemnr) then begin
  print,' #### ABORT partfunc_chianti: ionstage > elemnr'
  return,-3
endif

; define Chianti element and ion stage label arrays
element=['H','He','Li','Be','B','C','N','O','F','Ne','Na',$
         'Mg','Al','Si','P','S','Cl','Ar','K','Ca','Sc','Ti',$
         'V','Cr','Mn','Fe','Co','Ni','Cu','Zn']  ;RR first 30 elements

; construct .elvlc file name (has excitation level data for this ion)
elem=strlowcase(element[elemnr-1])
ion=elem+'_'+ntostr(ionstage+1)
ionfile=!xuvtop+'/'+elem+'/'+ion+'/'+ion+'.elvlc'

; check: file name
;; print,ionfile

; check Chianti file existence
if (not file_test(ionfile)) then begin
  if (verbose) then print,$
    ' #### ABORT partfunc_chianti: no file '+ionfile
  return,-4    ; use this value to switch to partfunc_rr in usage
endif

; get ionization energy 
read_ip,!xuvtop+'/ip/chianti.ip',ip,ref
ionerg=ip[elemnr-1,ionstage]
; convert from cm-1 to Rydbergs
ergpercmwav=1.98649E-16   ; erg per cm-1 wave number
rydbergcgs=2.1798741D-11  ; Rydberg unit (erg)
ionerg=ionerg*ergpercmwav/rydbergcgs
if (verbose) then print,' ionization energy (Rydberg) = ',ntostr(ionerg)

; read elvlc file (_direct also returns mult although it doesn't say so)
; eg: ew /home/rutten/rr/ssw/packages/chianti/dbase/si/si_4/si_4.elvlc
read_elvlc_direct,ionfile,l1,term,conf,ss,ll,spd,jj,mult,$
  ecm,eryd,ecmth,erydth,ref

; get number of tabulated levels
nlevels=n_elements(l1)
if (verbose) then print,' ===== nr of levels in file = ',ntostr(nlevels)

; abort at small number of levels
if (nlevels lt 3) then begin
  if (verbose) then print,$
    ' #### ABORT partfunc_chianti: nlevels = '+ntostr(nlevels)
  return,-5    ; use this value to switch to partfunc_rr in usage
endif

; now compute partition function (real men use cgs)
kcgs=1.380650D-16         ; Boltzmann constant (erg/deg)
rydbergcgs=2.1798741D-11  ; Rydberg unit (erg)
pfold=0
for ilevel=0,nlevels-1 do begin 
  if (theory) then energ=erydth[ilevel] else energ=eryd[ilevel]

; don't include levels bove limit (autoiinization levels)
  if (energ ge ionerg) then break

; don't include entries with zero energy (as in fe_8)
  if (ilevel gt 0 and energ lt 1.E-3) then goto, NEXTLEVEL

; add contribution by this level
  pfnew=pfold+mult[ilevel]*exp(-(energ*rydbergcgs)/(kcgs*temp)) 

; print diagnostic
  if (verbose) then $
    print,ntostr(theory)+'  '+ ntostr(ilevel)+'  '+ntostr(energ)+'  '+ntostr(pfnew)

; cut at small increase  neglect all higher levels 
  if ((pfnew[0]-pfold[0]) lt 1.E-5*pfold[0] and ilevel gt 4) then break 

; OK, next level
  pfold=pfnew

NEXTLEVEL:
endfor

return,pfnew
end


; =============== main for testing per IDLWAVE H-c ======================

elemnr=26 ; Fe
ionstage=7
; OOPS Chianti has no fe_1 .elvrc file; for fe_3 only ground level

;; elemnr=14 ; Si
;; ionstage=4
;; ; OOPS Chianti has no si_1 .elvrc file
;; ; si_2 good correspondence with partfunc_rr 

temp=indgen(5)*1000.+5000.  ; try array temp
temp=1.E5

u=partfunc_chianti(elemnr,ionstage,temp,/verbose) 
print,' partfunc_chianti = ',ntostr(u)

u=partfunc_chianti(elemnr,ionstage,temp,/theory,/verbose) 
print,' partfunc_chianti theory = ',ntostr(u)

rr=partfunc_rr(temp,elemnr,ionstage)
print,' partfunc_rr = ',ntostr(rr)

end
