; file: gaspress_wsa = RJR adaptation of Wittmann-Sanchez-Asensio version
; last: Oct  5 2013  Rob Rutten
; note: follows Wittmann 1974SoPh...35...11W p. 21-23

pro gaspress_wsa,pe,t,ph,phmi,phpl,ph2,ph2pl,pg,rho

;+
; PURPOSE:
; gives the partial pressure of h, h-, h+, h2, h2+ as well as the
; total gas pressure and density. It requires information about atmospheric
; abundances which is provided by subroutine atomdata_wsa.pro.
;
; references:
; Mihalas, Dimitri (1967), Methods in Computational Physics,
;   Vol. 7, The Calculation of Model Stellar Atmospheres, (Alder, B.,
;   Fernbach, S., and Rotenberg, M., eds)
;     OOPS: not on ADS for Mihalas...
;     OOPS: Oct  9 2013: OOPS, Amazon 214$...
;     AHA:  Mar 20 2014: pdf from Nikola Vitas, in cdperauthor mihalas 
;     Jul 21 2015: NOT found on ADS, not even the whole book....
;     AHA: found for sale and ordered it (34$)
; Wittmann: 1974, Solar Phys. 35, 11 = 1974SoPh...35...11W
; input:
;	pe	electron pressure (dyne cm^(-2))
;	t	temperature	  (K)
;
; output:
;	ph	partial pressure of h (dyne/cm2)
;	phmi	   "       "     "  h-   "
;	phpl	   "       "     "  h+   "
;	ph2	   "       "     "  h2   "
;	ph2pl	   "       "     "  h2+  "
;	pg	total gas pressure
;       rho     gas density
;
; Requires subroutines: electron_wsa, saha_wsa, atomdata_wsa, partfunc_wsa
;
; MODIFICATION HISTORY:
;	Sept 1st, 95	fortran > idl (Jorge Sanchez Aleida)
;	?? Included updated dissociation constant from ??
; Sep 16 2013 RJR: only cosmetic changes
; Oct  4 2013 RJR: use Y+Z abundance sum from common atmdat
;                  added rho to outputs
;-

; no-parameter reply
  if (n_params() lt 9) then begin    ; nr required parameters
    print,' gaspress_wsa,pe,t,ph,phmi,phpl,ph2,ph2pl,pg,rho'
    print,'   pe, t = electron pressure, temperature = input; may be arrays'
    print,'   rest is output'
    return
  endif

;RR add common with atomic data for the Wittmann-Sanchez-Asensio programs 
common common_atomdata_wsa,at_weight,at_abu,at_ionerg1,at_ionerg2,at_ionerg3,at_sym
;; atomdata_wsa,1   ; no initialization here to permit overwrite in main

; parameters for h2, h2+, h+ and h-
ch2=[12.533505d0,-4.9251644d0,5.6191273d-2,-3.2687661d-3]
ch2pl=[1.1206998d1,-2.7942767d0,-7.9196803d-2,2.4790744d-2]
chpl=[-13.595d0,2.5d0,-0.4772d0]
chmi=[-0.747d0,2.5d0,.1249d0]

;JS?  total abundance by number (it is not that of atomdata_wsa but
; comes from that adopted in the HSRA model atmosphere of Gingerich et
; al. 1971, Solar Phys. 18, 347)
abutotYZ=.101d0
;RR I don't see the logic for this fixing Y+Z; get it from the common
abutotYZ=total(at_abu)-at_abu[0]

; let's go with the equilibrium constants
theta=5040./t

coh2=ch2(0)+(ch2(1)+(ch2(2)+ch2(3)*theta)*theta)*theta
coh2=10.^coh2

coh2pl=ch2pl(0)+(ch2pl(1)+(ch2pl(2)+ch2pl(3)*theta)*theta)*theta
coh2pl=10.^coh2pl

; Saha for H-plus
cohpl=chpl(0)*theta+chpl(1)*alog10(t)+chpl(2)
cohpl=10.^cohpl

; Saha for H-min
cohmi=chmi(0)*theta+chmi(1)*alog10(t)+chmi(2)
cohmi=10.^cohmi

;JS?  now it solves a system of six equations with six unknows.
; contribution to pe from elements which are not H (two stages only)
; (the relevant elements have been taken from Mihalas (1967))
; Warning, I've commented out those elements which 
; are not used for the HSRA (table II, SPh 18, 357)
;RR ?? but nothing was commented out.  The below selection looks good.
g1=electron_wsa(2,t,pe)     ; He
g1=g1+electron_wsa(6,t,pe)  ; C
g1=g1+electron_wsa(7,t,pe)  ; N
g1=g1+electron_wsa(8,t,pe)  ; O
g1=g1+electron_wsa(11,t,pe) ; Na
g1=g1+electron_wsa(12,t,pe) ; Mg
g1=g1+electron_wsa(13,t,pe) ; Al
g1=g1+electron_wsa(14,t,pe) ; Si
g1=g1+electron_wsa(16,t,pe) ; S
g1=g1+electron_wsa(19,t,pe) ; K
g1=g1+electron_wsa(20,t,pe) ; Ca
g1=g1+electron_wsa(24,t,pe) ; Cr
g1=g1+electron_wsa(26,t,pe) ; Fe

g2=cohpl/pe
g3=pe/cohmi
g4=pe/coh2pl
g5=pe/coh2

a=1+g2+g3
b=2*(1+g2*g4/g5)
c=g5
d=g2-g3
e=g2*g4/g5

c1=c*b*b+a*d*b-e*a*a
c2=2*a*e-d*b+a*b*g1
c3=(-1)*(e+b*g1)

f1=(-1)*c2/2/c1
f2=sqrt((c2/2/c1)^2.-c3/c1)
pos=where(c1 ge 0.)
if(pos(0) ne -1)then f1(pos)=f1(pos)+f2(pos)
pos=where(c1 lt 0.)
if(pos(0) ne -1)then f1(pos)=f1(pos)-f2(pos)
f2=g2*f1
f3=g3*f1
f5=(1-a*f1)/b
f4=e*f5
f6=pe/(f2-f3+g1+f4)

ph=float(f1*f6)
phpl=float(f2*f6)
phmi=float(f3*f6)
ph2pl=float(f4*f6)
ph2=float(f5*f6)
pg=pe+float(f6*(abutotYZ+(f1+f2+f3+f4+f5)))

; Oct  4 2013 RR: add rho to output
kcgs=1.380650D-16         ; Boltzmann constant (erg/deg)
matom=1.660539D-24        ; atomic mass unit (gram)
melectron=9.109382D-28    ; electron mass (gram)

nelec=pe/(kcgs*t)
ngas=pg/(kcgs*t)
nhtot=(ph+phpl+phmi+2*ph2pl+2*ph2)/(kcgs*t) ; sum protons in any disguise
totweight=total(at_abu*at_weight)*matom*nhtot+nelec*melectron 
npart=(ph+phpl+phmi+ph2pl+ph2)/(kcgs*t)+nhtot*abutotYZ+nelec
meanweight=totweight/npart
rho=meanweight*ngas

end
