; file: findline_chianti.pro
; init: Apr  2 2015  Rob Rutten  Deil
; last: Nov  3 2015  Rob Rutten  Deil
; note: enter "setssw chianti" in ed /home/rutten/rr/bin/idl
;       adapt line selection at end of this file and run per Hyper C

pro findline_chianti,wavaa,elemnr=elemnr,ionstage=ionstage,air=air,$
  marginaa=marginaa,verbose=verbose

;+
 ; NAME:
 ;   findline_chianti 
 ; PURPOSE:
 ;   find a line in the Chianti database and print its spectroscopic idents 
 ; CALL:
 ;   findline_chianti,wavaa,elemnr=elemnr,ionstage=ionstage,air=air,$
 ;     marginaa=marginaa,verbose=verbose
 ; INPUTS:
 ;   wavaa: wavelength in Angstrom (vacuum unless /air set)
 ; OPTIONAL KEYWORD INPUTS:
 ;   elemnr: element nr (H = 1) (default: all 1-30 elements in Chianti)
 ;   ionstage: ionization stage, neutral = 0 (default: all in Chianti)
 ;   air 1/0: air wavelength given instead of vacuum wavelength
 ;   marginaa: search margin in Angstrom (default 0.1)
 ;   verbose 1/0: more/less printout
 ; OUTPUTS:
 ;   prints line hit identifications and designations 
 ; HISTORY:
 ;   Apr  2 2015 RR: start   
 ;   Apr  7 2015 RR: all keywords
 ;   Nov  3 2015 RR: correction number of stages per element 
;-

; answer no-parameter query 
if (n_params() lt 1) then begin
  print,'  findline_chianti,wavaa,elemnr=elemnr,ionstage=ionstage,air=air,$'
  print,'    marginaa=marginaa,verbose=verbose)'
  return
endif

; defaults for keywords
if (not keyword_set(elemnr)) then elemnr=-1
if (not keyword_set(ionstage)) then ionstage=-1
if (not keyword_set(air)) then air=0
if (not keyword_set(marginaa)) then marginaa=0.1
if (not keyword_set(verbose)) then verbose=0

; various checks
if (elemnr gt 30) then begin
  print,' #### ABORT findline_chianti: element not in Chianti (only 1-30)'
  return
endif
if (elemnr ne -1 and ionstage gt elemnr-1)  then begin
  print,' #### ABORT findline_chianti: ionstage > elemnr'
  return
endif

; convert air to vacumm wavelength and reverse
if (air) then airtovac,wavaa,vacwavaa else vacwavaa=wavaa
if (not air) then vactoair,wavaa,airwavaa else airwavaa=wavaa

; convert vacwavaa to Rydberg etc
vacwavaa=1.D*vacwavaa
evperaa=1.239842E4  ; divide this by vacuum wavelength in AA to get eV
wavev=evperaa/vacwavaa
rydbergev=13.605692 ; Rydberg unit (eV) ;RR used to be 13.596 or such?
wavryd=wavev/rydbergev
wavnrcm=1./(1.E8*vacwavaa)
print,' ==== search wavelength:'+ $
  '  vac='+ntostr(vacwavaa)+$
  '  air='+ntostr(airwavaa)+$
  '  Rydberg='+ntostr(wavryd)+$
  '  wavnrcm='+ntostr(wavnrcm)

; convert margin to Rydberg
marginev=marginaa*evperaa/vacwavaa^2
marginrb=marginev/rydbergev

; define Chianti element and ion stage label arrays
element=['H','He','Li','Be','B','C','N','O','F','Ne','Na',$
         'Mg','Al','Si','P','S','Cl','Ar','K','Ca','Sc','Ti',$
         'V','Cr','Mn','Fe','Co','Ni','Cu','Zn']  ;RR first 30 elements

; start loop over elements if not specified
if (elemnr ne -1) then nelem=1 else nelem=30
for ielem=1,nelem do begin
  if (elemnr ne -1) then thiselem=elemnr else thiselem=ielem

; set ion 
  if (ionstage ne -1) then nion=1 else nion=thiselem+1
  for iion=0,nion-1 do begin
    if (ionstage ne -1) then thision=ionstage else thision=iion

; construct .elvlc file name (has excitation level data for this ion)
    elem=strlowcase(element[thiselem-1])
    ion=elem+'_'+ntostr(thision+1)
    ionfile=!xuvtop+'/'+elem+'/'+ion+'/'+ion+'.elvlc'

; print ion and file name
    if (verbose) then print,' ==== species: '+ion 
    if (verbose) then print,' ==== chianti file (use ew): '+ionfile

; check Chianti file existence (not when scanning all)
    if (not file_test(ionfile)) then begin
      if (ionstage ne -1) then begin 
        if (verbose) then print,$
          ' #### ABORT findline_chianti: no file '+ionfile
        return
      endif
      goto,NEXTION
    endif

; read elvlc file (_direct also returns mult although it doesn't say so)
; eg: ew /home/rutten/rr/ssw/packages/chianti/dbase/si/si_4/si_4.elvlc
    read_elvlc_direct,ionfile,l1,term,conf,ss,ll,spd,jj,mult,$
      ecm,eryd,ecmth,erydth,ref

; get number of tabulated levels
    nlevels=n_elements(l1)
;; if (verbose) then print,' ===== nr of levels in file = ',ntostr(nlevels)

; search for the line
    for ilow=0,nlevels-2 do begin
      for iup=1,nlevels-1 do begin
        diff=abs(eryd[iup]-eryd[ilow]-wavryd)

; bingo
        if (diff lt marginrb) then begin
          hitwavryd=eryd[iup]-eryd[ilow]
          hitwavev=hitwavryd*rydbergev 
          hitvacwavaa=evperaa/hitwavev
          vactoair,hitvacwavaa,hitairwavaa
          print,' ==== hit: transition (line?) of '+ion+$
            ' at wavelength:'+ $
            '  vac='+ntostr(hitvacwavaa)+$
            '  air='+ntostr(hitairwavaa)+$
            '  Rydberg='+ntostr(hitwavryd)
          print,'      lower:'+$
            '  levelnr='+ntostr(ilow+1,format='(I4)')+$
            '  term='+ntostr(term[ilow],format='(I4)')+$
            '  conf='+ntostr(conf[ilow])+$
            '  l='+ntostr(ll[ilow],format='(I4)')+$
            '  j='+ntostr(jj[ilow],format='(I4)')+$
            '  g='+ntostr(mult[ilow],format='(I4)')+$
            '  excergryd='+ntostr(eryd[ilow])+$
            '  excergev='+ntostr(eryd[ilow]*rydbergev)
          print,'      upper:'+$
            '  levelnr='+ntostr(iup+1,format='(I4)')+$
            '  term='+ntostr(term[iup],format='(I4)')+$
            '  conf='+ntostr(conf[iup])+$
            '  l='+ntostr(ll[iup],format='(I4)')+$
            '  j='+ntostr(jj[iup],format='(I4)')+$
            '  g='+ntostr(mult[iup],format='(I4)')+$
            '  excergryd='+ntostr(eryd[iup])+$
            '  excergev='+ntostr(eryd[iup]*rydbergev)
        endif
      endfor ; end loop over upper levels
    endfor ; end loop over lower levels

NEXTION:
  endfor ; end loop over all ionstages

endfor   ; end loop over all elements


end

; ================== main to test per IDLWAVE Hyper-c ====================


;; air=1
;; wavaa=4571.1 ; OOPS no file MgI 
;; wavaa=4554.0 ; OOPS no element Ba
;; wavaa=3933.7 ;fine

;; elem=6  ; IRIS CII
;; ion=1
;; air=0
;; margin=2
;; wavaa=1335.7 ; fine; get doublet at delta labda = 0.04 AA !

;; elem=14 ; IRIS SiIV
;; ion=3
;; air=0
;; wavaa=1393.8  ; 139.37499
;; wavaa=1402.7  ; 140.27652

;; elem=26 ; AIA FeIX 171
;; ion=8
;; air=0
;; wavaa=171.07
;; margin=0.1

;; elem=26 ; AIA FeXII 193
;; ion=11
;; air=0
;; wavaa=193.51
;; margin=0.05

;; elem=2 ; HeII 304
;; ion=1
;; air=0
;; wavaa=304
;; margin=0.5

;; elem=1 ; HI 1216 = Lyalpha
;; ion=0
;; air=0
;; wavaa=1215.67
;; margin=1.

elem=1 ; HI 6563 = Halpha
ion=0
air=1
wavaa=6563.
margin=1.

findline_chianti,wavaa,elemnr=elem,ionstage=ion,air=air,$
  marginaa=margin,/verbose

end
