; file: findgf_kurucz.pro
; init: Nov  4 2015  Rob Rutten  Deil
; last: Nov  6 2015  Rob Rutten  Deil

pro findgf_kurucz,wavaa,elemnr,ionstage,$
  wav_kur,loggf,gf,excev, glow,gup,lablow,labup,$
  marginaa=marginaa,quiet=quiet

;+
 ; NAME:
 ;   findgf_kurucz 
 ; PURPOSE:
 ;   find line parameter values in Kurucz line list  
 ; CALL:
 ;   findgf_kurucz,wavaa,elemnr,ionstage
 ; INPUTS:
 ;   wavaa = line wavelength in AA, within marginaa, air above 2000 AA 
 ;   elemnr = element number, example H=1
 ;   ionstage = ionization stage, example neutral=0
 ; OPTIONAL KEYWORD INPUTS:
 ;   marginaa = search bandwidth in AA, default 0.1 
 ;   quiet=1/0: printout or not
 ; OUTPUT (for last line if multiple components meet selection criteria)
 ;   wav_kur = wavelength in Kurucz list, nm, air above 2000 AA
 ;   loggf
 ;   gf
 ;   excev
 ;   glow    # maybe incorrect if lablow = AVERAGE
 ;   gup     # maybe incorrect if lablow = AVERAGE
 ;   lablow = transition label in Kurucz 
 ;   labup
 ; OPTIONAL OUTPUTS:
 ;   printout of the above if quiet ne 1
 ; RESTRICTIONS:
 ;   only lines in this Kurucz list
 ;   no lines for ionstage above 8
 ;   complexes (Halpha) = labels AVERAGE, ENERGIES have NOT total glow, gup
 ; HISTORY:
 ;   Nov  4 2015 RR: start
;-

; answer no-parameter query 
if (n_params() lt 3) then begin
  print,'  findline_kurucz,wavaa,elemnr,ionstage,$'
  print,'    marginaa=marginaa,quiet=quiet'
  return
endif

; defaults for keywords
if (not keyword_set(marginaa)) then marginaa=0.1
if (not keyword_set(quiet)) then quiet=0

; print pacifier
if (quiet ne 1) then print,' ===== reading Kurucz file, takes very long'

; constants
ergpercmwav=1.98649E-16         ; erg per cm-1 wave number
ergperev=1.60217657E-12         ; erg per electronvolt
evpercmwav=ergpercmwav/ergperev ; eV per cm wave number

; read first columns
kuruczfile='allgf_kurucz.dat'  ; copy in dir ltelib so no path needed

; faster reading but no good when labels are split in other than 2 parts
;; readcol,kuruczfile,wav,loggf,elemstage,exclow,jlow,label1low,label2low,$
;;   excup,jup,label1up,label2up,loggamrad,loggamstark,loggamvdw,$
;;   format='D,F,F,F,F,A,A,F,F,A,A,F,F,F'

fmt='F11.4,F7.3,F6.2,F12.3,F5.1,1X,A10,F12.3,F5.1,1X,A10,F6.2,F6.2,F6.2'
readfmt,kuruczfile,fmt,$
  wav,loggf,elemstage,exclow,jlow,labellow,$
  excup,jup,labelup,loggamrad,loggamstark,loggamvdw

; get indices of lines of this element and ionstage
thiselemstage=elemnr+ionstage/100.
selelemstage=where(elemstage eq thiselemstage)

; report how many lines found
if (quiet ne 1) then $
  print,' nr of lines of elemstage '+ntostr(thiselemstage,format='(F7.2)')$
  +' = '+ntostr(n_elements(selelemstage))

; now find lines in specified wav range
selwav=where(abs(wav[selelemstage]-wavaa/10.) lt marginaa) 

; report 
nsel=n_elements(selwav)
if (quiet ne 1) then print,$
  ' nr of elemstage lines in wav range = '+ntostr(nsel)

; output for selected line(s)
for iline=0,nsel-1 do begin

; get values
  wav_kur=wav[selelemstage[selwav[iline]]]
  loggf_kur=loggf[selelemstage[selwav[iline]]]
  gf=10^loggf_kur
  excev=exclow[selelemstage[selwav[iline]]]*evpercmwav
  glow=1+2*jlow[selelemstage[selwav[iline]]]
  gup=1+2*jup[selelemstage[selwav[iline]]]
  lablow=labellow[selelemstage[selwav[iline]]]
  labup=labelup[selelemstage[selwav[iline]]]

; print
  if (quiet ne 1) then begin
    print,' wav_kur (nm, air) = '+ntostr(wav_kur)+$
      '   log gf = '+ntostr(loggf_kur)+$
      '   gf = '+ntostr(gf)
    print,'     excev = '+ntostr(excev)+$
      '   glow = '+ntostr(glow)+$
      '   gup = '+ntostr(gup)
    print,'     lablow = '+lablow+$
      '   labup = '+labup+$
      ' (with l=0,1,2,3,4,5 for s,p,d,f,g,h)'
  endif

endfor

end

; ========= run per Hyper-C ===================================

;; wavaa=6562.81  ; Halpha ## labels AVERAGE ENERGIES so no good
;; elemnr=1
;; ionstage=0

;; wavaa=8542.14   ; Ca II IR
;; elemnr=20
;; ionstage=1

;; wavaa=3933.68   ; Ca II K
;; elemnr=20
;; ionstage=1

;; wavaa=2795.33    ; Mg II k
;; elemnr=12
;; ionstage=1

;; wavaa=5172.70    ; Mg I b2
;; elemnr=12
;; ionstage=0

;; wavaa=5895.94    ; Na I D1
;; elemnr=11
;; ionstage=0

;; wavaa=4554.04    ; Ba II 
;; elemnr=56
;; ionstage=1

;; wavaa=6625.02       ; Fe I
;; elemnr=26
;; ionstage=0

;; wavaa=1393.75      ; Si IV
;; elemnr=14
;; ionstage=3

;; wavaa=1335.71  ; CII
;; elemnr=6
;; ionstage=1

;; wavaa=171.073  ; Fe IX
;; elemnr=26
;; ionstage=8

;; wavaa=193.51  ; Fe XII   ## oops, no ionstage>8 in Kurucz list
;; elemnr=26
;; ionstage=11

;; wavaa=303.78 ; He II   ## labels AVERAGE ENERGIES so no good
;; elemnr=2
;; ionstage=1

;; wavaa=1215.67 ; Lyalpha  ## labels AVERAGE ENERGIES so no good
;; elemnr=1
;; ionstage=0

wavaa=6301.5 ; Fe I magnetometry line
elemnr=26
ionstage=0

findgf_kurucz,wavaa,elemnr,ionstage,marginaa=0.05

end
