; file: exthneubf_rr.pro = HI bound-free extinction
; init: Oct 16 2013  Rob Rutten  Deil
; last: Oct 27 2016  Rob Rutten  Texel
; note: effectively identical to exthneubf_gray.pro
;       unless Seaton gaunt factor is selected within the code

function exthneubf_rr,wav,temp

;+
 ; NAME:
 ;   exthneubf_rr
 ; PURPOSE:
 ;   compute neutral hydrogen bound-free extinction assuming LTE
 ; DESCRIPTION:
 ;   follows Gray 2005 3rd edition
 ; CALL:
 ;   exthneubf=exthneubf_rr(wav,temp)
 ; INPUTS:
 ;   wav = wavelength in Angstrom, may be an array
 ;   temp = temperaure in K, may be an array bot both simultaneous
 ; OUTPUTS:
 ;   extinction per cm per neutral hydrogen atom,
 ;   already including LTE correction for stimulated emission
 ; RESTRICTIONS:
 ;   LTE excitation is assumed for the H levels that contribute edges
 ; NOTE:
 ;   effectively identical to exthneubf_gray unless Seaton is selected below
 ; HISTORY:
 ;   Oct 16 2013 RR: start from exthneubf_gray 
;-

; answer no-parameter query 
if(n_params(0) lt 2) then begin
  print,'ext=exthneubf_rr(wav,temp)'
  print,'  wav = number or array [AA]'
  print,'  temp = number or array [K]   - but not both array' 
  return,-1
endif

; read physics constants (cgs)
@constants_cgs.idl

; bound-free: Gray but with explicit summation loop over levels
;RR essential to make level and wav double precision floats
lambda=1.D*float(wav)
nlevels=500 ; trial Oct 27 2016 
nlevels=200 ; up to Oct 27 2016 ; edgewav limit plm 3.5 mm = ALMA range
edgewav=fltarr(nlevels+1)
boltzpop=fltarr(nlevels+1)
parfunh=partfunc_rr(temp,1,0)  
sigmabf=0 ; initiate summation

; loop over hydrogen edges beyond this wavelength 
for ilevel=1,nlevels do begin  ;RR real n, not IDL index with zero
  level=ilevel*1.D0
  edgewav(ilevel)=level^2/rydbergaa

; gaunt factor: Gray 2005 (8.5)
  gauntbf_gray=1.0-0.3456/ $
    ((lambda*rydbergaa)^0.3333)*(lambda*rydbergaa/level^2-0.5)

; gaunt factor: Seaton 1960RPPh...23..313S eq (2.4) found in RH; u=eps=0
  gauntbf_seaton=(1.-0.1728*level^(-2./3)-0.0496*level^(-4./3))

; take Gray values (slightly larger than Seaton for n=3)
;;  gauntbf=gauntbf_seaton
  gauntbf=gauntbf_gray
  
  ; set gauntbf to zero for wavelengths beyond this edge (with Krijger trick)
  if (size(lambda,/n_elements) gt 1) then begin
    if (max(lambda) ge edgewav(ilevel)) then $
      gauntbf[where(lambda gt edgewav(ilevel))]=0.
  endif else if (lambda ge edgewav(ilevel)) then gauntbf=0.

; summation with Boltzmann populations as Gray III 2005 8.7 2nd version
  chiexc=rydbergev*(1.-1./(level^2))   ; in eV, Gray IIII (8.3)
  add=1.0449E-26*lambda^3/level^3*gauntbf*exp(-chiexc/(kev*temp))
  sigmabf=sigmabf+add
  if (max(add/sigmabf) lt 1E-4) then break  ; 1 promille accuracy, overdone  

;; print,' === ' ,ntostr([ilevel,edgewav(ilevel),gauntbf,gauntbf_gray,$
;;       sigmabf],format='(I5,F12.3,2F7.3,E15.6)')
  
endfor

; correction stimulated emission although this is still per neutral H atom
alphabf=sigmabf*(1.-exp(-hcgs*ccgs/(lambda*1E-8*kcgs*temp)))

return,alphabf
end

; ================ run/test per Hyper C ======================

temp=10.E4 ; K

;; ; show Lyman, Balmer, Paschen edges
;; wav=indgen(1000)*10.+ 500.   
;; alphabf=exthneubf_rr(wav,temp)
;; plot,wav,alphabf
;; ; looks OK

; show very hign-n edges (after setting levelmax=500)
logwav=indgen(8000)*0.001+ 0.2
alphabf=exthneubf_rr(10^logwav,temp)
plot,logwav,alphabf
plot,10.^logwav,alphabf,xrange=[0,5.E7],xstyle=1

end
