; file: iris_sji_clean
; init: Mar  6 2018  Rob Rutten  Deil
; last: Apr  4 2018  Rob Rutten  Deil

;+

pro iris_sji_clean,indir,$
  refwav=refwav,shiftfile=shiftfile,dustoff=dustoff,$
  histopt=histopt,despike=despike,$
  trimbox=trimbox,trange=trange

 ; NAME:
 ;   iris_sji_clean.pro    
 ;   
 ; PURPOSE:
 ;   convert IRIS SLI slit-image files to simple cleaned fits files
 ;           with cutcentral to common size
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   indir: string with dir with input iris_l2 files and for output files   
 ;   refwav: SJI wavelenth (4 digits) to synch others to (default 1400)
 ;   shiftfile: string with shifts from iris_sji_findalign.pro WHEN EXISTS
 ;   dustoff = 1/0: do or don't, very slow and needs (too?) much memory
 ;   histopt = cutoff evel for histo_opt (default 1.E-3)
 ;   despike: 0 = don't; value 4.5 = beyond 4.5 rms
 ;   trimbox = cut frame [lower-x,lower-y,upper-x,upper-y]
 ;   trange = [itstart,itend] for faster tests or only overlap with eg SST
 ;
 ; COMMENTS: iris_intscale.pro needs too much memory for my laptop,
 ;   returns floats or double, doesn't do the claimed exposure normalization.
 ; 
 ; HISTORY:
 ;   Mar  9 2018 RR: start  
;-

; answer no-parameter query 
if (n_params(0) lt 1) then begin
  sp,iris_sji_clean
  return       
endif

; set wall-clock timer (seconds)
timestart=systime(1) 

; defaults for keywords
if (n_elements(refwav) eq 0) then refwav=1400
if (n_elements(shiftfile) eq 0) then shiftfile=''
if (n_elements(dustoff) eq 0) then dustoff=0
if (n_elements(histopt) eq 0) then histopt=1.E-3
if (n_elements(despike) eq 0) then despike=0
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(trimbox) eq 0) then trimbox=[-1,-1,-1,-1]

; set swap_endian for this computer 
;RR l = little, b = big; fits = big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'
if (endian eq 'l') then swap_endian=1 else swap_endian=0

; refwav as string
refstring=trim(string(refwav))
refname='SJI_'+refstring

; get all SJI files
sjifiles=findfile(indir+'/iris_l2*SJI*.fits')
nfiles=n_elements(sjifiles)
if (nfiles ge 1) then $ 
  print,' ##### rob2rob: this program eats your laptop - better bike' $
else print,' ##### iris_sji_clean abort: no input SJI files'

; reorder file list to put refwav first
for ifile=0,nfiles-1 do begin
  if (strmatch(sjifiles[ifile],'*'+refstring+'*')) then begin
    tempfile=sjifiles[ifile]
    sjifiles[ifile]=sjifiles[0]
    sjifiles[0]=tempfile
  endif
endfor

; start loop over re-ordered SJI files
for ifile=0,nfiles-1 do begin

; open as object d
  print,' ----- open sji file = '+sjifiles[ifile]
  d=iris_load(sjifiles[ifile])

; get timing for this file; set first file = ref
  tai_in=d->ti2tai()  
  nt_in=n_elements(tai_in)
  if (ifile eq 0) then begin
    reftai=tai_in
    refutc=d->ti2utc()  ; for writing out 
  endif

; get single-frame header (not time-dependent arrays)
  head_in=d->gethdr(/struct)

; it seems necessary reread the data for hdr and dustbuster
  read_iris_l2,sjifiles[ifile],hdr,dat

; float > signed integer for memory
  dat=round(dat)

; get infile sizes
  sizedat=size(dat)
  nx_in=sizedat[1]
  ny_in=sizedat[2]
  nt_in=sizedat[3]
  if (ifile eq 0) then begin
    nx_ref=nx_in
    ny_ref=ny_in
    nt_ref=nt_in
  endif else begin
    if (nx_in ne nx_ref or ny_in ne ny_ref) then $
      print,' ##### iris_sji_clean warning: files have inequal nx or ny'
  endelse

; trange
  itstart=trange[0]
  itend=trange[1]
  if (itend eq -1) then itend=nt_in-1 
  nt_out=itend-itstart+1

; trimbox
  xmin=0
  ymin=0
  xmax=nx_in-1
  ymax=ny_in-1
  if (trimbox[0] ne -1) then begin
    if (trimbox[0] gt 0) then xmin=trimbox[0]
    if (trimbox[2] lt nx_in-1) then xmax=trimbox[2]
    if (trimbox[1] gt 0) then ymin=trimbox[1]
    if (trimbox[3] lt ny_in-1) then ymax=trimbox[3]
  endif
  nx_out=xmax-xmin+1  ; needed to open unit for assoc out
  ny_out=ymax-ymin+1

; long > signed integer to reduce memory need
  dat=fix(dat)

; optional dust removal (problematic in memory need; needs full duration)
  if (dustoff ne 0) then dat=iris_dustbuster(hdr,dat) ; do in place

; histopt
    dat=histo_opt_rr(dat,histopt)

;  select trange, trim to trimbox or make border triangles grey, despike
    if (trimbox[0] eq -1) then $
      reformcube,dat,dat2,despike=despike,trange=trange,/greytriangles $
    else reformcube,dat,dat2,despike=despike,trange=trange,trimbox=trimbox

; histopt again
    dat=histo_opt_rr(dat2,histopt) 

;; ; check
;; showex,dat,dat2
;; STOP

; convert to positive integers (NB: no longer counts hereafter)
    minval=min(dat)
    maxval=max(dat)
    dat=fix((dat-minval)/float(maxval-minval)*20000.) 

;; ; inspect
;;   showex, dat 
;;   STOP

; ######### @@@@ TODO: read and apply shifts from disk-center determination

; define outheader including cdelt keywords
    bitpix=16
    mkhdr,outheader,abs(bitpix)/8,[nx_out,ny_out,nt_out]

; copy some keywords into outheader (########@@@@@ add shifts)
    sxaddpar,outheader,'cdelt1',head_in.cdelt1,$
      'x-scale [arcsec/px]'
    sxaddpar,outheader,'cdelt2',head_in.cdelt2,$
      'y-scale [arcsec/px]'
    sxaddpar,outheader,'crpix1',head_in.crpix1,$
      'disk-center offset in px from leftmost of FOV = nr 1' 
    sxaddpar,outheader,'crpix2',head_in.crpix2,$
      'disk-center offset in px from bottom of FOV = nr 1'
    sxaddpar,outheader,'xcen',head_in.xcen,$
      'FOV center X [arcsec from sun center]'
    sxaddpar,outheader,'ycen',head_in.ycen,$
      'FOV center Y [arcsec from sun center]'

; write temporary image file
    ident=head_in.tdesc1
    wavname=ident[0]
    print,' ----- write temporary image file = '+wavname+'_temp.fits'
    writefits,indir+'/'+wavname+'_temp.fits',dat,outheader

; convert the cubefile 
    reformcubefile,indir+'/'+wavname+'_temp.fits',$
      indir+'/iris_'+wavname+'.fits',$
      timearr_in=tai_in[itstart:itend],timearr_out=reftai[itstart:itend]
  endfor ; end of loop over files

; write the common timing file (only trange)
  print,' ----- write timing file = iris_times.txt'
  writecol,indir+'/iris_'+refname+'-times.txt',$
    refutc[itstart:itend],fmt='(A)'

; delete temp files
  spawn,'rm -f '+indir+'/*temp.fits'

; print elapsed time
  timedone=systime(1)
  timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
  print,' ===== iris_sji_clean took '+timelaps+' minutes'

end


; =============== main for testing per IDLWAVE H-c ======================

;; indir='/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/iris'

indir='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/iris/target'
dustoff=1
;; histopt=5.E-4 
histopt=1.E-4
despike=4
;; trange=[410,420] ; many cosmic ray hits (test despike)
;; trange=[94,345]  ; only segment containing SST
;; trimbox=[410,70,910,465] ; within SDO and containing SST (I hope)
;; trange=[100,120] ; trial

iris_sji_clean,indir,$
  dustoff=dustoff,histopt=histopt,despike=despike,$
  trimbox=trimbox,trange=trange

; check
;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/iris/'
cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/iris/target'
showex,'iris_SJI_1400.fits','iris_SJI_2796.fits'
;;    'iris_SJI_2832.fits','iris_SJI_1330.fits'

end
