; file: showex.pro = start movex.pro on image(s) or cube array(s) or file(s)
; init: Nov  7 2017  Rob Rutten  Deil = cube wrapper to movex.pro
; last: Dec  8 2021  Rob Rutten  Deil

;+  
pro showex,var1,var2,var3,var4,var5,var6,var7,var8,var9,var10,$
  var11,var12,var13,var14,var15,var16,var17,var18,var19,var20,$
  var21,var22,var23,var24,var25,var26,var27,var28,var29,var30,$
  var31,var32,var33,var34,var35,var36,var37,var38,var39,var40,$
  var41,var42,var43,var44,var45,var46,var47,var48,var49,var50,$
  fullmonty=fullmonty,montydirs=montydirs,_extra=movexkeywords

 ;   Starts movex.pro on one or (many) more (x,y) image arrays or (x,y,t) or 
 ;     (x,y,w*t) cube arrays in memory or 2D jpg or png or fits image files or
 ;     mp4, mpg, mpeg, mov movie files or 3D image-sequence data files
 ;     or multi-wavelength image-sequence data files in different formats
 ;     (fits, DOT, SST "la Palma"), or ps, eps, pdf figure files.
 ;   Conditions:
 ;     - all the same (nx,ny,nt) dimensions (types may differ)
 ;     - only monochrome greyscale (I think; I never use color)
 ;   See movex.pro for detail and keyword options.
 ;   Examples of usage:
 ;     - one image: zoom in, tinker with intscale, find pixel coordinates
 ;         and brightness value, get fits, ps, png file of current image
 ;     - two or more images: blink pairs, inspect pairwise scatterplot
 ;     - one cube: as for one image, plus time-delay blinks and scatterplots
 ;     - two or more cubes: full movex palette of live display features
 ;     - compare all SDO fitscubes in SDO dir(s) (/allsdo).
 ;     - compare all wavelength samplings in multiwav dir(s) (/allmwf)
 ;     - compare all fitscubes in fits dir(s( (/allfits, ignores SDO and mwf)
 ;     - compare all such cubefiles (/fullmonty)
 ;     - blink two publication figures (on the command line with script movex
 ;       shown in my sdomanual)
 ;
 ; INPUT PARAMETERS:
 ;   var1 = array [nx,ny] or [nx,ny,nt] or [nx,ny,nw*nt] of any type 
 ;          (w=wavelength and/or Stokes as in SST/CRISP data files)
 ;          or a string or a string array 'path+filename' for files
 ;          (2D or 3D fits, 3D DOT, 3D LaPalma) or a string or string
 ;          array with jpg, tif, png, mp4, mpeg, mpg, mov files 
 ;   [var2,......] = idem; mixing arrays and files permitted
 ; 
 ; OPTIONAL KEYWORDS INPUT PARAMETERS:
 ;   fullmonty = 1/0: set /allfits, /alldso, /allmwf, /allspect,
 ;     /addlabels, /plotscatter (setting /full is enough)
 ;   montydirs: string array dirs for full-monty loading (default './')
 ;   all movex keywords (type IDL> movex or IDL> sp,movex for specification)
 ;
 ; OUTPUT:
 ;   active movex.pro player/blinker/scatterplotter 
 ;
 ; RESTRICTIONS:
 ;   - all variables must be synchronous and have the same (nx,ny,nt) sizes
 ;   - specify keyword nt_mw when loading only CRISP La Palma file(s)
 ;   - at multiple zoom-in's or calls IDL may run out of file units: restart
 ;   - startup may be slow for many long movies
 ;
 ; EXAMPLES (see also tests at bottom of this file):
 ;  - one image variable: zoom in, find pixel coordinates and brightness
 ;      value, obtain ps or png output
 ;      IDL> showex,imagearr
 ;      IDL> showex,'dirname/image.fits'  
 ;  - two image variables: blink, get scatter plot, locate funny pixels
 ;      IDL> showex,image1,image2,/scatterplot
 ;  - four cube variables: play, (time-delay) blink, scatterplot, etc.
 ;      IDL> showex,cube1,cube2,cube3,cube4,/scatterplot,/plotpower
 ;  - one cubefile: play, time-delay blink and scatterplot for evolution
 ;      IDL> showex,'cube.fits',/plotscatter
 ;  - two SDO files: play, (time-delay) blink, scatterplot, etc.
 ;      IDL> showex,'sdo2sst/aia171_alfor_px.fits',$
 ;                   'sdo2sst/aia304_alfor_px.fits
 ;      IDL> showex,['sdo2sst/aia171_alfor_px.fits',$
 ;                   'sdo2sst/aia304_alfor_px.fits]  ; also array permitted
 ;  - all SDO files in a directory to play, blink, scatterplot any pair:
 ;      IDL> showex,/allsdo,sdodirs='sdo_cube_directory'
 ;  - one multi-wavelength SST/CRISP file:
 ;      IDL> showex,'crispex.6563.09:48:31.time_corrected.aligned.icube',$
 ;                  nt_mw=35
 ;  - all co-aligned SDO files and two particular SST/CRISP files:
 ;      IDL> showex,'sst/crispex.8542.09:48:31.stokesI.icube',$
 ;                  'sst/crispex.6563.09:48:31.time_corrected.aligned.icube',$
 ;                  sstspecsavs=['sst/spectfile.6563.idlsave',$
 ;                               'sst/spectfile.8542.idlsave'],$
 ;                 /allsdo,/plotscatter,/addlabels
 ;  - all co-aligned SDO, SST/CRISP and other files in collective dir(s):
 ;      IDL> showex,/full,montydirs='sst2rotsdo',spectdirs='crispex'
 ;  - two figure files on command line:
 ;      > showex gongha_rev.pdf aia304.pdf
 ;      
 ; HISTORY:
 ;   Feb  4 2011 RR: cubeximovie = wrapper to Oslo ximovie.pro
 ;   Jul  3 2017 RR: cubemovex = two-cube wrapper to movex.pro
 ;   Sep 29 2017 RR: many input cubefiles 
 ;   Nov  7 2017 RR: also image files > showex.pro
 ;   Nov 14 2017 RR: mix array variables and files, also fits images
 ;   Feb 19 2020 RR: fullmonty
 ;   Jul 18 2020 RR: jpg, png, mp4, mpg, mpeg, mov; shell pipe script
 ;   Aug 28 2020 RR: ps, eps, pdf figure files
 ;   Dec  8 2021 RR: tif figure files
;-

; no-parameter query return
if (n_params() lt 0) then begin
  print,' ##### showex calls movex: type movex or sp,movex'
  sp,showex
  return
endif

; keyword defaults
if (n_elements(var1) eq 0) then var1='' 
if (n_elements(fullmonty) eq 0) then fullmonty=0 
if (n_elements(montydirs) eq 0) then montydirs='./'

; movex keywords possibly adapted here 
if (n_elements(allsdo) eq 0) then allsdo=0 
if (n_elements(sdodirs) eq 0) then sdodirs='./'
if (n_elements(mwfdirs) eq 0) then mwfdirs='./'
if (n_elements(fitsdirs) eq 0) then fitsdirs='./'
if (n_elements(spectdirs) eq 0) then spectdirs='./'

; get number of variables and check
nvars=n_params()
if (nvars gt 50) then begin
  print,' ##### showex abort: nvars = '+trimd(nvars)+' exceeds max = 50'
  print,'       add more variables in showex.pro if you need more'
  return
endif

; ----- start big loop over variables
allfilenames=strarr(1)
iarr=0
if (nvars eq 2) then blink=1 else blink=0
for ivar=0,nvars-1 do begin

; get variable name; check it exits (F**K IDL)
  varname='var'+trim(ivar+1)     ; var1, var2, etc
  if (n_elements(scope_varfetch(varname)) eq 0)  then begin
    print,' ##### showex abort: variable var'+trim(ivar+1)+' does not exist'
    return
  endif
  var=scope_varfetch(varname)    ; actual variable

; get size
  sizevar=size(var)

; var = string or string array, presumably file(s)
  if (sizevar[n_elements(sizevar)-2] eq 7) then begin

; get nrfiles in this var
    if (sizevar[0] eq 0) then nvarfiles=1 else nvarfiles=sizevar[1]

; start loop over files in this var
    for ivarfile=0,nvarfiles-1 do begin 

; check whether all strings are bonafide files
      infile=var[ivarfile]
      realfile=file_test(infile) 
      if (realfile ne 1) then begin
        print,' ##### showex abort: variable '+trim(ivar)+' contains '+$
          infile+' which is no file'
        return
      endif else outfile=infile

; check for and extend a 2D image fits file or a 3D nt=2 fits file
; (no check for DOT or LP files: I think these are always 3D)
      barename=cgrootname(infile,extension=infileext)  ;RR needs coyotelib
      if (infileext eq 'fits') then begin
        header=headfits_rr(infile)
        ndim=sxpar(header,'NAXIS')
        nt=sxpar(header,'NAXIS3')
        if (ndim eq 2) then begin
          im1=readfits(infile)  
          im3=[[[im1]],[[im1]],[[im1]]]  
          outfile='/tmp/showex_filefake_1to3_'+file_basename(infile)
          writefits,outfile,im3 
        endif 
        if (ndim eq 3 and nt eq 2) then begin
          arr2=readfits(infile)
          im1=bytscl(arr2[*,*,0])
          im2=bytscl(arr2[*,*,1])  
          im3=[[[im1]],[[im2]],[[im2]]]
          outfile='/tmp/showex_filefake_2to3_'+file_basename(infile)
          writefits,outfile,im3 
        endif
;; ; Jun 26 2022 SSTRED "5D" fits files  ## not deeded?
;;         if (ndim eq 5) then begin
;;           nwavs=fxpar(header_in,'NAXIS3')*fxpar(header_in,'NAXIS4') 
;;           nt=fxpar(header_in,'NAXIS5')
;;           nz_file=nt*nwavs   ; !!!!  redescribe as 3D file
;;         endif
      endif
      
; jpg image
      if (infileext eq 'jpg') then begin
        read_jpeg,infile,im1,/grayscale
        im3=[[[im1]],[[im1]],[[im1]]]  
        outfile='/tmp/showex_jpg_1to3_'+barename+'.fits'
        writefits,outfile,im3 
      endif
      
; tif image
      if (infileext eq 'tif') then begin
        im1r=read_tiff(infile)
        reformimage,im1r,im1,/yreverse
        im3=[[[im1]],[[im1]],[[im1]]]  
        outfile='/tmp/showex_tif_1to3_'+barename+'.fits'
        writefits,outfile,im3 
      endif

; png image
      if (infileext eq 'png') then begin
        im1=read_png(infile)
        szim=size(im1)
        if (szim[0] eq 3) then im1=total(im1,1)
        im3=[[[im1]],[[im1]],[[im1]]]  
        outfile='/tmp/showex_png_1to3_'+barename+'.fits'
        writefits,outfile,im3 
      endif

; ps or eps figure
      if (infileext eq 'ps' or infileext eq 'eps') then begin
        spawn,'pstoimg -density 1000 -color 24 '+infile
        pngfile=file_dirname(infile)+'/'+barename+'.png'
        im1=read_png(pngfile)
        szim=size(im1)
        if (szim[0] eq 3) then im1=total(im1,1)
        im3=[[[im1]],[[im1]],[[im1]]]  
        outfile='/tmp/showex_png_1to3_'+barename+'.fits'
        writefits,outfile,im3
        spawn,'rm -f '+pngfile
      endif

; pdf figure (single page, split multipage first)
      if (infileext eq 'pdf') then begin
        spawn,'pdftops '+infile
        psfile=file_dirname(infile)+'/'+barename+'.ps'
        spawn,'pstoimg -density 1000 -color 24 '+psfile
        pngfile=file_dirname(infile)+'/'+barename+'.png'
        im1=read_png(pngfile)
        szim=size(im1)
        if (szim[0] eq 3) then im1=total(im1,1)
        im3=[[[im1]],[[im1]],[[im1]]]  
        outfile='/tmp/showex_png_1to3_'+barename+'.fits'
        writefits,outfile,im3
        spawn,'rm -f '+psfile+' '+pngfile
      endif
      
; mp4, mpg, mpeg, mov movie (slow in distilling loose jpg's)
      if (infileext eq 'mp4' or infileext eq 'mpg' or $
          infileext eq 'mpeg' or infileext eq 'mov') then begin
        spawn,'rm -f /tmp/showex_movim*jpg'
        spawn,'ffmpeg -i '+infile+' -f image2 /tmp/showex_movim%04d.jpg'
        jpgfiles=file_search('/tmp/showex_movim*.jpg')
        read_jpeg,jpgfiles[0],im1,/grayscale
        sizeim=size(im1)
        nx=sizeim[1]
        ny=sizeim[2]
        nt=n_elements(jpgfiles)
        mp4arr=bytarr(nx,ny,nt)
        for it=0,nt-1 do begin
          read_jpeg,jpgfiles[it],imit,/grayscale
          mp4arr[*,*,it]=imit[*,*]
        endfor
        outfile='/tmp/showex_mov_'+barename+'.fits'
        writefits,outfile,mp4arr
      endif
      
; put outfile into the lot
      allfilenames=[allfilenames,outfile]
    endfor ; end of loop over files within this var

  endif else begin   ; end of string treatment, this var is an array

; step array counter
    iarr=iarr+1

; get a unique fitsfile name = random with padding leading zeroes
    varname=SCOPE_VARNAME(scope_varfetch(varname),level=-1) ; orig var
    rannr=fix(abs(randomn(seed))*10000)
    filestart='/tmp/showex_'+strmid(string(rannr+1E5,format='(i6)'),1)

; check on misuse of addlabels (not applicable for arrays)
    if (n_elements(_extra) ne 0) then begin
      if (tag_exist(movexkeywords,'addlabels') ne 0) then begin
        print,' ------ showex: no /addlabels for array variables, put to zero'
        movexkeywords.addlabels=0
      endif
    endif

; get dimensions
    if (sizevar[0] eq 0) then begin
      print,' ##### showex abort: var'+trimd(ivar)+' is no good'
      return
    endif
    nx=sizevar[1]
    ny=sizevar[2]

; single 2D array = image: triplicate for movex 
    if (sizevar[0] eq 2) then begin
      var=[[[var]],[[var]],[[var]]]  
      filestart=filestart+'_2Dimage_fake_1to3'
    endif

; sequence of only two images: duplicate second for movex
    if (sizevar[0] eq 3 and sizevar[3] eq 2) then begin
      im1=bytscl(var[*,*,0])
      im2=bytscl(var[*,*,1]) 
      var=[[[im1]],[[im2]],[[im2]]]  
      filestart=filestart+'3Dcube_fake_2to3'
    endif

; sequence of 3 or more images: regular movex 
    if (sizevar[0] eq 3 and sizevar[3] gt 2) then $
      filestart=filestart+'_3Dcube'

; set sizes at first array, check subsequent
    if (iarr eq 1) then begin 
      nx0=nx
      ny0=ny
    endif else if (nx ne nx0 or ny ne ny0) then begin
      print,' ##### showex abort: var'+trimd(ivar+1)+$
        ' other dimensions than earlier variable'
      return
    endif

;  convert long or double into float (not accepted by movex.pro)
    if (sizevar[sizevar[0]+1] eq 4 or sizevar[sizevar[0]+1] eq 5) then $
      var=float(var)

; write fits file
    outfile=filestart+'_'+varname+'.fits'
    writefits,outfile,var

; add this fits copy to the lot
    allfilenames=[allfilenames,outfile]

; end of array treatment
  endelse

endfor ; end of loop over variables

; take initiall null string off (how do this more elegantly?)
nall=n_elements(allfilenames)
if (nall gt 1) then allfilenames=allfilenames[1:nall-1]

; always add labels when SDO files present or /fullmonty
if (allsdo eq 1) then addlabels=1 

; finally: start movex.pro on all, fullmonty or plain 
if (fullmonty) then begin
; full monty call
  if (montydirs[0] ne './') then begin
    if (sdodirs eq './') then sdodirs=montydirs
    if (mwfdirs eq './') then mwfdirs=montydirs
    if (fitsdirs eq './') then fitsdirs=montydirs
    if (spectdirs eq './') then spectdirs=montydirs
  endif
  movex,allfilenames,$
    /allsdo,sdodirs=sdodirs,/allmwf,mwfdirs=mwfdirs,$
    /allfits,fitsdirs=fitsdirs,/allspect,spectdirs=spectdirs,$
    /addlabels,/plotscatter,blink=blink,_extra=movexkeywords
; plain call
endif else movex,allfilenames,_extra=movexkeywords

end

; ------------------- tests per IDLWAVE Hyper C -----------------

;; ; test on demo SDO sequences
;; cd,'/home/rutten/data/SST/2016-09-05-demo/'

;; ; 3D fitscube files
;; f1600='sdo2sst/aia1600_alfor_px.fits'
;; f304='sdo2sst/aia304_alfor_px.fits'
;; f171='sdo2sst/aia171_alfor_px.fits'

;; ; 3D cube arrays
;; a1600=readfits(f1600)
;; a304=readfits(f304)
;; a171=readfits(f171)

;; ; 2D image arrays
;; im1600=a1600[*,*,0]
;; im304=a304[*,*,0]
;; im171=a171[*,*,0]

;; ; (nx,ny,2) two-image array
;; imtwo=[[[im1600]],[[im171]]]
 
;; ; 2D image files
;; writefits,'/tmp/fim1600.fits',im1600
;; writefits,'/tmp/fim304.fits',im304
;; writefits,'/tmp/fim171.fits',im171

;; ; (nx,ny,2) two-image fitscube file
;; writefits,'/tmp/fimtwo.fits',imtwo

;; ; various tests - select by  uncommenting
;; ; -------------

;; ;; showex,im1600
;; ;; showex,a1600
;; ;; showex,f1600,a1600,f304,f171,/allsdo

;; ;; showex,'sdo2sst/aia1600_alfor_px.fits','sdo2sst/aia304_alfor_px.fits'
;; ;; showex,['sdo2sst/aia1600_alfor_px.fits',$
;; ;;         'sdo2sst/aia304_alfor_px.fits'],/plotscatter 
;; ;; showex,'sdo2sst/aia1600_alfor_px.fits'

;; ;; showex,'',/allsdo
;; ;; showex,/allsdo,/plotscatter   ; all SDOs in default sdodirs = sdo2sst
;; ;; showex,a1600,'haha',/allsdo   ; haha is no file

;; ;; showex,'crispex/crispex.6563.09:48:31.time_corrected.aligned.icube',$
;; ;;   nt_mw=35

;; ;; ; full monty specified
;; ;; showex,$
;; ;;   'crispex/wb.6563.09:48:31.corrected.aligned.icube',$
;; ;;   'crispex/crispex.6563.09:48:31.time_corrected.aligned.icube',$
;; ;;   'crispex/crispex.8542.09:48:31.stokesI.icube',$
;; ;;   /allsdo,/plotscatter,intscale=-10

;; ; standard SST+SDO
;; cd,'/home/rutten/data/SST/2016-09-05-demo/'

;; showex,'sst2rotsdo/aia1600_rot_ip.fits','sst2rotsdo/aia1700_rot_ip.fits',$
;;    /addlabels,/blink

;;  showex,/allsdo,sdodirs='sst2rotsdo',/plotpower
;; showex,/full,montydirs='sst2rotsdo',spectdirs='crispex'

;; ; mp4 movie
;; cd,'/home/rutten/data/SolO/2020-07-16-first/sdomovies/test'
;; showex,['june0171.mp4','june0304.mp4']

;; ; png image
;; cd,'/home/rutten/data/SolO/2020-05-30-first'
;; ;; showex,'pressrelease/im_whole.png'
;; showex,'figs/fieldfigs/field_gongha.ps'

; tif image
cd,'/home/rutten/rr/www/rrweb/rjr-archive/observations/KPNO-shgs'
showex,'10468-3.tif'


end
