; file: selfaligncube_v1.pro
; init: Sep 22 2020  Rob Rutten  Deil 
; last: Oct  7 2020  Rob Rutten  Deil
; note: uses Metcalf p,q + poly_2d image transformation (linear, not warp)
;       works very well but I don't know how to apply p,q to larger image

;+
pro selfaligncube_v1,incube,outcube,$
  smear=smear,flatten=flatten,naverprev=naverprev,$
  alignshift=alignshift,alignmetcalf=alignmetcalf,$
  resultfile=resultfile,plotpath=plotpath,verbose=verbose

 ; self-align cube in memory successively pair-wise
 ; 
 ; INPUTS:
 ;   incube: image sequence [nx,ny,nt], any type
 ;  
 ; OPTIONAL KEYWORD INPUTS:
 ;   smear: smooth by boxcar [px] smearing
 ;   flatten: boxcar size in px to flatten = subtract boxcar-smeared
 ;   naverprev: nr aligned images before present for averaged reference
 ;   alignshift = 1/0: self-align by shifting only
 ;   alignmetcalf = 1/0: self-align per Metcalf (SLOW!)
 ;   resultfile: filestring writecol results, default '' = none
 ;   plotpath: path/filestart results graphs, default '' = none
 ;   verbose = 1/0: printout
 ;
 ; OUTPUTS:
 ;   outcube: same size and type as incube
 ;   resultfile for application to parallel cubes
 ;   plotpath files with shifts, (scales, rotate when Metcalf)
 ;
 ; REMARKS:
 ;   use selfalignfitscube.pro to only shift-align larger data cubes  
 ;   no internal Metcalf iteration here but you may iterate the call
 ;   test run underneath also shows how to apply 
 ;
 ; HISTORY:
 ;   Oct  3 2020 RR: start
;-

; answer no-parameter query 
if (n_params(0) lt 2) then begin
  sp,ratiocubefiles
  return  
endif

; defaults for keywords
if (n_elements(flatten) eq 0) then flatten=0
if (n_elements(smear) eq 0) then smear=0
if (n_elements(naverprev) eq 0) then naverprev=0
if (n_elements(alignshift) eq 0) then alignshift=0
if (n_elements(alignmetcalf) eq 0) then alignmetcalf=0
if (n_elements(resultfile) eq 0) then resultfile=''
if (n_elements(plotpath) eq 0) then plotpath=''
if (n_elements(verbose) eq 0) then verbose=0

; dimensions
szcube=size(incube)
nx=szcube[1]
ny=szcube[2]
nt=szcube[3]
outcube=incube
alimuckcube=incube

; check inputs
if (alignshift eq 0  and alignmetcalf eq 0) then begin
  print,' ##### selfaligncube_v1 abort: no shift nor metcalf align'
  return
endif
if (nx lt 1 or ny lt 1 or nt lt 1) then begin
  print,' ##### selfaligncube_v1 abort: input not dim (2,2,2) or larger'
  return
endif

; muck cube
reformcube,incube,muckcube,smear=smear,flatten=flatten

; first image initialization
muckim_prev=muckcube[*,*,0]
alimuckcube[*,*,0]=muckim_prev
missingvalue=avg(muckim_prev)

; define selfalign output arrays for writecol 
shiftx=fltarr(nt)
shifty=fltarr(nt)
allp=dblarr(4,nt)
allq=dblarr(4,nt)

; initialize for first image = start-off reference
shiftit=[0,0]
rotateit=0
scaleit=[1.,1.]

; loop over it starting at 1
for it=1,nt-1 do begin
  im=incube[*,*,it]
  im_muck=muckcube[*,*,it]

; shift align only (no rotate, scale)
  if (alignshift eq 1) then begin
    shiftit=-findimshift_rr(muckim_prev,im_muck,/subpix,filter=10)
    if (verbose ne 0) then $
      print,' ----- selfaligncube_v1 finds shifts it ='+trimd(it)+$
      '  shift ='+trimd(shiftit)
    shiftx[it]=shiftit[0]
    shifty[it]=shiftit[1]
  endif ; end shift-only mode

; full Metcalf but only 1 time; iterate program call if desired
;RR here straightforward Metcalf order, no minus pq2rss results etc
;RR unlike HMI on to SXT in findalignimages.pro (there order problem)
  if (alignmetcalf eq 1) then begin
    if (verbose ne 0) then $
      print,' ----- selfaligncube_v1 Metcalving it ='+trimd(it)
    pin=[[0,0],[1.0,0]]  
    qin=[[0,1.0],[0,0]]
;RR trial for strange solrot error initial test, didn't help
    ;; im_muck_al=$
    ;;   auto_align_images(im_muck,muckim_prev,pin,qin,pint,qint,$
    ;;                     /amoeba,/quiet,itmax=250)
    ;; im_muck_al=$
    ;;   auto_align_images(im_muck,muckim_prev,pint,qint,pout,qout,$
    ;;                     /quiet,/powell)
    im_muck_al=$
      auto_align_images(im_muck,muckim_prev,pin,qin,pout,qout,/quiet)

; copy 4-elem pout, qout coefficients for poly_2D into output files
    allp[*,it]=pout
    allq[*,it]=qout
    
; reform original image and stick into outcube 
    outim=poly_2d(im,pout,qout,2,nx,ny,cubic=-0.5)
    outcube[*,*,it]=outim

; stick aligned muck into alimuckcube for naverprev
    alimuckcube[*,*,it]=im_muck_al 

  endif ; end Metcalf mode
  
; optional alimuck averaging for next reference  
  if (naverprev eq 0) then muckim_prev=im_muck_al
  if (naverprev gt 0) then begin
    naver=min([it+1,naverprev])
    muckim_prev=total(alimuckcube[*,*,(it-naver+1):it],3)/naver
  endif
  
endfor ; end loop over it 

; ----- optional write alignment results
if (resultfile ne '') then begin
  if (alignshift eq 1) then writecol,resultfile,shiftx,shifty,fmt='(2F10.3)'
  if (alignmetcalf eq 1) then writecol,resultfile,$
    allp[0,*],allp[1,*],allp[2,*],allp[3,*],$
    allq[0,*],allq[1,*],allq[2,*],allq[3,*],fmt='(8D15.10)'
endif

; ----- optional plots alignment results

if (plotpath ne '') then begin
  if (alignmetcalf eq 1) then begin
    shiftx=fltarr(nt)
    shifty=fltarr(nt)
    scalex=fltarr(nt)+1.0
    scaley=fltarr(nt)+1.0
    rotate=fltarr(nt)
    for it=1,nt-1 do begin
      p=allp[*,it]
      q=allq[*,it]
      pq2rss,p,q,erot,exscl,eyscl,exshft,eyshft,$
        enrss,nx,ny,/center,/quiet
      rotate[it]=float(erot)
      scalex[it]=float(exscl)
      scaley[it]=float(eyscl)
      shiftx[it]=float(exshft)
      shifty[it]=float(eyshft)
    endfor
    
; Metcalf scales plot
    psfilename=plotpath+'scales.ps'
    axrat=1.62 ; golden ratio
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
    xtitle='time step'
    ytitle='scale factors'
    xrange=[0-0.2*nt,nt-1+0.2*nt]
    mima=minmax([scalex,scaley])
    range=mima[1]-mima[0]
    yrange=[mima[0]-0.1*range,mima[1]+0.1*range]
    plot,indgen(nt),scalex,$
      position=[0.2,0.2,0.95,0.95],$        ; margins
      xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
      psym=1,symsize=1.5,xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
    xyouts,-0.1*nt,scalex[0],/data,charsize=1.3,'X'
    xyouts,nt-1+0.1*nt,scalex[nt-1],/data,charsize=1.3,'X'
    oplot,indgen(nt),scaley,psym=6,symsize=1.5
    xyouts,-0.1*nt,scaley[0],/data,charsize=1.3,'y'
    xyouts,nt-1+0.1*nt,scaley[nt-1],/data,charsize=1.3,'y'
    closepsplot,psfilename,opengv=0

; Metcalf rotate plot
    psfilename=plotpath+'rotate.ps'
    axrat=1.62 ; golden ratio
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
    xtitle='time step'
    ytitle='rotation angle  [deg]'
    xrange=[0-0.2*nt,nt-1+0.2*nt]
    mima=minmax(rotate)
    range=mima[1]-mima[0]
    yrange=[mima[0]-0.1*range,mima[1]+0.1*range]
    plot,indgen(nt),rotate,$
      position=[0.2,0.2,0.95,0.95],$        ; margins
      xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
      psym=1,symsize=1.5,xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
    closepsplot,psfilename,opengv=0
  endif  

; shifts plot
  psfilename=plotpath+'shifts.ps'
  axrat=1.62 ; golden ratio
  openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
  xtitle='time step'
  ytitle='shifts  [px]'
  xrange=[0-0.2*nt,nt-1+0.2*nt]
  mima=minmax([shiftx,shifty])
  range=mima[1]-mima[0]
  yrange=[mima[0]-0.1*range,mima[1]+0.1*range]
  plot,indgen(nt),shiftx,$
    position=[0.2,0.2,0.95,0.95],$        ; margins
    xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
    psym=1,symsize=1.5,xtitle=xtitle,ytitle=ytitle,$
    xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
  xyouts,-0.1*nt,shiftx[0],/data,charsize=1.3,'X'
  xyouts,nt-1+0.1*nt,shiftx[nt-1],/data,charsize=1.3,'X'
  oplot,indgen(nt),shifty,psym=6,symsize=1.5
  xyouts,-0.1*nt,shifty[0],/data,charsize=1.3,'y'
  xyouts,nt-1+0.1*nt,shifty[nt-1],/data,charsize=1.3,'y'
  closepsplot,psfilename,opengv=0

endif ; end plot production

end ; end of program


; =============== main for testing per IDLWAVE H-c ======================

cd,'/home/rutten/data/SST/2016-09-05-demo/sst2rotsdo/' 
infile='hmicont_rot_ip.fits'
incube=readfits(infile)
smallcube=incube[0:300,0:300,0:4]
; fix in time so mucked has the same basis and should always regain it=0 
for it=0,4 do smallcube[*,*,it]=smallcube[*,*,0]
missing=avg(smallcube)

; muck the cube all three ways
szcube=size(smallcube)
nx=szcube[1]
ny=szcube[2]
nt=szcube[3]
r=randomn(seed,nt)
shifts=3.*[[r],[r]]
rotate=0.5*r
scale=[[1.0+0.03*r*indgen(nt)],[1.0+0.03*r*indgen(nt)]]

; first image no muck, 2nd fixed large muck for checking
shifts[0,0:1]=0
rotate[0]=0
scale[0,0:1]=[1.0,1.0]

shifts[1,0:1]=[4,2]
rotate[1]=3
scale[1,0:1]=[0.95,1.05]

; muck
reformcube,smallcube,shiftcube,shift=shifts,rotate=rotate,$
  missingvalue=missing,scale=scale

; parameters
smear=0
flatten=0
naverprev=3

; run twice as successive iteration
selfaligncube_v1,shiftcube,out1cube,$
   smear=smear,flatten=flatten,naverprev=naverprev,$
   /alignmetcalf,resultfile='/tmp/selfalign1.dat',$
   plotpath='/tmp/selfalign1_',/verbose
selfaligncube_v1,out1cube,out2cube,$
   smear=smear,flatten=flatten,naverprev=naverprev,$
   /alignmetcalf,resultfile='/tmp/selfalign2.dat',$
   plotpath='/tmp/selfalign2_',/verbose

; check (use showex first play button)
;; showex,smallcube,shiftcube,out1cube,out2cube

; check per application for it=1 (large muck)

it=1
inim=smallcube[*,*,it]
muckim=shiftcube[*,*,it]
out1im=out1cube[*,*,it]
out2im=out2cube[*,*,it]

readcol,'/tmp/selfalign1.dat',p1a,p2a,p3a,p4a,q1a,q2a,q3a,q4a,format='D'
pa=[p1a[it],p2a[it],p3a[it],p4a[it]]
qa=[q1a[it],q2a[it],q3a[it],q4a[it]]
try1=poly_2d(muckim,pa,qa,2,nx,ny,cubic=-0.5)
;; showex,out1im,try1,inim,muckim,/blink

;RR can't matrix combine I think = pity since double /splinip interpolation
readcol,'/tmp/selfalign2.dat',p1b,p2b,p3b,p4b,q1b,q2b,q3b,q4b,format='D'
pb=[p1b[it],p2b[it],p3b[it],p4b[it]]
qb=[q1b[it],q2b[it],q3b[it],q4b[it]]
try2=poly_2d(try1,pb,qb,2,nx,ny,cubic=-0.5)
showex,out2im,try2,out1im,try1,inim,muckim,/blink ; minimal improvement 1>2

end

