; file: reformimage.pro = manipulate (x,y) image 
; init: Aug  1 2014  Rob Rutten  Deil as extract from reformcubefile.pro
; last: Aug 11 2022  Rob Rutten  Deil
; site: rridl/imagelib
; note: many keyword options should also be implemented in reformcubefile.pro

;+
pro reformimage,inimage,outimage,$
  xrange=xrange,yrange=yrange,trimbox=trimbox,$
  greyborders=greyborders,cropborders=cropborders,$
  greytriangles=greytriangles,croptriangles=croptriangles,cropbox=cropbox,$
  xreverse=xreverse,yreverse=yreverse,$
  intrange=intrange,black2grey=black2grey,$
  intreverse=intreverse,absint=absint,$
  rebinfactor=rebinfactor,congridfactor=congridfactor,$
  splinip=splinip,$
  sqrtint=sqrtint,logint=logint,bytscale=bytscale,flt2int=flt2int,$
  nxlarge=nxlarge,nylarge=nylarge,$
  cutcentralx=cutcentralx,cutcentraly=cutcentraly,$
  shift=shift,rotate=rotate,scale=scale,missingvalue=missingvalue,$
  despike=despike,sharpen=sharpen,treatrr=treatrr,$
  flatten=flatten,histopttop=histopttop,histoptboth=histoptboth,smear=smear,$
  muckdark=muckdark,muckbright=muckbright,muckmag=muckmag,muckosc=muckosc,$
  flagerror=flagerror

 ; manipulate an image:
;    flatten, cut, resize, shift, rotate, rescale, enlargen,
 ;   sharpen, despike, flatten, change greyscale,
 ;   cutout central part, strip borders and triangles - undsoweiter
 ;
 ; INPUTS:
 ;   inimage = 2D image array
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   xrange, yrange: partial cutout specification (2-element px arrays)
 ;   trimbox = 4-elem px box [xmin,ymin,xmax,ymax] for subfield selection
 ;   greyborders = 1/0: make borders (rectangular monochrome edges) grey 
 ;   cropborders = 1/0: remove borders
 ;   cropbox = output cropping
 ;   greytriangles = 1/0: make borders and triangular edges grey
 ;   croptriangles = 1/0: remove borders and triangular edges 
 ;      the above four are only active when trimbox is not specified;
 ;      resulting frame-cut values given in cropbox = [xmin,ymin,xmax,ymax]
 ;      these are done on the input image before any rotation, rebinning etc.
 ;   xreverse, yreverse = 1/0: flip axis
 ;   intrange=[intmin,intmax]: cut intensity range at these values
 ;   black2grey: [intmin,meanint] set px intensity below 1st to 2nd
 ;   intreverse = 1/0: flip greyscale between min and max
 ;   absint = 1/0: take absolute value 
 ;   rebinfactor = [nxfac,nyfac] multiplier for integral rebinning with rebin
 ;   congridfactor = [nxfac,nyfac] multiplier to resize array with congrid
 ;      [2,2] delivers same image with 2x2 more = 2x2 finer pixels
 ;   splinip = 1 use cubic spline interpolation 
 ;           = 0 use nearest neighbour 
 ;   nxlarge,nylarge = increase field size (specify in new px after resize)
 ;   rotate, scale, shift = inputs to modified Metcalf's shift_img_rr.pro
 ;     rotate = rotate counterclockwise degrees
 ;     shift and scale 2-element vectors [x,y] components
 ;       operation in this order (forced by unrolling); scale retains (nx,ny)
 ;       scale > 1 means less arcsec/px, enlarges scene within given (nx,ny)
 ;       scale < 1 means more arcsec/px, shrinks scene within given (nx,ny)
 ;       specify shift in new px when rescaling
 ;   missingvalue = inimage intensity value for missing borders 
 ;           = -1 use mean over input image (before manipulation)
 ;           = 0  black (default)    
 ;   despike = 1/0: despike (not here; done in reformcube(file).pro
 ;   sharpen = 1/0: sharpen with convert (ImageMagick)
 ;   treatrr = 1/0: convert my way
 ;   flatten = boxcar size in input px to flatten = subtract boxcar-smeared
 ;   smear = smooth by boxcar smearing (smear = width in old px)
 ;   histopttop: histo_opt value for /top_only
 ;   histoptboth: histo_opt value for bottom and top
 ;   muckdark: set intensities below threshold expressed in mean=1 to mean
 ;     if negative then set to minimum intensity
 ;   muckbright: set intensities above threshold expressed in mean=1 to mean
 ;     when negative set instead to pixelwise random around 2*threshold 
 ;   muckmag = 1/0: muck HMI magnetogram into white only
 ;   muckosc = 1/0: remove acoustics in AIA 1600 and 1700 images
 ;     NB: muck order is: histopt, muck, smear
 ;   sqrtint = 1/0: greyscale = sqrt(int)
 ;   logint = 1/0: greyscale = alog10(int)
 ;   bytscale = 1/0: greyscale = bytscale 
 ;   flt2int = convert from float to integer (above zero, not above -32000)
 ;   cutcentralx = x-axis length of central final image part to be cut out
 ;   cutcentraly = y-axis length of central final image part to be cut out
 ;
 ; OUTPUT:
 ;   outimage = 2D array with output image
 ;
 ; OPTIONAL KEYWORD OUTPUT
 ;   cropbox = 4 element array cut values from border and triangle removal
 ;   flagerror = 1/0: gets 1 if abort error
 ;
 ; METHOD:
 ;   cut to trimbox or strip borders or borders and edges as requested
 ;   rebin or congrid, then apply Metcalf (order shift, rotate, scale)
 ;   uses the same keywords as reformcubefile.pro, use this for prior testing
 ;     (for example using findalignimages.pro to find shift and rotation)
 ;   apply results to reformcubefile.pro, eg via call doallfiles.pro 
 ; 
 ; HISTORY:
 ;   Aug  1 2014 RR: start = copy of reformcubefile.pro
 ;   Dec 14 2015 RR: rebin/congrid [xfac,yfac], congrid before Metcalf
 ;                   field expansion to nxlarge,nylarge
 ;   Apr 25 2016 RR: splinip
 ;   Mar 16 2017 RR: muckmag, muckosc
 ;   May 17 2017 RR: muckdark, muckbright, trimbox, cropborders
 ;   Dec 25 2017 RR: greytriangles, croptriangles (derot triangles)
 ;   Mar 20 2018 RR: absint
 ;   Apr 14 2018 RR: intreverse
 ;   Nov  2 2020 RR: intrange, black2grey
 ;   Feb 28 2022 RR: flatten corrected
 ;   Aug  1 2022 RR: flatten recorrected
 ;   Aug 11 2022 RR: use shift_img_rr.pro unrolled to rotate, scale, shift
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  sp,reformimage
  return
endif

; keyword defaults
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trimbox) eq 0) then trimbox=-1
if (n_elements(greyborders) eq 0) then greyborders=0
if (n_elements(cropborders) eq 0) then cropborders=0
if (n_elements(greytriangles) eq 0) then greytriangles=0
if (n_elements(croptriangles) eq 0) then croptriangles=0
if (n_elements(xreverse) eq 0) then xreverse=0
if (n_elements(yreverse) eq 0) then yreverse=0
if (n_elements(absint) eq 0) then absint=0
if (n_elements(intrange) eq 0) then intrange=-1
if (n_elements(black2grey) eq 0) then black2grey=-1
if (n_elements(intreverse) eq 0) then intreverse=0
if (n_elements(rebinfactor) eq 0) then rebinfactor=[0,0]
if (n_elements(congridfactor) eq 0) then congridfactor=[0,0]
if (n_elements(splinip) eq 0) then splinip=0
if (n_elements(nxlarge) eq 0) then nxlarge=0
if (n_elements(nylarge) eq 0) then nylarge=0
if (n_elements(shift) eq 0) then shift=[0,0]
if (n_elements(rotate) eq 0) then rotate=0
if (n_elements(scale) eq 0) then scale=[1.0,1.0]
if (n_elements(missingvalue) eq 0) then missingvalue=0
if (n_elements(despike) eq 0) then despike=0
if (n_elements(sharpen) eq 0) then sharpen=0
if (n_elements(treatrr) eq 0) then treatrr=0
if (n_elements(flatten) eq 0) then flatten=0
if (n_elements(smear) eq 0) then smear=0
if (n_elements(histopttop) eq 0) then histopttop=0
if (n_elements(histoptboth) eq 0) then histoptboth=0
if (n_elements(muckdark) eq 0) then muckdark=0
if (n_elements(muckbright) eq 0) then muckbright=0
if (n_elements(muckmag) eq 0) then muckmag=0
if (n_elements(muckosc) eq 0) then muckosc=0
if (n_elements(sqrtint) eq 0) then sqrtint=0
if (n_elements(logint) eq 0) then logint=0
if (n_elements(bytscale) eq 0) then bytscale=0 
if (n_elements(flt2int) eq 0) then flt2int=0 
if (n_elements(cutcentralx) eq 0) then cutcentralx=0
if (n_elements(cutcentraly) eq 0) then cutcentraly=0

; checks
flagerror=0
if (n_elements(inimage) eq 1 and inimage[0] eq -1) then begin
  print,'  ##### reformimage.pro abort: inimage undefined'
  flagerror=1
  return
endif
if (n_elements(rebinfactor) ne 2) then begin
  print,' ##### reformimage.pro abort: rebinfactor not 2-elem array'
  flagerror=1
  return
endif
if (n_elements(congridfactor) ne 2) then begin
  print,' ##### reformimage.pro abort: congridfactor not 2-elem array'
  flagerror=1
  return
endif
if (trimbox[0] ne -1) then begin
  if (n_elements(trimbox) ne 4) then begin
    print,' ##### reformimage.pro abort: trimbox not 4-elem array'
    flagerror=1
    return
  endif
  if (trimbox[2] lt trimbox[0] or trimbox[3] lt trimbox[1]) then begin
    print,' ##### reformimage.pro abort: trimbox format [x1,y1,x2,y2]'
    flagerror=1
    return
  endif
endif
if ((cropborders gt 0 and greyborders gt 0) or $
    (croptriangles gt 0 and greytriangles gt 0)) then begin
  print,' ##### reformimage abort: crop and grey borders both set'
  flagerror=1
  return
endif
if ((greyborders gt 0 or greytriangles gt 0 or $
     cropborders gt 0 or croptriangles gt 0) and trimbox[0] ne -1) then begin
  print,' ##### reformimage abort: grey or crop set but trimbox also'
  flagerror=1
  return
endif

; get unique temporary file identifier for sharpening or treatrr
rannr=fix(abs(randomn(seed))*10000)
strrannr=strmid(string(rannr+1E5,format='(i6)'),2)

; define temporary fileps for optional sharpen, treatrr
pnginfile='/tmp/raw'+strrannr+'.png'
pngoutfile='/tmp/sharp'+strrannr+'.png'

; get image and dimensions
image=inimage
imagesize=size(image)
nx=imagesize[1]
ny=imagesize[2]

; define x and y ranges
xmin=xrange[0]
xmax=xrange[1]
ymin=yrange[0]
ymax=yrange[1]
if (xmax eq -1) then xmax=nx-1
if (ymax eq -1) then ymax=ny-1
if (trimbox[0] ne -1) then begin
  if (trimbox[0] gt xmin) then xmin=trimbox[0]
  if (trimbox[2] lt xmax) then xmax=trimbox[2]
  if (trimbox[1] gt ymin) then ymin=trimbox[1]
  if (trimbox[3] lt ymax) then ymax=trimbox[3]
endif

; define output size for optional regridding
resize=[1.0,1.0]
if (rebinfactor[0] ne 0) then resize=rebinfactor
if (congridfactor[0] ne 0) then resize=congridfactor
nxnew=fix((xmax-xmin+1)*resize[0]+0.5)
nynew=fix((ymax-ymin+1)*resize[1]+0.5)

; manipulate image
; ----------------

; set thismissingvalue (default = average central area)
thismissingvalue=missingvalue
if (missingvalue eq -1) then $
  thismissingvalue=avg(image[nx/4:nx*3./4,ny/4:ny*3./4])

; reverse image 
if (xreverse ne 0) then image=reverse(image,1)
if (yreverse ne 0) then image=reverse(image,2)

;RR Aug  1 2022 moved back down to below cutting part
;; ; flatten if requested (full image to avoid problem large value [????])
;; if (flatten ne 0) then begin
;;   flatim=smooth(image,flatten,/edge_truncate)
;;   szim=size(image) ; get the type of variable
;;   image=image-flatim+fix(avg(image),type=szim[3]) ; convert to same type
;; endif

; cut image 
image=image[xmin:xmax,ymin:ymax]

; cut original intensity range 
if (intrange[0] ne -1) then image=image>intrange[0]<intrange[1]

; black2grey
if (black2grey[0] ne -1) then begin
  greyim=image*0+black2grey[1]
  image[where(image le black2grey[0])]=greyim[where(image le black2grey[0])]
  ;; ; check
  ;; wdelall
  ;; sv,image
  ;; window
  ;; cghistoplot,image,bins=50
endif

; absint 
if (absint) then image=abs(image)

; reverse greyscale (after absint and intrange, keep min max (not sign flip)
if (intreverse) then image=min(image)+max(image)-image

; ------- treat borders (i.e., monochrome rectangles along the sides)

if (greyborders gt 0 or cropborders gt 0 or $
    greytriangles gt 0 or croptriangles gt 0) then begin
  fullimage=image
  fullsize=size(image)
  nxfull=fullsize[1]
  nyfull=fullsize[2]
  cropbox=[0,0,nxfull-1,nyfull-1]

; detect/measure flat rectangular borders
; iterate in case one color (eg black) overlies another (eg grey)
  for itercut=1,4 do begin
    imagesize=size(image)
    nxcut=imagesize[1]
    nycut=imagesize[2]
    xcut=[0,nxcut-1]
    ycut=[0,nycut-1]
; along left side 
    for ix=0,nxcut-1 do begin
      column=image[ix,1:nycut-2] ;RR skip corner pixels (maybe nonzero)
      mom=moment(column,sdev=sdev)
      xcut[0]=ix
      if (sdev ne 0) then break
    endfor
; along right side
    for ix=nxcut-1,0,-1 do begin
      column=image[ix,1:nycut-2]
      mom=moment(column,sdev=sdev)
      xcut[1]=ix
      if (sdev ne 0) then break
    endfor
; along bottom side
    for iy=0,nycut-1 do begin
      row=image[1:nxcut-2,iy]
      mom=moment(row,sdev=sdev)
      ycut[0]=iy
      if (sdev ne 0) then break
    endfor
; along top side
    for iy=nycut-1,0,-1 do begin
      row=image[1:nxcut-2,iy]
      mom=moment(row,sdev=sdev)
      ycut[1]=iy
      if (sdev ne 0) then break
    endfor
; check that leftover image is not mostly blank 
    if (xcut[0] gt xcut[1]-10 or ycut[0] gt ycut[1]-10) then begin
; print warning (there may be many more as with eclipsed SDO)
      print,' !!!!! reformimage: whole image blank, no stripping'
      image=fullimage
      cropbox=[-2,-2,-2,-2]
      goto,NOTRIANGLES 
    endif

;; ; check within iteration
;; print,' ----- xcut ='+trimd(xcut)+'  ycut ='+trimd(ycut)
;; print,' ----- cropbox ='+trimd(cropbox)

; apply current result
    image=image[xcut[0]:xcut[1],ycut[0]:ycut[1]]
    cropbox=cropbox+[xcut[0],ycut[0],-(nxcut-1-xcut[1]),-(nycut-1-ycut[1])]

  endfor ; end iteration for multiple color borders

; apply final result
  stripimage=fullimage[cropbox[0]:cropbox[2],cropbox[1]:cropbox[3]]
  if (cropborders gt 0 or croptriangles gt 0) then image=stripimage
  grey=avg(stripimage)
  if (greyborders gt 0 or greytriangles gt 0) then begin
    fullimage[0:cropbox[0],*]=grey
    fullimage[cropbox[2]:nx-1,*]=grey
    fullimage[*,0:cropbox[1]]=grey
    fullimage[*,cropbox[3]:ny-1]=grey
    image=fullimage
  endif
endif    ; end treatment borders

; ------- treat triangular edges (after the above border stripping) 

;RR this is mucky business but by now seems fairly robust, be it ad-hoc
;RR in where to use croprotbox.pro reversedly; 
;RR also split between regular derotation triangles and irreguar triangles

if (greytriangles gt 0 or croptriangles gt 0) then begin

  szimage=size(stripimage) ; after above border stripping
  nx=szimage[1]
  ny=szimage[2]
  xblank=intarr(4)
  yblank=intarr(4)

QUADRANTREVERSE:
; get blank extents along frame sides
; lowerleft corner 
  if ((stripimage[0,0] eq stripimage[1,0]) and $
      (stripimage[0,0] eq stripimage[0,1])) then begin
    for ix=0,nx-1 do begin
      xblank[0]=ix
      if (stripimage[ix,0] ne stripimage[0,0]) then break
    endfor
    for iy=0,ny-1 do begin
      yblank[0]=iy
      if (stripimage[0,iy] ne stripimage[0,0]) then break
    endfor
  endif
; lowerright corner
  if ((stripimage[nx-1,0] eq stripimage[nx-2,0]) and $
      (stripimage[nx-1,0] eq stripimage[nx-1,1])) $
  then begin
    for ix=nx-1,0,-1 do begin
      xblank[1]=nx-1-ix
      if (stripimage[ix,0] ne stripimage[nx-1,0]) then break
    endfor
    for iy=0,ny-1 do begin
      yblank[1]=iy
      if (stripimage[nx-1,iy] ne stripimage[nx-1,0]) then break
    endfor
  endif
; upperleft corner
  if ((stripimage[0,ny-1] eq stripimage[1,ny-1]) and $
      (stripimage[0,ny-1] eq stripimage[0,ny-2])) $
  then begin
    for ix=0,nx-1 do begin
      xblank[2]=ix
      if (stripimage[ix,ny-1] ne stripimage[0,ny-1]) then break
    endfor
    for iy=ny-1,0,-1 do begin
      yblank[2]=ny-1-iy
      if (stripimage[0,iy] ne stripimage[0,ny-1]) then break
    endfor
  endif
; upperright corner
  if ((stripimage[nx-1,ny-1] eq stripimage[nx-2,ny-1]) and $
      (stripimage[nx-1,ny-1] eq stripimage[nx-1,ny-2])) $
  then begin
    for ix=nx-1,0,-1 do begin
      xblank[3]=nx-1-ix
      if (stripimage[ix,ny-1] ne stripimage[nx-1,ny-1]) then break
    endfor
    for iy=ny-1,0,-1 do begin
      yblank[3]=ny-1-iy
      if (stripimage[nx-1,iy] ne stripimage[nx-1,ny-1]) then break
    endfor
  endif

; check max blanks
  if (max(xblank) lt 3 and max(yblank) lt 3) then goto, NOTRIANGLES

; get corner monochromicity 
  blankcorner=$
; lowerleft
    [min(stripimage[0:xblank[0]/4.,0:yblank[0]/4.]) $
     eq max(stripimage[0:xblank[0]/4.,0:yblank[0]/4.]), $
; lowerright
     min(stripimage[nx-1-xblank[1]/4.:nx-1,0:yblank[1]/4.]) $
     eq max(stripimage[nx-1-xblank[1]/4.:nx-1,0:yblank[1]/4.]), $
; upperleft
     min(stripimage[0:xblank[2]/4.,ny-1-yblank[2]/4.:ny-1]) $
     eq max(stripimage[0:xblank[2]/4.,ny-1-yblank[2]/4.:ny-1]), $
; upperright
     min(stripimage[nx-1-xblank[3]/4.:nx-1,ny-1-yblank[3]/4.:ny-1]) $
     eq max(stripimage[nx-1-xblank[3]/4.:nx-1,ny-1-yblank[3]/4.:ny-1])]

; get apparent CCW rotation from each corner (also when image corners masked)
  rotan=fltarr(4)
  rotan[0]=atan(float(xblank[0])/yblank[0])   ; lower left
  rotan[1]=atan(float(yblank[1])/xblank[1])   ; lower right 
  rotan[2]=atan(float(yblank[2])/xblank[2])   ; upper left
  rotan[3]=atan(float(xblank[3])/yblank[3])   ; upper right

; maybe there are only two valid corners as happens at large rotation?
  if (product(finite(rotan)) eq 1) then finrotind=[0,1,2,3] $
  else finrotind=where(finite(rotan) eq 1)

; ---- mean of finite rotation values  
  meanrot=avg(rotan[finrotind])

; compute rotan sdev for checking spread
; (print,0.1*180./!pi = 5.7 degrees)
  if (n_elements(finrotind) gt 1) then $
    momrotan=moment(rotan[finrotind],sdev=sdev) else sdev=0.

;; ; check
;;   sv,stripimage
;;   print,' ----- xblank ='+trimd(xblank)+'  yblank ='+trimd(yblank)
;;   print,' ----- blankcorner = ',blankcorner
;;   print,' ----- rotan ='+trimd(rotan)
;;   print,' ----- finrotind ='+trimd(finrotind)
;;   print,' ----- rotan[finrotind] ='+trimd(rotan[finrotind])
;;   print,' ----- meanrot ='+trimd(meanrot)
;;   print,' ----- meanrot in degrees ='+trimd(meanrot*180./!pi)
;;   print,' ----- momrotan =',momrotan
;;   print,' ----- sdev ='+trimd(sdev)

; now test for presence of regular derotation edges
  if (finrotind[0] ne -1 and sdev le 0.1 and $ ; require small deviation
      min(blankcorner) gt 0 and $              ; exclude off-limb dark
      abs(xblank[0]-xblank[3]) lt nx/3. and $  ; if regular then near-equal
      abs(xblank[1]-xblank[2]) lt nx/3. and $
      abs(yblank[0]-yblank[3]) lt ny/3. and $
      abs(yblank[1]-yblank[2]) lt ny/3.) then begin

;;; print,' ===== regular derotation edges'

; for non-contained case = octagonal cut: get chords for cut-off corners 
; order bottom-top-left-right
    chords=[nx-xblank[0]-xblank[1],nx-xblank[2]-xblank[3],$
            ny-yblank[0]-yblank[1],ny-yblank[2]-yblank[3]]
    if (max(chords) le 3) then chords=[0,0,0,0]  ; typical when borders cut

; get nxjust, nyjust of the frame that just contains the rotated frame
    delxy=chords*sin(meanrot)*cos(meanrot)
    delx=avg([delxy[2],delxy[3]])
    dely=avg([delxy[0],delxy[1]])
    nxjust=nx+2.*delx
    nyjust=ny+2.*dely
    
;; ;check = show
;; reformimage,stripimage,justimage,nxlarge=nxjust,nylarge=nyjust,missing=1
;; while !d.window ne -1 do wdelete,!d.window
;; sv,justimage

; use croprotbox.pro to find the original image size before rotation 
;RR seems to work correctly only well away from 45 degrees so cut at 30
;RR this 30-deg cut is arbitrary, waiting for reactions on stackexchange
    if (abs(meanrot)*180./!pi le 30) then $
      croprotbox,nxjust,nyjust,meanrot*180./!pi,nxorig,nyorig $
    else begin

;;; print,' ===== get (nxorig,nyorig) the hard way'

; for larger angles reverse croprotbox doesn't work so now do this
; gritty-nitty; the resulting cropbox is less sharply maximized
; but suffices in all cases tested sofar

; first adapt quadrant, needed for using tan (one-quadrant function) below; 
; when actual rotangle was negative then use the positive value of it
; obtained from its complement 90-abs(rotangle) through
; the equivalent reverse = mirror in x; trimbox is the same for both.  
; upshot: if original image rotation was negative, find trimbox from positive 
      quadrant=[xblank[0] le yblank[0],yblank[1] le xblank[1],$
                yblank[2] le xblank[2],xblank[3] le yblank[3]]
      if (fix(mean(quadrant)+0.5) eq 0) then begin
        stripimage=reverse(stripimage)
        goto,QUADRANTREVERSE
      endif 

; find nxorig, nyorig by measuring length of rotated-image diagonal
      cornerxy=fltarr(4,2) ; [justx,justy] coordinates of rotated image corner
      diagrot=fltarr(4,2) 
; lowerleft corner
      cornerxy[0,*]=[delx+xblank[0]+dely/tan(!pi/2.-meanrot),0.]
      diagrot[0,*]=[abs((nxjust-1)/2.-cornerxy[0]),$
                    (nyjust-1)/2.]
; lowerright corner
      cornerxy[1,*]=[nxjust-1,nyjust-1-dely-yblank[3]-delx/tan(meanrot)]
      diagrot[1,*]=[abs(cornerxy[1,0]-(nxjust-1)/2.),$
                    abs((nyjust-1)/2.-cornerxy[1,1])]
; upperleft corner
      cornerxy[2,*]=[0.,dely+yblank[2]+delx/tan(meanrot)]
      diagrot[2,*]=[abs((nxjust-1)/2.-cornerxy[2,0]),$
                    abs((nyjust-1)/2-cornerxy[2,1])]
; upperright corner
      cornerxy[3,*]=[delx+xblank[2]+dely/tan(meanrot),nyjust-1]
      diagrot[3,*]=[abs(cornerxy[3,0]-(nxjust-1)/2.),$
                    abs(cornerxy[3,1]-(nyjust-1)/2.)]

; get original half-diagonal from the four estimates
      diagorig=fltarr(4,2)
      for idiag=0,3 do $
        diagorig[idiag,*]=abs(rotatevec(diagrot[idiag,*],-meanrot*180./!pi))

; now we finally have estimates for the non-rotated original dimensions
      nxorig=fix(2.*avg(abs(diagorig[*,0]))+0.5)
      nyorig=fix(2.*avg(abs(diagorig[*,1]))+0.5)

;; ; check
;;       print,' ----- nx ='+trimd(nx)+'  ny ='+trimd(ny)
;;       print,' ----- angle ='+trimd(rotan*180./!pi)
;;       print,' ----- xblank ='+trimd(xblank)+' yblank='+trimd(yblank)
;;       print,' ----- chords ='+trimd(chords)
;;       print,' ----- delx, dely ='+trimd([delx,dely])
;;       print,' ----- cornerxy_x ='+trimd(cornerxy[*,0])
;;       print,' ----- cornerxy_y='+trimd(cornerxy[*,1])
;;       print,' ----- diagrot_x ='+trimd(diagrot[*,0])
;;       print,' ----- diagrot_y ='+trimd(diagrot[*,1])
;;       print,' ----- diagorig_x ='+trimd(diagorig[*,0])
;;       print,' ----- diagorig_y ='+trimd(diagorig[*,1])
;;       print,' ----- nxjust,nyjust ='+trimd([nxjust,nyjust])
;;       print,' ----- nxorig,nyorig ='+trimd([nxorig,nyorig])

    endelse ; end switch between croprotbox and manual nxorig,nyorig finding

; get desired trimbox = common area between non-rotated and actual
    croprotbox,nxorig,nyorig,meanrot*180./!pi,nxcrop,nycrop
    xcut[0]=fix(nx/2.-nxcrop/2.)+1  ; add safety margin
    xcut[1]=fix(nx/2.+nxcrop/2.)-1
    ycut[0]=fix(ny/2.-nycrop/2.)+1
    ycut[1]=fix(ny/2.+nycrop/2.)-1

; safeguard
    if (xcut[0] lt 0) then xcut[0]=0 
    if (xcut[1] gt nx-1) then xcut[1]=nx-1
    if (ycut[0] lt 0) then ycut[0]=0
    if (ycut[1] gt ny-1) then ycut[1]=ny-1

;; ; final check
;;     print,' ----- nxorig,nyorig ='+trimd([nxorig,nyorig])
;;     print,' ----- nxcrop ='+trimd(nxcrop)+'  nycrop ='+trimd(nycrop)
;;     print,' ----- xcut='+trimd(xcut)+'  ycut='+trimd(ycut)

;; ; inspect result: show bordertrimmed original + overlaid crop frame
;;     while !d.window ne -1 do wdelete,!d.window
;;     sv,image
;;     cgplots,[xcut[0],xcut[0]],[ycut[0],ycut[1]],color='red'
;;     cgplots,[xcut[0],xcut[1]],[ycut[0],ycut[0]],color='red'
;;     cgplots,[xcut[1],xcut[1]],[ycut[0],ycut[1]],color='red'
;;     cgplots,[xcut[0],xcut[1]],[ycut[1],ycut[1]],color='red'
    
  endif else begin ; end treatment of well-behaved derotation triangles

; irregular triangles, not central-pivot derotation: strip largest one by one
; but only if they look convincing

;;    print,' ====== non-regular triangles'

    szcut=size(stripimage)
    nxcut=szcut[1]
    nycut=szcut[2]
    xcut=[0,nxcut-1]
    ycut=[0,nycut-1]
    for iter=1,4 do begin
      maxboth=max([xblank,yblank])
      if (maxboth lt 5) then break  
      maxx=where(xblank eq maxboth)
      maxy=where(yblank eq maxboth)
      maxx=maxx[0]  ;RR...IDL
      maxy=maxy[0]
      if (maxx ne -1) then begin
        if ((maxx eq 0 and blankcorner[0] gt 0 or $
             maxx eq 1 and blankcorner[1] gt 0) $
            and yblank[maxx] lt nycut/3.) then ycut[0]=ycut[0]+yblank[maxx] 
        if ((maxx eq 2 and blankcorner[2] gt 0 or $
             maxx eq 3 and blankcorner[3] gt 0) $
            and yblank[maxx] lt nycut/3.) then ycut[1]=ycut[1]-yblank[maxx]
        xblank[maxx]=0
        yblank[maxx]=0
      endif
      if (maxy ne -1) then begin
        if ((maxy eq 0 and blankcorner[0] gt 0 or $
             maxy eq 2 and blankcorner[2] gt 0) $
            and xblank[maxy] lt nxcut/3.) then xcut[0]=xcut[0]+xblank[maxy]
        if ((maxy eq 1 and blankcorner[1] gt 0 or $
             maxy eq 3 and blankcorner[3] gt 0) $
            and xblank[maxy] lt nxcut/3.) then xcut[1]=xcut[1]-xblank[maxy]
        xblank[maxy]=0
        yblank[maxy]=0
      endif
      ;; print,' ----- xblank ='+trimd(xblank)+'  yblank='+trimd(yblank)
      ;; print,' ----- nxcut ='+trimd(nxcut)+'  nycut ='+trimd(nycut)
      ;; print,' ----- xcut='+trimd(xcut)+'  ycut='+trimd(ycut)
    endfor  ; end iteration over non-regular triangles
  endelse   ; end of non-regular triangles

; apply result for both regular and (reasonably behaved) non-regular triangles
  if (xcut[1] ge xcut[0]+2 and ycut[1] ge ycut[0]+2 $
      and xcut[0] ge 0 and ycut[0] ge 0 $ 
      and xcut[1] le nxcut-1 and ycut[1] le nycut-1) then begin
    stripimage=stripimage[xcut[0]:xcut[1],ycut[0]:ycut[1]]
    cropbox=cropbox+[xcut[0],ycut[0],-(nxcut-1-xcut[1]),-(nycut-1-ycut[1])]
  endif
  if (croptriangles gt 0) then image=stripimage
  if (greytriangles gt 0) then begin
    fullimage[0:cropbox[0],*]=grey
    fullimage[cropbox[2]:nxfull-1,*]=grey
    fullimage[*,0:cropbox[1]]=grey
    fullimage[*,cropbox[3]:nyfull-1]=grey
    image=fullimage
  endif

NOTRIANGLES:
endif   ; end of extensive strip/grey borders/triangles treatments

; histopt as requested
if (histopttop ne 0) then image=histo_opt_rr(image,histopttop,/top_only)
if (histoptboth ne 0) then image=histo_opt_rr(image,histoptboth)

; apply dark and bright mucks as requested
imbd=image
meanim=avg(image)
minim=min(image)
if (muckdark ne 0) then begin
  darkim=where(image lt abs(muckdark)*(meanim-minim)+minim)
  if (darkim[0] ne -1 and muckdark gt 0) then imbd[darkim]=meanim
  if (darkim[0] ne -1 and muckdark lt 0) then imbd[darkim]=minim
endif
if (muckbright ne 0) then begin
  brightim=where(image gt abs(muckbright)*(meanim-minim)+minim)
  if (brightim[0] ne -1 and muckbright gt 0) then imbd[brightim]=meanim
  if (brightim[0] ne -1 and muckbright lt 0) then     $
    imbd[brightim]=2.*(meanim-minim)+minim+$
    randomn(seed,n_elements(brightim))*0.1*(meanim-minim)
endif
image=imbd

; muckmag if requested
if (muckmag ne 0) then begin
  imclip=0.*image
  nospot=where(abs(image) lt 7500)      ; 750 Gauss for disk-center plage
  imclip[nospot]=abs(image[nospot]) 
  image=imclip>mean(imclip)             ; take out magnetogram noise
endif

; muckosc if requested
if (muckosc ne 0) then image=image>1.2*avg(image)  

; smear if requested
if (smear ne 0) then image=smooth(image,smear,/edge_truncate)

; rebin if requested
if (rebinfactor[0] ne 0) then begin
  if (splinip eq 0) then image=rebin(image,nxnew,nynew,sample=1)
  if (splinip ne 0) then image=rebin(image,nxnew,nynew)
endif

; congrid if requested (Jul  6 2016 RR: /center necessary for huge AIA px)
if (congridfactor[0] ne 0) then begin
  if (splinip eq 0) then image=congrid(image,/center,nxnew,nynew,1)
  if (splinip ne 0) then image=congrid(image,/center,nxnew,nynew,1,cubic=-0.5)
endif

; increase field if requested (e.g., to accommodate rotation corners)
if (nxlarge ne 0 and nylarge ne 0) then begin
  imsize=size(image)
  nxim=imsize[1]
  nyim=imsize[2]
  nxlar=nxlarge ; to make unsticky
  nylar=nylarge
  ; retain current size if larger
  if (nxlar lt nxim) then nxlar=nxim
  if (nylar lt nyim) then nylar=nyim
  if (imsize[3] eq 4) then largeim=fltarr(nxlar,nylar)$
    +float(thismissingvalue)
  if (imsize[3] eq 3) then largeim=lonarr(nxlar,nylar)$
    +float(thismissingvalue)
  if (imsize[3] eq 2) then largeim=intarr(nxlar,nylar)$
    +fix(thismissingvalue)
  if (imsize[3] eq 1) then largeim=bytarr(nxlar,nylar)$
    +byte(thismissingvalue)
  dx=fix((nxlar-nxim)/2.+0.5)
  dy=fix((nylar-nyim)/2.+0.5)
  largeim[dx:dx+nxim-1,dy:dy+nyim-1]=image[*,*]
  image=largeim
endif

; shift, rotate, size-scale per Metcalf shift_img (maintains (nx,ny))
;RR now after rebin/congrid so specify shifts in new px scales
;RR Metcalf's anchor default = image center
;RR no border/triangle stripping since done above
if (n_elements(shift) ne 2) then begin
  print,' ##### reformimage abort: shift not 2-element vector but = ',shift
  flagerror=1
  return
endif
if (n_elements(scale) ne 2) then begin
  print,' ##### reformimage abort: scale not 2-element vector but = ',scale
  flagerror=1
  return
endif
;; image=shift_img_rr(image,shift,rot12in=rotate,$
;;                    xscale=scale[0],yscale=scale[1],$
;;                    missing=thismissingvalue,splinip=splinip)

;RR Aug 11 2022 unroll order rotate-shift-scale (oops 3x splinip)
; rotate
imrot=shift_img_rr(image,[0,0],rot12in=rotate,$
                 xscale=1.,yscale=1.,$
                 missing=thismissingvalue,splinip=splinip)
; shift
imshift=shift_img_rr(imrot,shift,rot12in=0.,$
                 xscale=1.,yscale=1.,$
                 missing=thismissingvalue,splinip=splinip)
; scale
image=shift_img_rr(imshift,[0,0],rot12in=0.,$
                 xscale=scale[0],yscale=scale[1],$
                 missing=thismissingvalue,splinip=splinip)

; adapt intensity if requested
if (sqrtint ne 0) then image=sqrt(image)
if (logint ne 0) then image=alog10(image+0.1)

; convert to integer if requested
if (flt2int ne 0) then image=fix(image)

; sharpen if requested  
if (sharpen ne 0) then begin
  write_png,pnginfile,image
  spawn,'convert -unsharp 3.0x1.0+1.5+0 '+pnginfile+' '+pngoutfile
  image=read_png(pngoutfile)
endif

; treatrr if requested (comand I use for my digishots)  
if (treatrr ne 0) then begin
  write_png,pnginfile,image
  spawn,'convert -contrast-stretch 0%x0% -sharpen 0x3.0 -sigmoidal-contrast 1,0% '+pnginfile+' '+pngoutfile
  image=read_png(pngoutfile)
endif

; Nov 15 2021 moved up before cutting part
; Aug  1 2022 move back here again to avoid triangles but with type setting
if (flatten ne 0) then begin
  flatim=smooth(image,flatten,/edge_truncate)
  szim=size(image)                                ; get the type of variable
  image=image-flatim+fix(avg(image),type=szim[3]) ; convert to same type
endif

; bytscale if requested
if (bytscale eq 1) then image=bytscl(image)

; cut center part out if requested
if (cutcentralx ne 0 or cutcentraly ne 0) then begin
  imsize=size(image)
  nxim=imsize[1]
  nyim=imsize[2]
  if (cutcentralx gt nxim or cutcentraly gt nyim) then begin
    print,' ##### reformimage abort: cutcentralx or y too large'
    print,'       nxim,nyim = '+trimd([nxim,nyim])+'  cutcentralx,y ='$
      +trimd([cutcentralx,cutcentraly])
    flagerror=1
    return
  endif
  if (cutcentralx gt nxim) then cutcentralx=nxim
  if (cutcentraly gt nyim) then cutcentraly=nyim
  image=$
    image[fix(nxim/2.-cutcentralx/2.+0.5):fix(nxim/2.+cutcentralx/2.+0.5)-1,$
          fix(nyim/2.-cutcentraly/2.+0.5):fix(nyim/2.+cutcentraly/2.+0.5)-1]
endif

; assign result
outimage=image

end


; =================== main for testing per IDLWAVE H-c ===================

; /croptriangles only robust if all below fare well
;; path='/home/rutten/data/SST/tests/'
;; inimage=readfits(path+'gregal_problem.fits') 
;; inimage=readfits(path+'sdo-2016-09-05-demo.fits') ; nx,ny=181,177
;; reformimage,inimage,inimage,xrange=[0,150],yrange=[0,150]  ; make square
;; reformimage,inimage,inimage,yrange=[50,150] ; hor rectangular, spot inside
;; reformimage,inimage,inimage,xrange=[0,100] ; ver rectangular

; play with this one
;; reformimage,inimage,inimage,rotate=180 ; mucks top row: 1st px nonzero
;; reformimage,inimage,inimage,rotate=-90 ; mucks both side rows; all px zero 
;; reformimage,inimage,inimage,rotate=-80 ; partial case   
;; reformimage,inimage,inimage,rotate=-32,nxlarge=250,nylarge=250 ; inside
;; reformimage,inimage,inimage,rotate=-28,nxlarge=200,nylarge=200 
;; reformimage,inimage,inimage,rotate=45,nxlarge=190,nylarge=150

; Gregal images including limb and large (but <30) time-dependent rotation 
;; inimage=readfits(path+'gregal-0_halpha-wb.fits')   ; start
;; inimage=readfits(path+'gregal-362_halpha-wb.fits') ; midway no triangles
;; inimage=readfits(path+'gregal-544_halpha-wb.fits')
;; inimage=readfits(path+'gregal-734_halpha-wb.fits') ; end

; others
;; inimage=readfits(path+'luc-quiet-0.fits')  ;; single bad triangle
;; inimage=readfits(path+'sdo-2014-06-14-small-muckim.fits') ;; two triangles

;; ; Mar 14 2019
;; inimage=lp_get('/home/rutten/data/SST/2016-09-05-demo/crispex/wb.6563.09:48:31.corrected.aligned.icube',10)

;; ;; sharpen=1
;; treatrr=1

;; ; Nov  2 2020 
;; filelist=findfile('/home/rutten/data/SDO/2014-06-14-small/target/level2/HMI*fits*')
;; infile=filelist[0]
;; spawn,'funpack -D '+infile
;; inimage=readfits(infile)
;; ;; black2grey=[3E4,5.8E4] ; found per cghistoplot; spots as dark as borders
;; ;; smear=5
;; intreverse=0
;; flatten=5

; Aug 11 2022
inimage=lp_get('/media/rutten/RRHOME/alldata/SST/2016-09-05-demo/crispex/wb.6563.09:48:31.corrected.aligned.icube',10)
flatten=5

; ------ run program

reformimage,inimage,outimage,$
  xrange=xrange,yrange=yrange,trimbox=trimbox,$
  greyborders=greyborders,cropborders=cropborders,$
  greytriangles=greytriangles,croptriangles=croptriangles,cropbox=cropbox,$
  xreverse=xreverse,yreverse=yreverse,$
  intrange=intrange,black2grey=black2grey,$
  intreverse=intreverse,absint=absint,$
  rebinfactor=rebinfactor,congridfactor=congridfactor,$
  splinip=splinip,$
  sqrtint=sqrtint,logint=logint,bytscale=bytscale,flt2int=flt2int,$
  nxlarge=nxlarge,nylarge=nylarge,$
  cutcentralx=cutcentralx,cutcentraly=cutcentraly,$
  shift=shift,rotate=rotate,scale=scale,missingvalue=missingvalue,$
  despike=despike,sharpen=sharpen,treatrr=treatrr,$
  flatten=flatten,histopttop=histopttop,histoptboth=histoptboth,smear=smear,$
  muckdark=muckdark,muckbright=muckbright,muckmag=muckmag,muckosc=muckosc,$
  flagerror=flagerror


; ----- show result
;; while (!d.window ne -1) do wdelete,!d.window
;; sv,inimage
;; cgplots,[cropbox[0],cropbox[2]],[cropbox[1],cropbox[1]],color='red'
;; cgplots,[cropbox[0],cropbox[2]],[cropbox[3],cropbox[3]],color='red'
;; cgplots,[cropbox[0],cropbox[0]],[cropbox[1],cropbox[3]],color='red'
;; cgplots,[cropbox[2],cropbox[2]],[cropbox[1],cropbox[3]],color='red'
;; sv,outimage
;; print,' ----- cropbox =',cropbox

showex,inimage,outimage,/blink  
cghistoplot,outimage

 end
