; file: reformcubefile.pro = manipulate (x,y,t) cubefile 
; init: Jun  2 2017  Rob Rutten  Deil formerly reformfitscube.pro
; last: Aug 19 2022  Rob Rutten  Deil
; todo: difference movie, boxcar movie

;+
pro reformcubefile,infile,outfile,verbose=verbose,$
; ---- most keywords of reformimage.pro for individual images
  xrange=xrange,yrange=yrange,trimbox=trimbox,$
  greyborders=greyborders,greytriangles=greytriangles,$
  xreverse=xreverse,yreverse=yreverse,$
  intrange=intrange,black2grey=black2grey,$
  intreverse=intreverse,absint=absint,$
  rebinfactor=rebinfactor,congridfactor=congridfactor,$
  splinip=splinip,$
  sqrtint=sqrtint,logint=logint,$
  bytscale=bytscale,flt2int=flt2int,intrescale=intrescale,$
  nxlarge=nxlarge,nylarge=nylarge,$
  cutcentralx=cutcentralx,cutcentraly=cutcentraly,$
  shift=shift,rotate=rotate,scale=scale,missingvalue=missingvalue,$
  despike=despike,sharpen=sharpen,flatten=flatten,$
  histopttop=histopttop,histoptboth=histoptboth,smear=smear,$
  muckdark=muckdark,muckbright=muckbright,muckmag=muckmag,muckosc=muckosc,$
  reformdenan=reformdenan,$
; ---- extra keywords for whole cube 
  nt_mw=nt_mw,trange=trange,tselect=tselect,treverse=treverse,skip=skip,$
  freezerange=freezerange,deforepix=deforepix,$
  diffdesolrot=diffdesolrot,dedesolrot=dedesolrot,$
  cubebytscale=cubebytscale,addzero=addzero,missingmean=missingmean,$
  cubehistopttop=cubehistopttop,cubehistoptboth=cubehistoptboth,$
  subtractrunmean=subtractrunmean,$
  starttime_in=starttime_in,cadence_in=cadence_in,timearr_in=timearr_in,$
  starttime_out=starttime_out,stoptime_out=stoptime_out,$
  cadence_out=cadence_out,timearr_out=timearr_out

 ; manipulate cubefile (fits or La Palma format, usually image sequence)
 ;   cut, shift, rescale, rotate, reverse, subsample, rebin, congrid, 
 ;   freeze a part, resample in time, 
 ;   take log or sqrt, bytscale, histo_opt, take absolute value,
 ;   despike, sharpen, smear, subtract running mean - and more
 ;
 ; USAGE EXAMPLES:
 ;   regularize irregular time sampling
 ;   extend duration (with freeze option) and field size (missingmean option)
 ;   pixel-align to another cube (congrid, shift, rotate, resample, cut)
 ;   change intensity scale from linear to log or sqrt
 ;   histo_opt or bytscale whole fitscube or bytscale individual images
 ;   despike or sharpen or smear
 ;   subtract running mean
 ;   differentially derotate a non-tracked or tracked fitscube
 ;   convert "La Palma" crispex.xxxx.icube file into crispex.xxxx.fits file
 ;
 ; INPUT:
 ;   infile = string with path/filename of the 3D input cube in a cubefile
 ;            (format fits or SST "La Palma", [nx,ny,nt] or [nx,ny,nt+nwav]
 ;
 ; OUTPUT:
 ;   outfile = string with path/filename for output file 
 ;             (.fits or "La Palma" .xcube; may differ from infile)
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   verbose = 1/0: print info or not
 ; ---- keywords of reformimage.pro used for individual image reform
 ;      except croptriangles (since border widths may vary between frames)
 ;   xrange, yrange = partial size cutout specification (2-element arrays)
 ;   trimbox = 4-elem px box [xmin,ymin,xmax,ymax] to ignore eg triangle edges
 ;   greyborders = 1/0: make edge borders grey = image mean (default 0)
 ;   greytriangles = 1/0: also make additional triangular edges grey
 ;   xreverse, yreverse = 1/0: reverse x or y axis
 ;   intrange=[intmin,intmax]: cut intensity range at these values
 ;   black2grey: [intmin,meanint] set px intensity below 1st to 2nd
 ;   intreverse = 1/0: reverse greyscale
 ;   absint = 1/0: take absolute value (e.g., Dopplergrams)
 ;   rebinfactor = multiplier for rebin; integer [x,y] (not (nt,2) array)
 ;   congridfactor = multiplier for congrid; float [x,y] (not (nt,2) array)
 ;   splinip = 0: nearest neighbor (keep coarse pixels, default)
 ;           = 1: cubic spline interpolation in poly_2d and congrid
 ;   nxlarge,nylarge =  increase field size (specify in new px after resize)
 ;   shift, rotate, scale = inputs to reformimage.pro (=> shift_img.pro)
 ;     shift = vector [shiftx,shifty] (time-constant)
 ;             or (nt_in,2) array [shiftx,shifty] at intiming moments
 ;             or (nt_out,2) array [shiftx,shifty] at outtiming moments
 ;             values must be regridded/rescaled new px sizes
 ;     rotate = counterclockwise angle in degrees to rotate over
 ;              number or array [nt_in] or array [nt_out]
 ;     scale = vector [scalex,scaley] or array [nt_in,2] or [nt_out,2]
 ;   missingvalue: greyscale value for missing corners/extensions
 ;     missingvalue=-1: use mean per individual image
 ;     missingmean (below) overwrites missingvalue when enabled
 ;   despike = value: value = sigma (e.g. IRIS 4.5) for particle hit removal
 ;   sharpen = 1/0 (using fixed ImageMagic parameters)
 ;   flatten = boxcar size in input px to flatten by subracting boxcar-smeared
 ;   smear = smooth 2D with boxcar = 1D smear [width old px], maybe nt array 
 ;   histopttop: histo_opt value per image, uses /top_only
 ;   histoptboth: histo_opt value per image, bottom and top
 ;   muckdark: set intensities to mean below threshold expressed in mean=1
 ;   muckbright: set intensities to mean above threshold expressed in mean=1
 ;   muckmag = 1/0: muck HMI magnetogram into white only
 ;   muckosc = 1/0: remove acoustics in AIA 1600 and 1700 images
 ;   refromdenan = 1/0: reform and deNaN (for bad SSTNEW files) 
 ;     NB: muck order is: histopt, muck, smear
 ;   sqrtint = 1/0: convert image intensities into  sqrt(int)
 ;   logint = 1/0: convert image intensities into alog10(int)
 ;   cubebytscale = various options for bytscaling the image intensities
 ;     cubebytscale=0: don't bytscale (retain type) - default
 ;     cubebytscale=1: use (min,max) of first image for whole cube
 ;     cubebytscale=2: use (min, max) of the whole image sequence
 ;     cubebytscale=3: bytscale every image individually
 ;     cubebytscale=4: use (min, max) from histo_opt_rr(mean(cube))
 ;   flt2int = 1/0: float cube > integer
 ;   intrescale = 1/0: rescale to range max(abs(min),max)=32000
 ;     use to increase range for sqrtint, logint, flt2int
 ;   cutcentralx = x-axis length of central image part to be cut out in new px
 ;   cutcentraly = y-axis length of central image part to be cut out in new px
 ; ----- additional keywords for temporal business:
 ;   nt_mw = nr time steps needed for multi-wavelength La Palma F-P files
 ;   trange: cut time span [it1,it2]
 ;   tselect: index array = retain only these infile it time steps
 ;   treverse = 1/0: reverse time axis
 ;   skip = skip images (eg skip=1: use every second)
 ;     trange and skip work only if no new time sampling is requested
 ;   freezerange = set images during freezerange=[t1,t2] equal to image[t1]
 ;   deforepix = undo angular pixel stretching from foreshortening 
 ;   diffdesolrot = 1/0: differentially de-solar-rotate non-tracked fitscube
 ;   dedesolrot = 1/0: undo prior center-field tracking in diffdesolrot
 ;   addzero = add this value to the zero level (eg +32768 for IRIS)
 ;   missingmean = 1/0: use cube mean for blank corners and a weakened first
 ;     image for output times prior to or after the input timing
 ;     not allowed for multi-wavelength (Fabry-Perot) files (CRISP, IBIS)
 ;   cubehistopttop = value: histo_opt top cutoff (eg 1E-3) for sequence mean
 ;   cubehistoptboth = value: histo_opt cutoff (eg 1E-3) for sequence mean 
 ;   subtractrunmean = nr of preceding frames of which to subtract the mean
 ;   starttime_in: needed if no "STARTTIM" in header or irregular cadence 
 ;   cadence_in = data cadence in s, needed if not in header nor timearr_in
 ;   timearr_in = 1D array of individual image times in input file, in TAI 
 ;   starttime_out: of new fixed-cadence sequence, eg '2014-01-10 12:03:00'
 ;   stoptime_out: of new fixed-cadence sequence, eg '2014-01-10 12:09:00'
 ;   cadence_out: of new cadence, in seconds; if -1 then old cadence
 ;   timearr_out = 1D array of desired new individual image times in TAI 
 ;     timearr's: double-precision arrays, eg from anytim2tai("date-time")
 ;     input timing: cadence_in and timearr_in are mutually exclusive
 ;     new output timing: either start+stop+cadence or timearr_out
 ;     when timearr specified the start and stop times are taken from array
 ;     when output timing starts before or ends after input timing the image
 ;       is frozen at first or last image, respectively (to not upset ximovie)
 ;
 ; METHOD:
 ;   largely a wrapper to reformimage.pro but without keyword inheritance
 ;   because multiple keywords are needed before to set up the output assoc. 
 ;   Uses the same keywords as reformimage.pro, use that for prior testing.
 ;   Uses assoc input and output to accommodate large files exceeding memory.
 ;   Suited to call by doallfiles.pro for processing multiple similar cubes
 ;
 ; HISTORY:
 ;   Nov 22 2013 RR: start as reformfitscube.pro
 ;   Nov 23 2013 RR: added ImageMagick sharpening
 ;   Dec  2 2013 RR: added Metcalf shift, scale, rotate 
 ;   Jul 28 2014 RR: added resize with congrid
 ;   Aug 17 2014 RR: options new cadence, duration, timing array 
 ;   Aug 18 2014 RR: time-dependent shifts
 ;   Aug 31 2014 RR: rewrote timing part
 ;   Sep  6 2014 RR: added subtractrunmean = difference movie 
 ;   Nov  3 2014 RR: added reverse, flt2int (for Sunrise/IMAX)
 ;   Feb 14 2015 RR: added freezerange
 ;   Sep 28 2015 RR: permit space in dirpath (for Mac owners)
 ;   Oct  6 2015 RR: added cubehistop, cubebytscale=4, despike
 ;   Dec 18 2015 RR: changed to calling reformimage.pro
 ;   Jan 10 2016 RR: improved keyword treatment
 ;   Jan 15 2016 RR: shift, rotate, scale may also be nt_out arrays
 ;   Jul  7 2016 RR: splinip = choice of grid interpolation
 ;   Mar 16 2017 RR: muckmag, muckosc
 ;   May 29 2017 RR: more mucks 
 ;   Jun  2 2017 RR: also La Palma files, so name fitscube > cubefile
 ;   Dec 19 2017 RR: greytriangles
 ;   Jan 17 2018 RR: permit multi-wavelength (Fabry-Perot) files
 ;   Mar 20 2018 RR: absint
 ;   Apr 14 2018 RR: intreverse
 ;   Dec 13 2018 RR: smeared prior/post extension placeholders
 ;   Dec 24 2019 RR: intrescale
 ;   Jan  2 2020 RR: rotation
 ;   Jul 13 2020 RR: diffdesolrot, dedesolrot
 ;   Jul 21 2020 RR: deforepix
 ;   Oct 26 2020 RR: smear as array, tselect
 ;   Nov  2 2020 RR: intrange, black2grey
 ;   Feb 24 2022 RR: .fts for .fits for SOLIS
 ;   Jun 23 2022 RR: accommodate "5D" fits files from new SSTRED pipe
 ;   Jun 26 2022 RR: reformdenan for SSTRED files
 ;   Aug 19 2022 RR: more adaptations SSTRED files
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  sp,reformcubefile
  return
endif

; ---- top keywords
if (n_elements(verbose) eq 0) then verbose=0 else verbose=1
; ---- keyword defaults for reformimage.pro   ## keep the same
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(greyborders) eq 0) then greyborders=0
if (n_elements(greytriangles) eq 0) then greytriangles=0
if (n_elements(trimbox) eq 0) then trimbox=-1
if (n_elements(xreverse) eq 0) then xreverse=0
if (n_elements(yreverse) eq 0) then yreverse=0
if (n_elements(intrange) eq 0) then intrange=-1
if (n_elements(black2grey) eq 0) then black2grey=-1
if (n_elements(intreverse) eq 0) then intreverse=0
if (n_elements(absint) eq 0) then absint=0
if (n_elements(rebinfactor) eq 0) then rebinfactor=[0,0]
if (n_elements(congridfactor) eq 0) then congridfactor=[0,0]
if (n_elements(splinip) eq 0) then splinip=0
if (n_elements(nxlarge) eq 0) then nxlarge=0
if (n_elements(nylarge) eq 0) then nylarge=0
if (n_elements(shift) eq 0) then shift=[0,0]
if (n_elements(rotate) eq 0) then rotate=0
if (n_elements(scale) eq 0) then scale=[1.0,1.0]
if (n_elements(missingvalue) eq 0) then missingvalue=0
if (n_elements(despike) eq 0) then despike=0
if (n_elements(sharpen) eq 0) then sharpen=0
if (n_elements(flatten) eq 0) then flatten=0
if (n_elements(smear) eq 0) then smear=0
if (n_elements(histopttop) eq 0) then histopttop=0
if (n_elements(histoptboth) eq 0) then histoptboth=0
if (n_elements(muckdark) eq 0) then muckdark=0
if (n_elements(muckbright) eq 0) then muckbright=0
if (n_elements(muckmag) eq 0) then muckmag=0
if (n_elements(muckosc) eq 0) then muckosc=0
if (n_elements(reformdenan) eq 0) then reformdenan=0
if (n_elements(sqrtint) eq 0) then sqrtint=0
if (n_elements(logint) eq 0) then logint=0
if (n_elements(bytscale) eq 0) then bytscale=0 
if (n_elements(flt2int) eq 0) then flt2int=0 
if (n_elements(intrescale) eq 0) then intrescale=0
if (n_elements(addzero) eq 0) then addzero=0
if (n_elements(cutcentralx) eq 0) then cutcentralx=0
if (n_elements(cutcentraly) eq 0) then cutcentraly=0
; ---- keyword defaults for temporal cube business
if (n_elements(nt_mw) eq 0) then nt_mw=0
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(tselect) eq 0) then tselect=[0,-1]
if (n_elements(treverse) eq 0) then treverse=0
if (n_elements(skip) eq 0) then skip=0
if (n_elements(freezerange) eq 0) then freezerange=[0,-1]
if (n_elements(deforepix) eq 0) then deforepix=0
if (n_elements(diffdesolrot) eq 0) then diffdesolrot=0
if (n_elements(dedesolrot) eq 0) then dedesolrot=0
if (n_elements(missingmean) eq 0) then missingmean=0
if (n_elements(cubebytscale) eq 0) then cubebytscale=0
if (n_elements(cubehistopttop) eq 0) then cubehistopttop=0
if (n_elements(cubehistoptboth) eq 0) then cubehistoptboth=0
if (n_elements(subtractrunmean) eq 0) then subtractrunmean=0
if (n_elements(starttime_in) eq 0) then starttime_in=0
if (n_elements(cadence_in) eq 0) then cadence_in=0
if (n_elements(timearr_in) eq 0) then timearr_in=0
if (n_elements(starttime_out) eq 0) then starttime_out=0
if (n_elements(stoptime_out) eq 0) then stoptime_out=0
if (n_elements(cadence_out) eq 0) then cadence_out=0
if (n_elements(timearr_out) eq 0) then timearr_out=0

; ========== start

; print pacifier
if (verbose eq 1) then print,' ----- reformcubefile starts on '+infile

; set endian for this computer 
;RR l = little, b = big; fits files are big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'

; get unique temporary file identifier for sharpening 
rannr=fix(abs(randomn(seed))*10000)
strrannr=strmid(string(rannr+1E5,format='(i6)'),2)

; define temporary files for optional sharpening
pnginfile='/tmp/raw'+strrannr+'.png'
pngoutfile='/tmp/sharp'+strrannr+'.png'

; find whether input file is fits of SOLIS fts or La Palma format
dummy=cgrootname(infile,extension=ext)  ;RR needs coyotelib
if (ext eq 'fits' or ext eq 'fts') then fitsinput=1 else fitsinput=0
if (ext eq 'icube' or ext eq 'fcube' or ext eq 'bcube') $
then lpinput=1 else lpinput=0
if (fitsinput eq 0 and lpinput eq 0) then begin
  print,' ##### reformcubefile abort: infile extension not fits nor La Palma'
  return
endif

; default = single-wavelength (no FP wavelength, no Stokes) file 
nchan=1

;; ; ===== input = fits file

if (fitsinput ne 0) then begin

; get cube geometry and file datatype from the fits header
  header_in=headfits_rr(infile)
  header_insize=(1+fix(n_elements(header_in)/36.))*2880
  bitpix_in=fxpar(header_in,'bitpix')
  endian_in='b'  ; fits files are always big endian
  ndims=fxpar(header_in,'naxis')
  nx_in=fxpar(header_in,'naxis1') 
  ny_in=fxpar(header_in,'naxis2') 

; classical 3D fits, my SDO fits, sanitized SSTRED fits
  if (ndims eq 3) then begin
    nz_file=fxpar(header_in,'naxis3')
    nt_in=nz_file
    nrchan=fxpar(header_in,'nrchan')
; sanitized SSTRED, if no nrchan keyword then not sanitized and zero
    if (nrchan ne 0) then begin
      nchan=nrchan
      nt_in=nz_file/nrchan
      nt_mw=nt_in
    endif
  endif

;RR Jun 23 2022 new "5D" SSTRED files
;RR are [nx,ny,nwav,npol,nt] even when nwav and/or npol 1 (eg wide band)
;RR put n_lambda * n_Stokes into nz_file describing normal 3D fits file
;RR (I guess it actually is for assoc). They are also float, how stupid.
;RR And have NaN outside the detector frame for jitter bands and rotation
;RR triangles, very nasty in IDL.  Rotation to North bad idea anyhow.
  if (ndims eq 5) then begin
    nwavs=fxpar(header_in,'naxis3')
    nstokes=fxpar(header_in,'naxis4')
    nchan=nwavs*nstokes
    nt_in=fxpar(header_in,'naxis5')
    nz_file=nt_in*nchan   ; redescribe as 3D file
  endif
  
; check
  if (nx_in eq 0 or ny_in eq 0 or nz_file eq 0) then begin
    print,' ##### reformcubefile abort: no nx, ny, nz in fits header '+infile
    return
  endif
  
;  get key SDO-style keywords (as in RR SDO cubes, they become 0 otherwise)
  starttime_inhead=fxpar(header_in,'starttim') ; start date-time string
  cadence_inhead=fxpar(header_in,'cadence')    ; cadence in seconds
  cdelt1ref_in=fxpar(header_in,'cdelt1')       ; x-scale [arcsec/px]
  cdelt2ref_in=fxpar(header_in,'cdelt2')       ; y-scale [arcsec/px]
  xcenarcref_in=fxpar(header_in,'xcen') ; FOV center X arcsec from sun center 
  ycenarcref_in=fxpar(header_in,'ycen') ; FOV center Y arcsec from sun center
  crota2_in=fxpar(header_in,'crota2')   ; FOV rotation angle deg NESW = CCW
  
endif  ; ----- end of fitsinput startup

; ===== input = LP file

; get old-type La Palma file cube geometry from the minimal header
if (lpinput ne 0) then begin

; get header
  lp_readheader,infile,header=header_in,datatype=datatype_in, $
    dims=dims,nx=nx_in,ny=ny_in,nt=nz_file,endian=endian_in
; example inheader (miminal indeed, wavelengths folded into time: 
;    datatype=2 (integer), dims=3, nx=976, ny=960, nz=525, endian=l
;    in this case nt=35, nwav=15, no Stokes (otherwise 4).  NOT in file info..
; check
  if (nx_in eq 0 or ny_in eq 0 or nz_file eq 0) then begin
    print,$
      ' ##### reformcubefile abort: no nx, ny, nt in La Palma header '+infile
    return
  endif

; set bitpix
  if (datatype_in eq 1) then bitpix_in=8
  if (datatype_in eq 2) then bitpix_in=16
  if (datatype_in eq 4) then bitpix_in=-32

; set header size
  header_insize=512

; no timing parameters in headers La Palma files
  starttime_inhead=0
  cadence_inhead=0

; set nt_in, nwavs, nchan for classic LP file
;RR no application of trange and skip when timearr_in given  RR??
;RR if no nt_mw or timearr_in then La Palma FP files are done image by image
;RR upsetting any reforms having to do with time (old La Palma
;RR CRISP files stupidly have no nwavs keyword)
  if (lpinput eq 1) then begin
    if (nt_mw ne 0) then nt_in=nt_mw
    if (nt_mw eq 0 and timearr_in[0] eq 0) then nt_in=nz_file else begin
      if (nt_mw eq 0) then nt_in=n_elements(timearr_in)
      if (nt_in ne nz_file and nchan lt 2) then begin
        nchan=nz_file/nt_in
        print,' ----- reformcubefile: unequal nt_file, timearr_in: assume '+$
          infile+' = La Palma Fabry-Perot file'+' with nt ='+trimd(nt_in)+$ 
          ' nw ='+trimd(nchan)
      endif
      if (timearr_in[0] ne 0) then begin
        cadence_in=(timearr_in[nt_in-1]-timearr_in[0])/float(nt_in-1)
        if (cadence_inhead ne 0 and abs(cadence_in-cadence_inhead) gt 0.1) $
        then begin
          print,'##### reformcubefile abort: '+$
            'cadences timearr_in, header differ for file '+infile
          return
        endif
      endif
    endelse
  endif
endif ; ----- end of LP file startup

; set input timing switches
incadence=(starttime_in ne 0)*(cadence_in ne 0)
intiming=(timearr_in[0] ne 0 or incadence eq 1)

; check input timing specifications
if ((starttime_in ne 0) ne incadence) then begin
  print,' ##### reformcubefile abort: specify starttime_in and cadence_in'$
    +' both or none'
  return
endif
if (intiming eq 1 and (trange[1] ne -1 or skip ne 0)) $
then begin print,' ####### reformcubefile abort: trange or skip'+$
  ' conflicts with specifying intiming'
  return
endif

; xrange and yrange; trimbox
xmin=xrange[0]
xmax=xrange[1]
ymin=yrange[0]
ymax=yrange[1]
if (xmax eq -1) then xmax=nx_in-1
if (ymax eq -1) then ymax=ny_in-1
if (trimbox[0] ne -1) then begin
  if (trimbox[0] gt xmin) then xmin=trimbox[0]
  if (trimbox[2] lt xmax) then xmax=trimbox[2]
  if (trimbox[1] gt ymin) then ymin=trimbox[1]
  if (trimbox[3] lt ymax) then ymax=trimbox[3]
endif
nx_out=xmax-xmin+1  ; needed to open unit for assoc out
ny_out=ymax-ymin+1

; trange
itstart=trange[0]
itend=trange[1]
if (itend eq -1) then itend=nt_in-1 

; redefine output image size if regridding requested
; (use first pair if arrays)
resize=[0,0]
pxratio=[1.,1.]
if (rebinfactor[0] ne 0) then resize=rebinfactor     ; 2-element vector
if (congridfactor[0] ne 0) then resize=congridfactor ; 2-element vector
if (resize[0] ne 0) then begin
  nxnew=fix((xmax-xmin+1)*resize[0]+0.5)
  nynew=fix((ymax-ymin+1)*resize[1]+0.5)
  nx_out=nxnew
  ny_out=nynew
  pxratio=1./resize  ; 2-element vector
endif

; redefine output image size if enlargement requested
if (nxlarge ne 0) then nx_out=nxlarge
if (nylarge ne 0) then ny_out=nylarge

; redefine output image size if central cutout requested
if (cutcentralx ne 0) then nx_out=cutcentralx
if (cutcentraly ne 0) then ny_out=cutcentraly

; warning if diffdesolrot but no fitscube
if (diffdesolrot eq 1 and cadence_inhead eq 0) then begin
  print,' ----- recormcubefile can only diffdesolrotate RR fitscubes'
  print,'       option diffdesolrot set to zero'
  diffdesolrot=0
endif

; redefine output image size if deforepix requested, to its first result
; (deforepix application should be in reformimage with here only dimension 
; setting for outfile, but reformimage doesn't know xcen, ycen)
if (deforepix eq 1) then begin
  if (cadence_inhead eq 0) then begin
    print,' ----- reformcubefile: not RR-style fitscube so no (X,Y)'
    print,' ----- option deforepix therefore changed to zero'
    deforepix=0
  endif else begin
    fakeim=bytarr(nx_in,ny_in)
    forward_function deforepix  ;RR why needed? f**k IDL
    deforim=deforepix(fakeim,xcenarcref_in,ycenarcref_in, $
                      starttime_inhead,orientang=-crota2_in)
    szdefor=size(deforim)
    nx_defor=szdefor[1]
    ny_defor=szdefor[2]
    nx_out=nx_defor
    ny_out=ny_defor
  endelse
endif

; transpose nt_in shift and scale arrays for straddle interpolation
if (n_elements(shift) eq  2*nt_in) then $
  tpshift=transpose(shift) else tpshift=shift
if (n_elements(scale) eq  2*nt_in) then $
  tpscale=transpose(scale) else tpscale=scale

; open input file for assoc
get_lun, unit_in
openr,unit_in,infile,swap_endian=(endian ne endian_in)
if (bitpix_in eq -32) then $
  inassoc=assoc(unit_in,fltarr(nx_in,ny_in),header_insize)
if (bitpix_in eq 16) then $
  inassoc=assoc(unit_in,intarr(nx_in,ny_in),header_insize)
if (bitpix_in eq 8) then $
  inassoc=assoc(unit_in,bytarr(nx_in,ny_in),header_insize)

; check input image size
im0=inassoc[0]
sizeim0=size(im0)
nx_im0=sizeim0[1]
ny_im0=sizeim0[2]
if (nx_im0 ne nx_in or ny_im0 ne ny_in) then begin
  print, ' ##### reformcubefile abort: nx or ny header not 1st image [nx,ny]' 
  return
endif

; =========== begin timing stuff (only for fits files?)

; =========== input timing

; timearr_in not specified: set from keywords or set fake 
if (timearr_in[0] eq 0) then begin
  if (starttime_in eq 0) then starttime_in=starttime_inhead
  if (starttime_in eq 0) then starttime_in='1942-21-04 00:00:00' ; fake
  if (cadence_in eq 0) then cadence_in=cadence_inhead
  if (cadence_in eq 0) then cadence_in=10                       ; fake 
  timearr_in=anytim2tai(starttime_in)+indgen(nt_in)*cadence_in  ; may be fake
endif

;; ; check
;; print,nt_in,anytim2utc(timearr_in,/ccsds)
;; STOP

; ============ output timing 

; set output timing switches
outcadence=((starttime_out ne 0)*(stoptime_out ne 0)*(cadence_out ne 0)) 
outtiming=(timearr_out[0] ne 0 or outcadence eq 1)

; check output timing specifications
if (outcadence eq 0 and (starttime_out ne 0 or stoptime_out ne 0 or $
                         cadence_out ne 0)) then  begin
  print,' ##### reformcubefile abort: specify starttime_out, stoptime_out,'$
    +' cadence_out all three or none (cadence_out=-1 for cadence_in)'
  return
endif
if (outtiming eq 1 and (trange[1] ne -1 or skip ne 0)) then begin
  print,' ##### reformcubefile abort: trange or skip conflict out timing'
  return
endif

; no new timing specified: accommodate trange and skip if specified
if (outtiming eq 0) then begin
  if (verbose eq 1) then print,' ----- use old timing, no time interpolation'
  timearr_out=timearr_in 
  starttime_out=anytim2utc(timearr_in[itstart],/ccsds)
  nt_out=fix(float(itend-itstart)/(skip+1))+1
  cadence_out=cadence_in*(skip+1)
endif
;; ; check
;; print,anytim2utc(timearr_out,/ccsds)
;; STOP

; new timing specified, either timearr or start, stop, cadence
if (outtiming eq 1) then begin
  if (verbose eq 1) then $
    print,' ----- interpolation to new timing ### takes long'

; timearr_out specified; get values for output header
  if (timearr_out[0] ne 0) then begin
    starttime_out=anytim2utc(timearr_out[0],/ccsds)
    nt_out=n_elements(timearr_out)
    cadence_out=(timearr_out[nt_out-1]-timearr_out[0])/float(nt_out-1)
  endif

; instead starttime_out & stoptime_out & cadence_out specified
  if (outcadence eq 1) then begin
    starttai_out=anytim2tai(starttime_out) 
    stoptai_out=anytim2tai(stoptime_out) 
    if (cadence_out eq -1) then cadence_out=cadence_in
    nt_out=fix((stoptai_out-starttai_out)/cadence_out)+1
    timearr_out=starttai_out+indgen(nt_out)*cadence_out
  endif

endif

; only tselect passing output: shorten nt_out > nt_outfile
if (tselect[1] ne -1) then nt_outfile=n_elements(tselect) else $
  nt_outfile=nt_out

;; ; check
;; print,nt_out,anytim2utc(timearr_out,/ccsds)
;; STOP


; =========== start output file

; find whether output file is fits or La Palma format
fitsoutput=0
lpoutput=0
dummy=cgrootname(outfile,extension=ext)  ;RR needs coyotelib
if (ext eq 'fits') then fitsoutput=1 else $
  if (ext eq 'icube' or ext eq 'fcube' or ext eq 'bcube') then lpoutput=1
if (fitsoutput eq 0 and lpoutput eq 0) then begin
  print,' ##### reformcubefile abort: output file = '+outfile+$
    ' extension not fits nor La Palma'
  return
endif

; set bitpix_out = word type
bitpix_out=bitpix_in
if (cubebytscale ne 0) then bitpix_out=8
if (sharpen ne 0 and cubebytscale eq 0) then bitpix_out=16
if (flt2int) then bitpix_out=16

; make output header 
if (fitsoutput ne 0) then begin
  endian_out='b'

  if (fitsinput eq 0) then begin
; no fits input: make minimal fits output header
    if (bitpix_out eq 8) then outtype=1
    if (bitpix_out eq 16) then outtype=2
    if (bitpix_out eq -32) then outtype=4
    mkhdr, header_out, outtype,[nx_out,ny_out,nt_outfile*nchan]
    header_outsize=(1+fix(n_elements(header_out)/36.))*2880  
  endif

  if (fitsinput ne 0) then begin
; copy and modify input header (eg whole SDO one)
    header_out=header_in
    endian_out='b'

; modify dimensions (sxaddpar can add but also modify, RR = warning flag)
    sxaddpar,header_out,'naxis1',nx_out,'RR mod? reform'
    sxaddpar,header_out,'naxis2',ny_out,'RR mod? reform'
    sxaddpar,header_out,'naxis3',nt_outfile*nchan,'RR nrtimes*nchan reform'
    sxaddpar,header_out,'bitpix',bitpix_out,'RR mod? reform'
;;    sxaddpar,header_out,'nchan',nchan,'RR nr channels reform' ;RR wrong
;;    sxaddpar,header_out,'nrtimes',nt_outfile,'RR nr times reform' ;RR wrong
    sxaddpar,header_out,'crota2',crota2_in+mean(rotate),'RR mod reform'
    
; modify timing keywords if changed
    if (intiming ne 0) then begin
      sxaddpar,header_out,'starttim',starttime_in,'RR first mid-exposure'
      sxaddpar,header_out,'cadence',cadence_in,'RR cadence [s]'
    endif  
    if (outtiming ne 0) then begin
      sxaddpar,header_out,'starttim',starttime_out,'RR first mid-exposure'
      sxaddpar,header_out,'cadence',cadence_out,'RR cadence [s]'
    endif

; adapt XCEN, YCEN keywords for output when xrange or yrange set
    xcenarcref_out=xcenarcref_in
    ycenarcref_out=ycenarcref_in
    if (xrange[1] ne -1) then $
      xcenarcref_out=xcenarcref_in+cdelt1ref_in*(xrange[1]-xrange[0])/2.
    if (yrange[1] ne -1) then $
      ycenarcref_out=ycenarcref_in+cdelt2ref_in*(yrange[1]-yrange[0])/2.

; adapt CDELT keywords
    cdelt1ref_out=cdelt1ref_in*pxratio[0]
    cdelt2ref_out=cdelt2ref_in*pxratio[1]
    
; adapt XCEN, YCEN keywords for fixed shift (### WRONG when also rotation!!) 
    if (n_elements(shift) eq 2 and max(shift) ne 0) then begin
      xcenarcref_out=xcenarcref_out+shift[0]*cdelt1ref_out
      ycenarcref_out=ycenarcref_out+shift[1]*cdelt2ref_out
    endif

; stick into header_out
    sxaddpar,header_out,'cdelt1',cdelt1ref_out,'RR x-scale [arcsec/px]',$
      format='f12.5'
    sxaddpar,header_out,'cdelt2',cdelt2ref_out,'RR y-scale [arcsec/px]',$
      format='f12.5'
    sxaddpar,header_out,'xcen',xcenarcref_out,$
      'RR FOV center X [" from sun center] no rot?'
    sxaddpar,header_out,'ycen',ycenarcref_out,$
      'RR FOV center Y [" from sun center] no rot?'

; add diffdesolrot and dedesolrot into header when set
    if (diffdesolrot ne 0) then $
      sxaddpar,header_out,'diffdesolrot',diffdesolrot,$
      'RR differential derotation'
    if (dedesolrot ne 0) then $
      sxaddpar,header_out,'dedesolrot',dedesolrot,$
      'RR prior de-derotation'

; add deforepix into header when set
    if (deforepix ne 0) then $
      sxaddpar,header_out,'deforepix',deforepix,'RR deforepix applied'
    
; get size header_out
    sizeheader_out=size(header_out) 
    header_outsize=(1+fix(sizeheader_out[1]/36.))*2880
  ;RR fits header = Nx36 "card images" = Nx2880 bytes

  endif ; end case input is also fits 
endif   ; end header setup for fitsoutput

; make header for LP file
if (lpoutput ne 0) then begin
  arr=intarr(nx_out,ny_out)
  header_out=make_lp_header(arr,nt=nt_outfile*nchan)
  header_outsize=512
  endian_out='l' ; intel chips at least in Oslo
endif

; open output file for assoc, write header_out
get_lun, unit_out
openw,unit_out,outfile,swap_endian=(endian ne endian_out)
if (bitpix_out eq -32) then $  
  outassoc=assoc(unit_out,fltarr(nx_out,ny_out),header_outsize)
if (bitpix_out eq 16) then $
  outassoc=assoc(unit_out,intarr(nx_out,ny_out),header_outsize)
if (bitpix_out eq 8) then $
  outassoc=assoc(unit_out,bytarr(nx_out,ny_out),header_outsize)
if (header_outsize ne 0) then begin
  rec=assoc(unit_out, bytarr(header_outsize))
  rec[0]=byte(header_out)
endif

; find mean value of subcube for missingmean or subtractrunmean rescaling
if (missingmean ne 0 or subtractrunmean gt 0) then begin
  if (nchan gt 1) then begin
    print,' ----- reformcubefile: no missingmean or subtractrunmean'+$
      ' for FP files; undone'
    missingmean=0
    subtractmean=0
    missingvalue=-1
  endif  else begin
    sumim=fltarr(xmax-xmin+1,ymax-ymin+1)
    for it=itstart,itend do begin 
      im=despike_rr(inassoc[it],despike,silent=(verbose eq 0))
      sumim=sumim+float(im[xmin:xmax,ymin:ymax])
    endfor
    cubemean=total(sumim)/((float(xmax)-xmin)*(ymax-ymin)*(itend-itstart))
    if (missingmean ne 0) then missingvalue=cubemean ; overrides missingvalue
  endelse
endif

; find cube (minint,maxint) for cubebytscale=1,2 or flt2int
imstart=despike_rr(inassoc[itstart],despike,silent=(verbose eq 0))
imstart=imstart[xmin:xmax,ymin:ymax]
minint=min(imstart)
maxint=max(imstart)
if (cubebytscale eq 2 or flt2int) then begin
  if (cubebytscale eq 2 and nchan gt 1) then begin
    print,' ##### reformcubefile abort: no cubebytscale for FP files'
    return
  endif  
; outer loop over wav for multi-wav files
  for ichan=0,nchan-1 do begin
; inner loop over time steps    
    for it=itstart+1,itend do begin 
      im=despike_rr(inassoc[it*nchan+ichan],despike,silent=(verbose eq 0))
      im=im[xmin:xmax,ymin:ymax]
      mmim=minmax(im)
      if (mmim[0] lt minint) then minint=mmim[0]
      if (mmim[1] gt maxint) then maxint=mmim[1]
    endfor
  endfor
endif 

; find histo_opted (minint,maxint) for cubebytscale=4 or cubehistop
if (cubebytscale eq 4 or cubehistopttop ne 0 or cubehistoptboth ne 0) $
then begin
  if (nchan gt 1) then begin
    print,' ##### reformcubefile abort: no cubebytscale or cubehistopt'+$
      ' for FP files'
    return
  endif  
  imsum=float(despike_rr(inassoc[itstart],despike,silent=(verbose eq 0)))
  imsumt=imsum[xmin:xmax,ymin:ymax]
  for it=itstart+1,itend do begin 
    im=float(despike_rr(inassoc[it],despike,silent=(verbose eq 0)))
    im=im[xmin:xmax,ymin:ymax]
    imsum=imsum+im
  endfor
  immean=imsum/float(itend-itstart+1)
  if (cubehistopttop ne 0) then $
    immean=histo_opt_rr(immean,histopttop,/top_only) 
  if (cubehistoptboth ne 0) then $
    immean=histo_opt_rr(immean,histoptboth)
  if (cubehistopttop eq 0 and cubehistoptboth eq 0) then $
    immean=histo_opt_rr(immean)
  mmmean=minmax(immean)
  minint=mmmean[0]
  maxint=mmmean[1]
endif

;RR set intensity rescale factor (for wordlength) if specified
if (intrescale) then begin
  cubeintscale=32000./maxint
  if (sqrtint) then begin
    cubeintscale=32000./sqrt(maxint)
    minint=cubeintscale*sqrt(minint) 
    maxint=cubeintscale*sqrt(maxint)   ; 32000 for integer cube
  endif
  if (logint) then begin
    cubeintscale=32000./alog10(maxint)
    minint=cubeintscale*alog10(minint+0.1)
    maxint=cubeintscale*alog10(maxint)      ; 32000 for integer cube
  endif
  if (flt2int) then begin  
    absmax=max([abs(minint),abs(maxint)])
    cubeintscale=32000./absmax
  endif
endif

; start outer loop over wavelengths (only 1 if not an FP file)
; ============================================================

for ichan=0,nchan-1 do begin  

; initialize first pair input images
  im_in_0=despike_rr(inassoc[itstart*nchan+ichan],despike,$
                     silent=(verbose eq 0))
  im_in_1=despike_rr(inassoc[(itstart+skip+1)*nchan+ichan],despike,$
                     silent=(verbose eq 0))
  time_in_0=timearr_in[itstart]
  time_in_1=timearr_in[itstart+skip+1]
  im_prior=im_in_0
  im_in_last=despike_rr(inassoc[itend*nchan+ichan],despike,$
                        silent=(verbose eq 0))
  im_post=im_in_last
  
; start big loop over it_out timesteps (frame by frame processing)
; ================================================================

  it_in=itstart
  itselect=0
  for it_out=0,nt_out-1 do begin

; no new timing = no interpolation
    if (outtiming eq 0) then begin
      if (it_in lt nt_in) then $
        image=despike_rr(inassoc[it_in*nchan+ichan],$
                         despike,silent=(verbose eq 0)) $
      else image=despike_rr(inassoc[(it_in-1-skip)*nchan+ichan],$
                            despike,silent=(verbose eq 0))
    endif
    
; new timing: find straddle pair and interpolate
; ----------------------------------------------
    if (outtiming eq 1) then begin

; get a new input image while im_in_1 is before current it_out time
      while (time_in_1 lt timearr_out[it_out] and it_in lt nt_in-1) do begin
        im_in_0=im_in_1
        time_in_0=time_in_1
        it_in=it_in+1
        im_in_1=despike_rr(inassoc[it_in*nchan+ichan],$
                           despike,silent=(verbose eq 0))
        time_in_1=timearr_in[it_in]
      endwhile

; interpolate current straddle pair to desired time
;RR bilinear interpolation; cubic=-0.5 doesn't work for 3D
      p=[[[im_in_0]],[[im_in_1]]]
      time_in_frac=(timearr_out[it_out]-time_in_0)/(time_in_1-time_in_0)
      if (time_in_frac gt 0) then image=interpolate(p,time_in_frac)
      if (time_in_frac le 0) then image=im_in_0

;RR Oct 29 2020 I smeared these but gave strange results in GONG-SDO
; first image as place indicator during prior time extension 
      if (timearr_out[it_out] lt timearr_in[0]) then image=im_prior
; last image as place indicator during post time extension 
      if (timearr_out[it_out] gt timearr_in[nt_in-1]) then image=im_post
      
;;;;; if (ichan eq 7) then STOP

;; ;; ; check = insert the below
;;   if (it_in gt 0) then $
;;   print,ntostr(it_in-1)+'   '+ntostr(it_in)+'   '+ntostr(it_out)+$
;;   '   '+anytim2utc(timearr_in[it_in-1],/ccsds)+$
;;   '   '+anytim2utc(timearr_in[it_in],/ccsds)+$
;;   '   '+anytim2utc(timearr_out[it_out],/ccsds)+$
;;   '   '+ntostr(timearr_out[it_out]-timearr_in[it_in-1],format='(f6.1)')+$
;;   '   '+ntostr(timearr_in[it_in]-timearr_out[it_out],format='(f6.1)')+$
;;   '   '+ntostr(time_in_frac,format='(f6.2)')
;; ;; STOP

    endif ; end of newtiming with temporal straddle interpolation
ENDNEWTIME:

; manipulate this image as requested
; ----------------------------------

; freeze movie if requested 
    if (freezerange[1] ne -1) then begin
      if (it_out eq freezerange[0]) then frozen=image
      if (it_out gt freezerange[0] and it_out le freezerange[1]) then $
        image=frozen
    endif 

; set shift
    shiftit=[-30000,-30000]
    if (n_elements(shift) eq 2) then shiftit=shift 
    if (n_elements(shift) eq 2*nt_in) then begin
      if (outtiming eq 0) then begin
        xshiftit=shift[it_in,0] 
        yshiftit=shift[it_in,1]
      endif else begin
        if (it_in eq 0) then begin
          xshiftit=tpshift[0,0] 
          yshiftit=tpshift[1,0]
        endif else begin
          xpair=[tpshift[0,it_in-1],tpshift[0,it_in]]
          xshiftit=interpolate(xpair,time_in_frac)
          ypair=[tpshift[1,it_in-1],tpshift[1,it_in]]
          yshiftit=interpolate(ypair,time_in_frac)
        endelse
      endelse
      shiftit=[xshiftit,yshiftit]
    endif
    if (outtiming ne 0 and n_elements(shift) eq 2*nt_out) then $
      shiftit=reform(shift[it_out,*])
    if (shiftit[0] eq -30000) then begin
      print,' ##### reformcubefile abort: wrong shift definition'
      STOP
      return
    endif

; set rotate
    rotateit=-30000
    if (n_elements(rotate) eq 1) then rotateit=rotate 
    if (n_elements(rotate) eq nt_in) then begin
      if (outtiming eq 0) then rotateit=rotate[it_in] else begin
        rotpair=[rotate[it_in-1],rotate[it_in]]
        rotateit=interpolate(rotpair,time_in_frac)
      endelse
    endif
    if (outtiming ne 0 and n_elements(rotate) eq nt_out) then $
      rotateit=reform(rotate[it_out])
    if (rotateit eq -30000) then begin
      print,' ##### reformcubefile abort: wrong rotate definition'
      return
    endif

; set (x,y) scaling
    scaleit=[-30000,-30000]
    if (n_elements(scale) eq 2) then scaleit=scale 
    if (n_elements(scale) eq 2*nt_in) then begin
      if (outtiming eq 0) then begin
        xscaleit=scale[it_in,0]
        yscaleit=scale[it_in,1]
      endif else begin
        xpair=[tpscale[0,it_in-1],tpscale[0,it_in]]
        xscaleit=interpolate(xpair,time_in_frac)
        ypair=[tpscale[1,it_in-1],tpscale[1,it_in]]
        yscaleit=interpolate(ypair,time_in_frac)
      endelse
      scaleit=[xscaleit,yscaleit]
    endif
    if (outtiming ne 0 and n_elements(scale) eq 2*nt_out) then $
      scaleit=reform(scale[it_out,*])
    if (scaleit[0] eq -30000) then begin
      print,' ##### reformcubefile abort: wrong scale definition'
      return
    endif

; do intensity rescaling when specified 
    if (intrescale) then image=image*cubeintscale

; smear may be time array
    if (n_elements(smear) eq 1) then thissmear=smear  
    if (n_elements(smear) eq nt_in) then thissmear=smear[it_out]
    if (n_elements(smear) gt 1 and n_elements(smear) ne nt_in) then begin
      print,' ##### reformcubefile has wrong smear array; use first'
      thissmear=smear[0]
    endif
    
; call reformimage with appropriate keywords
    reformimage,image,outimage,$
      xrange=xrange,yrange=yrange,$
      trimbox=trimbox,greyborders=greyborders,greytriangles=greytriangles,$
      xreverse=xreverse,yreverse=yreverse,$
      intrange=intrange,black2grey=black2grey,$
      intreverse=intreverse,absint=absint,$
      rebinfactor=rebinfactor,congridfactor=congridfactor,$
      splinip=splinip,$
      sqrtint=sqrtint,logint=logint,$
      bytscale=0,flt2int=flt2int,$
      nxlarge=nxlarge,nylarge=nylarge,$
      cutcentralx=cutcentralx,cutcentraly=cutcentraly,$
      shift=shiftit,rotate=rotateit,scale=scaleit,missingvalue=missingvalue,$
      despike=despike,sharpen=sharpen,flatten=flatten,$
      histopttop=histopttop,histoptboth=histoptboth,smear=thissmear,$
      muckdark=muckdark,muckbright=muckbright,muckmag=muckmag,muckosc=muckosc

;; ; check:  sv,image & sv,outimage    flickrr,image,outimage
;; print, ' -----= reformcubefile: shift =',shiftit
;; STOP

    image=outimage

; cut at cubehistop values if requested
    if (cubehistopttop ne 0) then image=image<maxint
    if (cubehistoptboth ne 0) then image=image>(minint)<maxint

; cubebytscale if requested
    if (cubebytscale eq 1 or cubebytscale eq 2 or cubebytscale eq 4) then $
      image=bytscl(image,min=minint,max=maxint)
    if (cubebytscale eq 3) then image=bytscl(image)
    
; subtract prior mean if requested
; --------------------------------
    if (subtractrunmean gt 0) then begin

; initially define subcube array
      if (it_out eq 0) then begin
        if (bitpix_out eq -32) then $
          subcube=fltarr(nx_out,ny_out,subtractrunmean)
        if (bitpix_out eq 16) then $
          subcube=intarr(nx_out,ny_out,subtractrunmean) 
        if (bitpix_out eq 8) then $
          subcube=bytarr(nx_out,ny_out,subtractrunmean) 
      endif

; fill first subcube with images
      if (it_out le subtractrunmean-1) then begin
        subcube[*,*,it_out]=image
      endif else begin 

; get the subcube mean and subtract from image 
        if (subtractrunmean gt 1) then $ 
          meansubcube=total(float(subcube),3)/(float(subtractrunmean)) else $
            meansubcube=subcube
        diffimage=image-meansubcube+cubemean
        if (bitpix_out eq 16) then diffimage=fix(diffimage)
        if (bitpix_out eq 8) then diffimage=byte(diffimage)

; shift subcube array and extend with new image
        if (subtractrunmean gt 1) then subcube=shift(subcube,[0,0,-1])
        subcube[*,*,subtractrunmean-1]=image

; replace image by diffimage
        image=diffimage

      endelse

; fix the first images in order not to upset eg ximovie scaling 
      if (it_out eq subtractrunmean) then $
        for itsub=0,subtractrunmean-1 do outassoc[itsub*nchan+ichan]=image

    endif  ; end subtract prior running mean

; add addzero if requested
    if (addzero ne 0) then image=image+addzero

; diffdesolrotate if requested
    if (diffdesolrot eq 1 and it_out gt 0) then image=$
      diffdesolrotate(image,cdelt1ref_out,$
                      xcenarcref_out,ycenarcref_out,$
                      timearr_out[it_out],starttime_out,$
                      dedesolrot=dedesolrot,orientang=-crota2_in)
    
; deforepix if requested
    if (deforepix eq 1) then begin
      image=deforepix(image,xcenarcref_out,ycenarcref_out,$
                      timearr_out[it_out],orientang=-crota2_in)
      szim=size(image)
      nxdefor=szim[1]
      nydefor=szim[2]
   ; cut outer excess borders to equal first image size
      if (nxdefor gt nx_defor or nydefor gt ny_defor) then $
        image=image[0:nx_defor,0:ny_defor] 
    ; add grey outer excess borders to equal first image size
      if (nxdefor lt nx_defor or nydefor lt ny_defor) then begin
        meanim=avg(image)
        if (szim[3] eq 1) then growim=bytarr(nx_defor,ny_defor)+meanim
        if (szim[3] eq 2) then growim=intarr(nx_defor,ny_defor)+meanim
        if (szim[3] eq 4) then growim=fltarr(nx_defor,ny_defory)+meanim
        growim[0,0]=image
        image=growim
      endif  
    endif
    
; AT LONG lAST: assoc-write result (at long last)
    if (treverse eq 0) then outassoc[it_out*nchan+ichan]=image $
    else outassoc[nt_out*nchan+ichan-(it_out*nchan+ichan)-1]=image 

    if (outtiming eq 0) then it_in=it_in+1+skip
    if (it_in gt nt_in-1) then break

; after all this work jump over image if not in tselect selection
    if (tselect[1] ne -1) then begin
      hit=min(abs(tselect-it_out),subscript)
      if (hit eq 0) then begin
        outassoc[itselect]=image
        itselect=itselect+1
      endif
    endif
    
  endfor    ; end big loop over time steps it_out
endfor      ; end outer loop over ichan (only multiple for multi-wav files)

; free the input and output files
free_lun,unit_in,unit_out

; print pacifier
if (verbose eq 1) then print,' ----- reformcubefile wrote '+outfile

; check parameters 
;; STOP

end


; =================== main for testing per IDLWAVE H-c =======================


; test time extension freezing image
cd,'/home/rutten/data/SDO/2014-06-14-small/target/cubes'
infile='aia193.fits'  ; [120,120,85] 
outfile='/tmp/trial.fits'
headin=headfits(infile)
nt_in=fxpar(headin,'naxis3')
start_in=fxpar(headin,'starttim')
cadence_in=fxpar(headin,'cadence')
intai=anytim2tai(start_in)+indgen(nt_in)*cadence_in
outtai=intai+10
for iex=0,10 do outtai=[min(outtai)-10,outtai,max(outtai)+10]
reformcubefile,infile,outfile,timearr_in=intai,timearr_out=outtai,$
  /intreverse
showex,outfile

end

