; file: movex_loadfiles.pro
; init: Jul 21 2017  Rob Rutten  Deil
; last: Mar 24 2022  Rob Rutten  Deil
; note: much too complex by accommodating different file types
;       with annoying habit variations SST file types and naming

;+
pro movex_loadfiles,infiles,$
; ----- outputs (do not specify on input but all are mandatory in a call)
  nxfile,nyfile,ntfile,nfiles,files,nwavs,$
  lunfile,facfile,nzfiles,bitpixfile,hofffile,swapendianfile,$
  mwfile,nwavsfile,$
  iwstartfile,iwendfile,iwlcfile,iwinfile,filewav,labelwav,fixzerowav,$
  xcutmin,xcutmax,nxcut,ycutmin,ycutmax,nycut,xyrangecut,$
  tcutmin,tcutmax,ntcut,$
  duplicate,$
; ----- internal memory keyword for new instances and new calls in session
  instance=instance,$
; ----- keyword inputs = user options
  nt_mw=nt_mw,mwspectfiles=mwspectfiles,$
  addlabels=addlabels,plotscatter=plotscatter,$
  xrange=xrange,yrange=yrange,trange=trange,$
; ----- special keyword input parameters grouped inputs
  allsdo=allsdo,sdodirs=sdodirs,allmwf=allmwf,mwfdirs=mwfdirs,$
  allfits=allfits,fitsdirs=fitsdirs,allspect=allspect,spectdirs=spectdirs

 ; open lun's and get file sampling parameters for movex.pro and company 
 ;   
 ; INPUTS:
 ;   infiles: string array with one or (many) more path+filename strings
 ;     may be  [x,y,t] files or [x,y,t+wav+stokes] multiwav files
 ;     may be empty string ('') for /allsdo, /allmwf, /allfits
 ;
 ; INPUT KEYWORD PARAMETERS:
 ;   nt_mw = number of time steps, MANDATORY when only crispex file(s) 
 ;   mwspectfiles: string array of multiwav files specifying wavelengths
 ;   addlabels = 1/0: distill labels from the file names 
 ;     when multiwav files but no mwspectfiles: use wav indices
 ;     default 1 when allsdo is set, otherwise 0
 ;   xrange, yrange, trange: 2-elem arrays [min,max] to limit to a subcube
 ; ----- special keyword parameters for grouped SDO input
 ;   allsdo = 1/0: take all SDO fits files in dir(s) sdodirs, in nice order
;      if both _ip and _px versions present only _ip gets selected
 ;   sdodirs: string string paths to SDO cubefiles dirs
 ;     defaults for /allsdo: 'sdo2sst/' or 'target/cubes/' or '/')
 ; ----- special keywords for other grouped inputs
 ;   allfits = 1/0: take all 3D fitscube files in dir(s) fitsdirs 
 ;     except SDO files and recognized multiwav files
 ;   fitsdirs: string array paths to dirs(s) with fits files
 ;       default 'sst/' or '.'
 ;   allmwf = 1/0: take all recognized multiwav files in dir(s) mwfdirs
 ;   mwfdirs: string array dir(s) with multiwav files 
 ;       default 'crispex/' or '/'
 ;   allspect = 1/0: use all files in spectdirs as
 ;     mwspectfiles, assuming SST-style "spect*sav*" files
 ;   spectdirs = string array dir(s) mwspectfiles, default 'crispex' or '/'
 ;
 ; OUTPUTS:
 ;   all others in the call    
 ;   
 ; RESTRICTIONS:
 ;  - all image (sequence) files must have the same (nx,ny,nt) dimensions
 ;     (but they may differ in type and in nwav, nstokes for multiwav files)
 ;  - for classic SST/F-P files only (no others) nt_mw must be specified
 ;  - keyword parameters are sticky when rerun from the same IDL session
 ;   
 ; MODIFICATION HISTORY:
 ;   Jul 21 2017 RR: start
 ;   Dec  9 2017 RR: allcrisp, allfits
 ;   Dec 28 2017 RR: DST/IBIS files
 ;   Jan  8 2018 RR: allcrisp > allmwf
 ;   Jan  1 2020 RR: SST/CHROMIS
 ;   Feb 17 2020 RR: allspect
 ;   Nov 16 2020 RR: aia3013, aiafire fitscubes
 ;   Dec 28 2020 RR: fixzero for magnetograms
 ;   Dec 31 2020 RR: aia1617
 ;   Jun 26 2022 RR: SSTRED "5D" fits files
;-

; answer no-parameter query 
if (n_params(0) lt 1) then begin
  sp,movex_loadfiles
  return
endif

; keyword defaults
if (n_elements(nt_mw) eq 0) then ntfile=0 else ntfile=nt_mw ; unsticky
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]
; ----- special SDO and SST
if (n_elements(allsdo) eq 0) then allsdo=0
if (n_elements(sdodirs) eq 0) then begin
  if (file_test('sdo2sst') eq 1) then sdodirs='sdo2sst/' else $
    if (file_test('sdo/target/cubes') eq 1) then $
      sdodirs='sdo/target/cubes/' else $
   if (file_test('target/cubes') eq 1) then $
      sdodirs='target/cubes/' else sdodirs='./'
endif
if (n_elements(allmwf) eq 0) then allmwf=0
if (n_elements(mwfdirs) eq 0) then begin
  if (file_test('crispex') eq 1) then mwfdirs='crispex/' else mwfdirs='./'
endif
if (n_elements(allfits) eq 0) then allfits=0
if (n_elements(fitsdirs) eq 0) then begin
  if (file_test('sst') eq 1) then fitsdirs='sst/' else fitsdirs='./'
endif
if (n_elements(mwspectfiles) eq 0) then mwspectfiles=''
if (n_elements(allspect) eq 0) then allspect=0
if (n_elements(spectdirs) eq 0) then begin
  if (file_test('crispex') eq 1) then spectdirs='crispex/' else spectdirs='./'
endif
; ----- special
if (n_elements(instance) eq 0) then instance=0
if (n_elements(addlabels) eq 0) then $
  if (allsdo eq 1) then addlabels=1 else addlabels=0

; store lun's for subsequent parallel calls in this IDL session
common lunsofmovexfiles,ilun_last
if (n_elements(ilun_last) eq 0) then ilun_last=0

; ----- define inputs 

; define permitted SDO channel strings in hot-temperature display order
allsdowavs=['hmicont','hmimag','hmidop','aia1700','aia1600',$
            'aia1617rat','aia1617mul','aia1617ebs',$
            'aia304','aia3013','aiafire',$
            'aia171','aia193','aia335','aia211','aia131','aia94']
nallsdowavs=n_elements(allsdowavs)
nwavs=1

if (n_elements(infiles) eq 0) then begin
  print,' ##### movex_loadfiles abort: no input file, need at least ""'
  sp,movex_loadfiles
  return
endif
; set files to infiles
files=infiles

; when /allsdo select all SDO files in sdodirs 
if (instance lt 1 and allsdo eq 1) then begin
  sdowavs=strarr(nallsdowavs+10)
  ifile=-1
  nsdodirs=n_elements(sdodirs)
  sdofiles=strarr(1)
  for idir=0,nsdodirs-1 do begin
    for iw=0,nallsdowavs-1 do begin
      infile=findfile(sdodirs[idir]+'/*'+allsdowavs[iw]+'*.fits')
    ; there may be multiple per SDO channel (_ip and _px, or other align set)
      ninfiles=n_elements(infile)
      for iin=0,ninfiles-1 do begin
        fileonly=file_basename(infile[iin])  
        if (strmatch(fileonly,'*aia*') or strmatch(fileonly,'*hmi*')) $
        then begin
          ifile=ifile+1
          sdowavs[ifile]=allsdowavs[iw]
          if (ifile eq 0) then sdofiles[0]=infile[0] else $
            sdofiles=[sdofiles,infile[iin]]
        endif
      endfor
    endfor
  endfor
  nsdofiles=n_elements(sdofiles) ;;; ??? -1 ;; -1 for the empty start
  if (nsdofiles eq 0) then begin
    print,' ##### movex_loadfiles abort: /allsdo but no SDO files in '+sdodirs
    nfiles=-1
    return
  endif
  sdowavs=sdowavs[0:nsdofiles-1]  
  if (nsdofiles gt 0 and files[0] ne '') then files=[sdofiles,files]  
  if (nsdofiles gt 0 and files[0] eq '') then files=sdofiles
  ;RR SDO files now come first
endif 
nfiles=n_elements(files)

; when /allfits select all .fits files in fitsdirs except multiwav and SDO 
if (instance lt 1 and allfits eq 1) then begin
  nfitsdirs=n_elements(fitsdirs)
  fitsfiles=strarr(1)
  for idir=0,nfitsdirs-1 do begin
    dirfitsfiles=findfile(fitsdirs[idir]+'/*.fits')
    ndirfitsfiles=n_elements(dirfitsfiles)
    if (ndirfitsfiles gt 0 and fitsfiles[0] ne '') then $
      fitsfiles=[fitsfiles,dirfitsfiles]  
    if (ndirfitsfiles gt 0 and fitsfiles[0] eq '') then $
      fitsfiles=dirfitsfiles
  endfor
  ; remove all multiwav fits files and also SDO fits files if /allsdo
  removelist=[-1]
  nfitsfiles=n_elements(fitsfiles)
  for ifile=0,nfitsfiles-1 do begin
    fileonly=file_basename(fitsfiles[ifile])
    if (strmatch(fileonly,'*crisp*') or $
        strmatch(fileonly,'*mwseq*')) then removelist=[removelist,ifile]
    if (allsdo ne 0 and (strmatch(fileonly,'*aia*') or $
                         strmatch(fileonly,'*hmi*'))) then removelist=[removelist,ifile]
  endfor 
  if (n_elements(removelist) ne nfitsfiles+1) then begin
    if (n_elements(removelist) ne 1) then $
      remove,removelist[1:n_elements(removelist)-1],fitsfiles
    nfitsfiles=n_elements(fitsfiles)
    if (nfitsfiles gt 0 and files[0] ne '') then files=[files,fitsfiles]  
    if (nfitsfiles gt 0 and files[0] eq '') then files=fitsfiles
    ;RR come after SDO files above
  endif
endif
nfiles=n_elements(files)

; when /allmwf select recognized multiwav and wb files in mwfdirs
if (instance lt 1 and allmwf eq 1) then begin
  nmwfdirs=n_elements(mwfdirs)
  for idir=0,nmwfdirs-1 do begin
    allmwfwavs=['3950','3968','6563','8542','6302','10830'] ; add more?
    nallmwfwavs=n_elements(allmwfwavs)
    mwfwavs=strarr(100)
    ifile=-1
    mwfiles=strarr(1)
; loop over the above line wavelengths
    for iw=0,nallmwfwavs-1 do begin
      
; SST LP files, both wb and narrowband

      ; wb = wide band files
      wbfiles=findfile(mwfdirs[idir]+'/wb.*'+allmwfwavs[iw]+'*cube')
      if (wbfiles[0] ne '') then begin
        nwbfiles=n_elements(wbfiles)
        for iwb=0,nwbfiles-1 do begin
          ifile=ifile+1
          mwfwavs[ifile]=allmwfwavs[iw]
          if (ifile eq 0) then mwfiles[0]=wbfiles[0] else $
            mwfiles=[mwfiles,wbfiles[iwb]]
        endfor
      endif

      ; CRISP files
      crispfiles=findfile(mwfdirs[idir]+'/*crispex*'+allmwfwavs[iw]+'*')
      ; take only the first combining 'crisp' and this fpwav
      fileonly=file_basename(crispfiles[0])
      if (not(strmatch(fileonly,'*.sp.*') or $
              strmatch(fileonly,'*_sp.*')) and $
          crispfiles[0] ne '') then begin
        ifile=ifile+1
        mwfwavs[ifile]=allmwfwavs[iw]
        if (ifile eq 0) then mwfiles[0]=crispfiles[0] else $
          mwfiles=[mwfiles,crispfiles[0]]
      endif

    ; CHROMIS files
      chromisfiles=findfile(mwfdirs[idir]+'/nb_*'+allmwfwavs[iw]+'*')
      ; take only the first combining 'nb_ and this fpwav
      fileonly=file_basename(chromisfiles[0])
      if (not(strmatch(fileonly,'*.sp.*') or $
              strmatch(fileonly,'*_sp.*')) and $
          chromisfiles[0] ne '') then begin
        ifile=ifile+1
        mwfwavs[ifile]=allmwfwavs[iw]
        if (ifile eq 0) then mwfiles[0]=chromisfiles[0] else $
          mwfiles=[mwfiles,chromisfiles[0]]
      endif

; DST/IBIS fits files with "ibis" and "wb" or "mwseq" and wav in name 
; assume there aren't both CRISP and IBIS files = quite unlikely combo
      wbfiles=findfile(mwfdirs[idir]+'/*ibis_wb_*'+allmwfwavs[iw]+'*fits')
      nwbfiles=n_elements(wbfiles)
      if (wbfiles[0] ne '') then begin
        for iwb=0,nwbfiles-1 do begin
          ifile=ifile+1
          mwfwavs[ifile]=allmwfwavs[iw]
          if (ifile eq 0) then mwfiles[0]=wbfiles[0] else $
            mwfiles=[mwfiles,wbfiles[iwb]]
        endfor
      endif 
      mwseqfiles=$
        [findfile(mwfdirs[idir]+'/*ibis_mwseq*'+allmwfwavs[iw]+'*fits')]
      ; take only the first one combining 'mwseq' and this mwfwav
      if (mwseqfiles[0] ne '') then begin
        ifile=ifile+1
        mwfwavs[ifile]=allmwfwavs[iw]
        if (ifile eq 0) then mwfiles[0]=mwseqfiles[0] else $
          mwfiles=[mwfiles,mwseqfiles[0]]
      ; get ntfile from first DST fileheader when not given
        dsthead=headfits(mwseqfiles[0]) 
        nzdst=fxpar(dsthead,'naxis3')
        nwavsdst=fxpar(dsthead,'nrwavs')
        ntdst=fix(nzdst/nwavsdst)
        if (ntfile eq 0) then ntfile=ntdst else $
          if (ntfile ne ntdst) then begin
          print,' ##### movex_loadfiles abort: DST nt ='+trimd(ntdst) +$
            '       differs from ntfile ='+trimd(ntfile)
          return
        endif
      endif
; DST/MXIS files with "mxis" and "wb" or "mwseq" and wav in name 
      wbfiles=findfile(mwfdirs[idir]+'/*mxis_wb_*'+allmwfwavs[iw]+'*fits')
      nwbfiles=n_elements(wbfiles)
      if (wbfiles[0] ne '') then begin
        for iwb=0,nwbfiles-1 do begin
          ifile=ifile+1
          mwfwavs[ifile]=allmwfwavs[iw]
          if (ifile eq 0) then mwfiles[0]=wbfiles[0] else $
            mwfiles=[mwfiles,wbfiles[iwb]]
        endfor
      endif
      mwseqfiles=$
        [findfile(mwfdirs[idir]+'/*mxis_mwseq*'+allmwfwavs[iw]+'*fits')]
      ; take only the first one combining 'mwseq' and this mwfwav
      if (mwseqfiles[0] ne '') then begin
        ifile=ifile+1
        mwfwavs[ifile]=allmwfwavs[iw]
        if (ifile eq 0) then mwfiles[0]=mwseqfiles[0] else $
          mwfiles=[mwfiles,mwseqfiles[0]]
      ; get ntfile from first DST fileheader when not given
        dsthead=headfits(mwseqfiles[0]) 
        nzdst=fxpar(dsthead,'naxis3')
        nwavsdst=fxpar(dsthead,'nrwavs')
        ntdst=fix(nzdst/nwavsdst)
        if (ntfile eq 0) then ntfile=ntdst else $
          if (ntfile ne ntdst) then begin
          print,' ##### movex_loadfiles abort: DST nt ='+trimd(ntdst) +$
            '       differs from ntfile ='+trimd(ntfile)
          return
        endif
      endif
    endfor ; end loop over wavs
    nmwfiles=n_elements(mwfiles)
    if (nmwfiles eq 0) then begin ;RR?? should be 1 for '' empty starter?
      print,' ##### movex_loadfiles abort: /allmwf but no mws in '+mwfdirs
      return
    endif
    mwfwavs=mwfwavs[0:nmwfiles-1]
; update total files list
    if (nmwfiles gt 0 and files[0] ne '') then files=[files,mwfiles]  
    if (nmwfiles gt 0 and files[0] eq '') then files=mwfiles
  endfor ; end loop over mwfdirs
  ;RR come after SDO and fits files above
endif 
nfiles=n_elements(files)

; allspect set: find and add spectfiles
if (instance lt 1 and allspect eq 1) then begin
  nspectdirs=n_elements(spectdirs)
  for idir=0,nspectdirs-1 do begin
    morespectfiles=findfile(spectdirs[idir]+'/*spect*'+'sav*')
    mwspectfiles=[mwspectfiles,morespectfiles]
  endfor
  nspectfiles=n_elements(mwspectfiles)
  if (mwspectfiles[0] eq '') then  mwspectfiles=mwspectfiles[1:nspectfiles-1]
endif

; check files present
if (nfiles eq 0) then begin
  print,' ##### movex_loadfiles abort: no input files'
  return
endif

; duplicate file if only a single one, to enable time-delay blinking
if (instance lt 1) then duplicate=0
if (nfiles eq 1 and instance lt 1) then begin 
  files=[files[0],files[0]]
  nfiles=2
  duplicate=1
endif 

; new instance always has ranges set
if (instance lt 1) then xyrangecut=1

; initalize file-property arrays (name xxxfile means "per file")
lunfile=intarr(nfiles)
facfile=strarr(nfiles)
nzfiles=lonarr(nfiles)  ; must be long for large nt*nwavs in mwfiles
bitpixfile=intarr(nfiles)
hofffile=intarr(nfiles) ; hoff = header offset of a file 
swapendianfile=intarr(nfiles)
mwfile=intarr(nfiles)
nxfile=0
nyfile=0

; open files and set lun, one by one
lunprev=0 ; previous lun nunmer
for ifile=0,nfiles-1 do begin
  thisfile=files[ifile]
  fileonly=file_basename(files[ifile])    
  
; get file extension
  dummy=cgrootname(thisfile,extension=ext)  ;RR needs coyotelib

; get lun = tricky dicky
;  bloody IDL: 100-128 "free" units is too few for nwavs*instance
;  two groups: private = per number 1-99, IDL 100-128 per get_lun, free_lun
;  first fill the IDL group, then the private group, total max = 127 files
;  common variable ilun_last passes to movex.pro
;  check upper limit
  if (ilun_last ge 127) then begin
    print,' ##### movex_loadfiles abort: all LUNs taken, renew IDL session'
    retall
  endif
; first fill the private lun's 1-99
  if (ilun_last ge 0 and ilun_last lt 99) then begin
    ilun_last=ilun_last+1
    lun=ilun_last
  endif
  ; then get IDL "free" lun's 100-128
  if (ilun_last ge 99 and ilun_last le 127) then begin
    get_lun,lun   
; fix the incomprehensible SDO hanging at the same lun in call get_lun
    if (lun le 127 and lun eq ilun_last) then get_lun,lun
    ilun_last=lun
  endif 

;; ; check
;;   print,' ----- ifile='+trim(ifile)+' file='+thisfile+' lun='+trim(lun)

; ===== fits files
  
; thisfile = fits file including path; fileonly is only the file name
  if  (ext eq 'fits') then begin

; fits file with single image only? Replicate into temporary triple
    header=headfits_rr(thisfile)
    ndim=sxpar(header,'NAXIS')
    if (ndim eq 2) then begin
      im=readfits(thisfile)
      im3=[[[im]],[[im]],[[im]]]
      thisfile='/tmp/dummy_3D_'+fileonly
      writefits,thisfile,im3
    endif

; open assoc    
    fits_openassoc,thisfile,lun,nx,ny,nz,bitpix,hoffset,swapendian
    swapendianfile[ifile]=swapendian
    
; recognize SST fits files (wide band or narrow band)
    if (strmatch(thisfile,'*crisp*') or strmatch(thisfile,'*sst*') $
        or strmatch(thisfile,'chromis')) then $
          facfile[ifile]='SST'  ; may be only in path, e.g., ha_lc.fits
    if (strmatch(fileonly,'*crisp*') or strmatch(fileonly,'*chromis*')) $
    then facfile[ifile]='SST' ; in file name
    
; recognize IRIS slitjaw fits files (may override SST above) 
    if (strmatch(fileonly,'*sji*') or strmatch(fileonly,'*IRIS*')) then $
      facfile[ifile]='IRIS'

; recognize DST fits files (wide band or narrow band) 
    if (strmatch(fileonly,'*ibis*') or strmatch(fileonly,'*mxis*' )) $
    then begin
      facfile[ifile]='DST'
      mwfile[ifile]=0 
      if (strmatch(fileonly,'*mwseq*')) then begin
        facfile[ifile]='IBIS/MXIS'
        mwfile[ifile]=1
        head=headfits_rr(thisfile)
        nzdst=fxpar(head,'naxis3')
        nwavs=fxpar(head,'nrwavs')
        ntdst=fix(nzdst/nwavs)
        if (ntfile eq 0) then ntfile=ntdst
      endif
    endif

; recognize SDO fits files
    if (strmatch(fileonly,'*aia*') or strmatch(fileonly,'*hmi*')) then $
      facfile[ifile]='SDO'
  endif ; end of recognizing fits files

; ===== non-fits files
  
; thisfile = DOT file (non-fits, no header, dimensions in file name)
  if  (strmatch(fileonly,'*-cube-*')) then begin
    dot_openassoc,thisfile,lun,nx,ny,nz,bitpix,hoffset,swapendian
    facfile[ifile]='DOT'
    if (addlabels eq 1) then print,' ----- no labels for DOT images'
  endif

; thisfile = file in "La Palma" format (non-fits, too minimal header) 
  if (ext eq 'icube' or ext eq 'fcube' or ext eq 'bcube') then begin
    lp_openassoc,thisfile,lun,nx,ny,nz,bitpix,hoffset,swapendian
    if (str_match(fileonly,'*crisp*') or str_match(thisfile,'*sst*') $
        or str_match(thisfile,'chromis')) then facfile[ifile]='SST'
    if (str_match(fileonly,'*iris*') or str_match(fileonly,'*sji*')) then $ 
      facfile[ifile]='IRIS'
    mwfile[ifile]=0
; check whether crispex file (mulwav = wavelength and/or Stokes resolved)
; is 'nb_' a new habit for new SST pipeline?
    if (str_match(fileonly,'*crisp*') or str_match(fileonly,'nb_')) then begin
      facfile[ifile]='SST'
      mwfile[ifile]=1
; below accepts using nt from non-multiwav file if no nt_mw given
      if (ifile ne 0) then nwavs=min([nz/ntfile,nz/nzfirst]) else begin
        if (ntfile eq 0) then begin
          print,' ##### movex_loadfiles abort: first file = '+thisfile
          print,'       is an SST multiwav file but no nt_mw given'
          return
        endif
        nwavs=nz/ntfile
      endelse
    endif
  endif

; check file equality
  if (ifile eq 0) then begin
    if (n_elements(nx) eq 0) then begin
      print,' ##### movex_loadfiles abort: no nx for file '+thisfile
      print,'       looks like no files were entered or found'
      return
    endif
    nxfirst=nx
    nyfirst=ny
    nzfirst=nz
  endif else begin
    if (ntfile ne 0) then nwavsfirst=nzfirst/ntfile else nwavsfirst=1
    if (nx ne nxfirst or ny ne nyfirst or $
        (nz ne nzfirst and $
         not(nz eq ntfile or nz/nwavs eq nzfirst/nwavsfirst))) $
    then begin
      print,' ##### movex_loadfiles warning: dimensions of file: '
      print,'      '+thisfile
      print,'   with (x,y,t) ='+trimd([nx,ny,nz])+$
        ' incompatible with first file: '
      print,'      '+files[0]
      print,'   with (x,y,t) ='+trimd(fix([nxfirst,nyfirst,nzfirst]))
    endif
  endelse

; now assign
  lunfile[ifile]=lun
  if (n_elements(nx) eq 0) then begin
    print,' ##### movex_loadfiles abort: no nx for file '+thisfile
    return
  endif
  if (nxfile eq 0) then nxfile=nx
  if (nyfile eq 0) then nyfile=ny
  if (nx ne nxfile or ny ne nyfile) then begin
    print, ' ##### movex_loadfiles abort: file '+thisfile+$
      ' differs in nx or ny'
    return
  endif
  nzfiles[ifile]=nz
  bitpixfile[ifile]=bitpix
  hofffile[ifile]=hoffset
  swapendianfile[ifile]=swapendian

endfor ; end loop over ifile

; seperate nz into ntfile and nwavs 
nwavsfile=intarr(nfiles)+1  ; init=1 because e.g., SDO files are single wav
nzmin=min(nzfiles)          ; when non-multiwav files present = ntfile
wavs=-1
if (ntfile eq 0) then ntfile=nzmin ; ntfile when there is a non-mw file
for ifile=0,nfiles-1 do begin
  if (facfile[ifile] eq 'SST' and nzfiles[ifile] le ntfile) then begin
    ;; print,' ##### movex_loadfiles abort: only crisp file(s) but no nt_mw'
    ;; STOP
    ;; return
    print,' ##### movex_loadfiles warning: '+files[ifile]+' SST not CRISP?'
  endif
  if (nzfiles[ifile] eq ntfile) then wavs=[wavs,max(wavs)+1]
  if (nzfiles[ifile] gt ntfile) then begin
    nwavsfile[ifile]=nzfiles[ifile]/ntfile    
    if (not(duplicate and ifile eq 1)) then $
      wavs=[wavs,max(wavs)+indgen(nwavsfile[ifile])+1]
  endif
  if (nzfiles[ifile] lt ntfile) then begin
    print,' ##### movex_loadfiles abort: file '+files[ifile]+$
      ' has nz ='+trimd(nzfiles[ifile])+' but ntfile ='+trimd(ntfile)
    return
  endif 
endfor
nwavs=n_elements(wavs)-1  ; total single-wav files and multiwav files
if (instance eq 0) then begin
  labelwav=strarr(nwavs)    ; labels if requested
  fixzerowav=intarr(nwavs)  ; for setting zero = 128 mag and dopp
endif

; set iw within files (both single wav and multiwav files)
iwstartfile=intarr(nfiles)
iwendfile=intarr(nfiles)
iwlcfile=intarr(nfiles)
iwinfile=intarr(nwavs) 
filewav=intarr(nwavs)
iw0=0
for ifile=0,nfiles-1 do begin
  iwstartfile[ifile]=iw0
  iw1=iw0+nwavsfile[ifile]-1
  iwendfile[ifile]=iw1
  iwinfile[iw0:iw1]=indgen(nwavsfile[ifile])
  iwlcfile[ifile]=fix((iw1-iw0)/2.) ;RR default line center = mid-profile
  filewav[iw0:iw1]=ifile
  if (duplicate eq 0) then iw0=iw0+nwavsfile[ifile] ; no doubling duplicate
endfor

; set cutout frame ranges

; xrange and yrange 
xcutmin=fix(xrange[0])
xcutmax=fix(xrange[1])
ycutmin=fix(yrange[0])
ycutmax=fix(yrange[1])
; beware of bad mouse cursor accidents
if (xcutmin eq 0 and xcutmax eq -1 and ycutmin eq 0 and ycutmax eq -1) $
then xyrangecut=0 else xyrangecut=1 
if (xcutmax eq -1) then xcutmax=nxfile-1
if (ycutmax eq -1) then ycutmax=nyfile-1
nxcut=xcutmax-xcutmin+1  
nycut=ycutmax-ycutmin+1
; check whether min=max, likely in a mouse cursor accident; then back to full
if (nxcut lt 3 or nycut lt 3) then begin
  xcutmin=0
  xcutmax=nxfile-1
  ycutmin=0
  ycutmax=nyfile-1
  nxcut=xcutmax-xcutmin+1  
  nycut=ycutmax-ycutmin+1
endif

; trange
tcutmin=trange[0]
tcutmax=trange[1]
if (tcutmax eq -1) then tcutmax=ntfile-1 
ntcut=tcutmax-tcutmin+1

; set labelwav for SDO and IRIS
if (addlabels ne 0 and instance eq 0) then begin
  for iw=0,nwavs-1 do begin
; SDO
    if (facfile[filewav[iw]] eq 'SDO') then begin
      for isdowav=0,nallsdowavs-1 do $
        if (strmatch(files[filewav[iw]],'*'+allsdowavs[isdowav]+'*')) then $
          labelwav[iw]=allsdowavs[isdowav]
    endif
; IRIS 
    if (facfile[filewav[iw]] eq 'IRIS') then begin
      if strmatch(files[filewav[iw]],'*sji1330*') then $
        labelwav[iw]='IRIS 1330'
      if strmatch(files[filewav[iw]],'*sji1400*') then $
        labelwav[iw]='IRIS 1400'
      if strmatch(files[filewav[iw]],'*sji2796*') then $
        labelwav[iw]='IRIS 2796'
      if strmatch(files[filewav[iw]],'*sji2796*') then $
        labelwav[iw]='IRIS 2832'
    endif
  endfor 
endif

; set labelwav for multiwav files, use wav indices when no mwfwavsfile
if (addlabels ne 0 and instance eq 0) then begin
  print,' ----- patience: defining labels may take long'
  specfile=intarr(nfiles)
  species=''
  for ifile=0,nfiles-1 do begin
    if (facfile[ifile] eq 'SST') then begin
;RR Dec 31 2019 CHROMIS seems to use 'nb_3950' for CaIIH = 3968.5 AA
      if strmatch(files[ifile],'*crispex_6563*') then species='Ha'
      if strmatch(files[ifile],'*crispex_8542*') then species='CaIR'
      if strmatch(files[ifile],'*crispex_6302*') then species='Fe'
      if strmatch(files[ifile],'*crispex_10830*') then species='He'
      if strmatch(files[ifile],'*nb_3950*') then species='CaH'
      if strmatch(files[ifile],'*_ha_*') then species='Ha'
      if strmatch(files[ifile],'*_cah_*') then species='CaH'
      if strmatch(files[ifile],'*_cair*') then species='CaIR'
      if strmatch(files[ifile],'*_fe_*') then species='Fe'
      
;RR May 28 2019: older SST files have . not underscore
      if strmatch(files[ifile],'*crispex.6563*') then species='Ha'
      if strmatch(files[ifile],'*crispex.8542*') then species='CaIR'
      if strmatch(files[ifile],'*crispex.6302*') then species='Fe'
      if strmatch(files[ifile],'*crispex.10830*') then species='He'
      if strmatch(files[ifile],'*crispex.3950*') then species='CaK'
;RR Feb 18 2020 SST wb files
      if strmatch(files[ifile],'*wb.6563*') then species='Ha'
      if strmatch(files[ifile],'*wb.8542*') then species='CaIR'
      if strmatch(files[ifile],'*wb.6302*') then species='Fe'
      if strmatch(files[ifile],'*wb.10830*') then species='He'
      if strmatch(files[ifile],'*wb.3950*') then species='CaK'
    endif
    if (facfile[ifile] eq 'DST' or facfile[ifile] eq 'IBIS/MXIS') then begin
      if strmatch(files[ifile],'*6563*') then species='Ha'
      if strmatch(files[ifile],'*8542*') then species='CaIR'
      if strmatch(files[ifile],'*6302*') then species='Fe'
      if strmatch(files[ifile],'*10830*') then species='He'
    endif
    if (mwfile[ifile] eq 1) then begin
      nspecsavs=n_elements(mwspectfiles)
      if (mwspectfiles[0] eq '') then nspecsavs=0 
      if (nspecsavs eq 0 and facfile[ifile] eq 'IBIS/MXIS') then $
        print,' ===== movex_loadfiles warning: IBIS/MXIS data '+$
        'without fpwavfile implies unreliable Dopplergrams'
      for ispec=0,nspecsavs-1 do begin
        if ((strmatch(mwspectfiles[ispec],'*6563*') and species eq 'Ha') or $
            (strmatch(mwspectfiles[ispec],'*8542*') and species eq 'CaIR') or $
            (strmatch(mwspectfiles[ispec],'*3968*') and species eq 'CaH') or $
            (strmatch(mwspectfiles[ispec],'*6302*') and species eq 'Fe') or $
            (strmatch(mwspectfiles[ispec],'*10830*') and species eq 'He')) $
        then begin
          specfile[ifile]=1
; SST
          if (facfile[ifile] eq 'SST') then begin
            restore,mwspectfiles[ispec]
            mwfwavs=spect_pos
;; SST reduc puts mean-profile line center ad-hoc at some "name" wavelength
            if (species eq 'Ha') then $
              mwlc=where(abs(mwfwavs-6563.0) lt 0.005)
            if (species eq 'CaIR') then $
              mwlc=where(abs(mwfwavs-8542.0) lt 0.005)
            if (species eq 'Fe') then $
              mwlc=where(abs(mwfwavs-6302.0) lt 0.005)
            if (species eq 'CaK') then $
              mwlc=where(abs(mwfwavs-3933.7) lt 0.005)
            if (species eq 'CaH') then $
              mwlc=where(abs(mwfwavs-3968.5) lt 0.005)
          endif
; IBIS/MXIS
          if (facfile[ifile] eq 'IBIS/MXIS') then begin
            readcol,mwspectfiles[ispec],mwfwavs 
; IBIS uses FTS-atlas disk-center line-center wavs; these are from MMH
            if (species eq 'Ha') then $
              mwlc=where(abs(mwfwavs-6562.8) eq min(abs(mwfwavs-6562.8)))
            if (species eq 'CaIR') then $
              mwlc=where(abs(mwfwavs-8542.1) eq min(abs(mwfwavs-8542.1)))
            if (species eq 'Fe') then $
              mwlc=where(abs(mwfwavs-6301.5) eq min(abs(mwfwavs-6301.5)))
          endif
          
          mwlc=fix(mwlc[0])     ; f--k IDL
          if (mwlc ne -1) then iwlcfile[ifile]=mwlc else $
            print,' ##### no line-center sampling found' 
          mwdelwavs=fix((mwfwavs-mwfwavs[mwlc])*100.)/100. ; 2 digits only  
          nmwdelwavs=n_elements(mwdelwavs)
          mwlabels=strarr(nmwdelwavs)
          for iw=0,nmwdelwavs-1 do $
; split + and - signs, add +
            if (mwdelwavs[iw] ge 0) then $
              mwlabels[iw]=species+'+'+trim(mwdelwavs[iw],"(F5.2)") else $
                mwlabels[iw]=species+trim(mwdelwavs[iw],"(F5.2)")
          labelwav[iwstartfile[ifile]:iwendfile[ifile]]=mwlabels[*]
        endif
      endfor
      
; use wav indices when no mwfwavsfile supplied
      if (specfile[ifile] eq 0) then begin
        nwavind=nwavsfile[ifile]
        mwlabels=species+'-w'+trim(indgen(nwavind))
; case of 4 Stokes parameters
        if (nwavind eq 4) and strmatch(files[ifile],'*stokes*') then begin
          Stokesparams=['I','Q','U','V']
          mwlabels=species+'-Stokes-'+Stokesparams
        endif
; add into labelwav
        labelwav[iwstartfile[ifile]:iwendfile[ifile]]=mwlabels[*]
      endif
    endif ; end files with multiwav file present

; add label for wide-band prefilter La Palma file 
    if (facfile[ifile] eq 'SST' and strmatch(files[ifile],'*wb.*')) then $
      labelwav[iwstartfile[ifile]]=species+' wide'

; add label for continuum-wav La Palma file as cont_cah... 
    if (facfile[ifile] eq 'SST' and strmatch(files[ifile],'*cont_*')) then $
      labelwav[iwstartfile[ifile]]=species+' cont'

; add label for wide-band IBIS/MXIS prefilter fitscube file
    if (facfile[ifile] eq 'DST' and strmatch(files[ifile],'*_wb_*')) then $
      labelwav[iwstartfile[ifile]]=species+' wide'

; add label for iris_SJI file
    if (strmatch(files[ifile],'*iris_SJI*')) then begin
      if (strmatch(files[ifile],'*1330*')) then $
        labelwav[iwstartfile[ifile]]='IRIS 1330'
      if (strmatch(files[ifile],'*1400*')) then $
        labelwav[iwstartfile[ifile]]='IRIS 1400'
      if (strmatch(files[ifile],'*2796*')) then $
        labelwav[iwstartfile[ifile]]='IRIS 2796'
      if (strmatch(files[ifile],'*2832*')) then $
        labelwav[iwstartfile[ifile]]='IRIS 2832'
    endif

;RR Feb 18 2020 out for demo solarnet
;; ; use wav indices also for chord index in "swp" files 
;;     if (facfile[ifile] eq 'SST' or facfile[ifile] eq 'DST') then begin
;;       swp=''
;;       if strmatch(files[ifile],'*shift*') then swp=' shift'
;;       if strmatch(files[ifile],'*width*') then swp=' width'
;;       if strmatch(files[ifile],'*shxwi*') then swp=' shxwi'
;;       if (swp ne '') then begin
;;         nwavind=nwavsfile[ifile]
;;         mwlabels=species+swp+'-'+trim(indgen(nwavind))
;;         labelwav[iwstartfile[ifile]:iwendfile[ifile]]=mwlabels[*]
;;       endif
;;     endif

  endfor                    ; end ifile loop over files

; add filename as labelwav for all remaining empty labelwavs
  for iwav=0,nwavs-1 do if (labelwav[iwav] eq '') then $
    labelwav[iwav]=file_basename(files[filewav[iwav]],'.fits')

  
endif                       ; end if addlabels ne 0
labelwav=' '+labelwav+' '   ; move them off the cursor line in profile plot

; set fixzerowav for Dopplergrams or magnetograms
for iwav=0,nwavs-1 do $
  if (strmatch(labelwav[iwav],'*dopp*') $
      or strmatch(labelwav[iwav],'*Dopp*') $
      or strmatch(files[filewav[iwav]],'*mag*') $
      or strmatch(labelwav[iwav],'*mag*') $
      or strmatch(labelwav[iwav],'*tokes-Q*') $
      or strmatch(labelwav[iwav],'*tokes-U*') $
      or strmatch(labelwav[iwav],'*tokes-V*')) then fixzerowav[iwav]=1

; printouts
if (instance eq 0 and ntfile gt 3) then begin
  print,' ----- movex_loadfiles dimensions: nx='+trim(nxfile)+$
    ' ny='+trim(nyfile)+' nt='+trim(ntfile)
  if (addlabels ne 0) then begin
    print,' ----- movex_loadfiles labelwav:'
    print,labelwav
  endif
endif

end ; ----- of movex_loadfiles.pro



;RR ================ main for testing per IDLWAVE H-c ==================

;; ; test with contrail data on internal disk
;; ;; cd,'/home/rutten/data/SST/2014-06-21-quiet'
;; cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet'
;; ;; files=['iris2sst/sji1400_algbo.fits','iris2sst/sji2796_algbo.fits',$
;; ;;        'crispex/crispex.6563.08:02:31.time_corrected.aligned.icube']
;; ;; mwspectfiles='crispex/spectfile.6563.idlsave'
;; files=''
;; allsdo=1
;; ;; addlabels=1
;; ;; xrange=[250,750]
;; ;; yrange=[250,750]
;; ;; trange=[60,150]

;; cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet'
;; files='sst/shxwi_frac_6563.icube'
;; nt_mw=377
;; nt_mw=3
;; addlabels=1

;; cd,'/home/rutten/data/SST/2016-09-05-demo'
;; files=''
;; mwspectfiles=['crispex/spectfile.6563.idlsave',$
;;                'crispex/spectfile.8542.idlsave']
;; nt_mw=35
;; allsdo=0
;; allmwf=1
;; allfits=0
;; addlabels=1

;; cd,'/media/rutten/SSTDATA/alldata/SST/2014-08-29-gregal'
;; files='crispex/crispex.6563.15:40:09.time_corrected.aligned.icube'
;; nt_mw=735

;; cd,'/home/rutten/data/SST/2016-09-05-demo'
;; files='crispex/crispex.6563.09:48:31.time_corrected.aligned.icube'
;; nt_mw=35

;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad'
;; files=['dst2sdo/hmicont_alrev_px.fits',$
;;         'dst2sdo/ibis_spectrims_6563_20170614_151541.alrev.fits']
;; ;; files=['ibis_wb_6563_20170614_164052.fits',$
;; ;;        'ibis_spectrims_6563_20170614_164052.fits']
;; ;; 
;; mwspectfiles='ibis/ibis_wavs_6563_20170614_164052.txt'
;; addlabels=1

;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/'
;; allsdo=1
;; allfits=1
;; allmwf=1
;; ;; sdodirs='ibisA2sdo' ;RR selectby line swapping or commenting 
;; sdodirs='ibisB2sdo'
;; ;; fitsdirs='ibisA2sdo'
;; fitsdirs='ibisB2sdo'
;; ;; mwfdirs='ibisA2sdo'
;; mwfdirs='ibisB2sdo'
;; addlabels=1
;; files=''  
;; ;; mwspectfiles=['ibisA/ibis_wavs_6563_20170614_151541.txt',$
;; ;;              'ibisA/ibis_wavs_8542_20170614_151541.txt']
;; mwspectfiles=['ibisB/ibis_wavs_6563_20170614_164052.txt',$
;;              'ibisB/ibis_wavs_8542_20170614_164052.txt']

;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/'
;; allsdo=0
;; allfits=0
;; allmwf=1
;; mwfdirs='mxis'
;; addlabels=1
;; files=''  
;; ;;; mwspectfiles=[mxis_wavs_10830_20170614_104106.txt]

;; cd,'/media/rutten/SSTDATA/alldata/SST/2014-08-29-gregal'
;; allsdo=1
;; sdodirs='sdo2sst'
;; allmwf=1
;; mwfdirs='crispex'
;; allfits=1
;; fitsdirs=['sst','crispex']
;; addlabels=1
;; files=''

;; cd,'/home/rutten/data/SST/2016-09-05-demo/'
;; allsdo=1
;; ;;; sdodirs='sdo2sst/'
;; allfits=1
;; fitsdirs=['sdo2sst','sst']
;; allmwf=1
;; ;;; mwfdirs='crispex'
;; addlabels=1
;; mwspectfiles=['crispex/spectfile.6563.idlsave',$
;;              'crispex/spectfile.8542.idlsave']
;; files=''

;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/ibis_mxis_2sdo'
;; allsdo=1
;; sdodirs='.'
;; allmwf=1
;; mwfdirs='.'
;; trimbox=[200,200,700,700]
;; addlabels=1
;; infiles=['ibis_shift_kevin_6563_0.9.fits',$
;;          'ibis_width_kevin_6563_0.9.fits',$
;;          'ibis_width_kevin_8542_0.6.fits',$
;;          'ibis_shxwi_kevin_6563_0.9.fits']

;; cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet'
;; infiles=['sst/widwid_ha_ca.fits',$ 
;;          'iris2sst/sji1400_algbo.fits',$
;;          'iris2sst/sji2796_algbo.fits',$
;;          'crispex/crispex.6563.08:02:31.time_corrected.aligned.icube',$
;;          'crispex/crispex.8542.08:02:31.time_corrected.icube']
;; mwspectfiles=['crispex/spectfile.6563.idlsave',$
;;               'crispex/spectfile.8542.idlsave']
;; nt_mw=377
;; addlabels=1
;; allsdo=1

;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/ibis_mxis_iris_sdo'
;; trimbox=[200,200,700,700]
;; infiles=''
;; allsdo=1
;; sdodirs='./'
;; allmwf=1
;; mwfdirs='./'
;; allfits=1
;; fitsdirs='./'
;; mwspectfiles=['ibis_wavs_6563_20170614_164052.txt',$
;;               'ibis_wavs_8542_20170614_164052.txt',$
;;               'mxis_wavs_10830_20170614_104106.txt']
;; addlabels=1


;; cd,'/media/rutten/SSTDATA/alldata/SST/2017-05-25-souvik/combined'
;; showex,$
;;   'crispex_3950_2017-05-25T09:12:00_scans=0-424_time-corrected_rotated2iris.fcube',$
;;   'crispex_6563_08:05:00_aligned_3950_2017-05-25T09:12:00_scans=0-424_rotated2iris.icube',$
;;   'IRIS_Mg_SJI_2796_07:33:00_intblown_aligned_3950_2017-05-25T09:12:00_rotated.fcube',$
;;   'IRIS_Si_SJI_1400_07:33:00_intblown_aligned_3950_2017-05-25T09:12:00_rotated.fcube',$
;;   mwspectfiles=['spectfile.3950.idlsave','spectfile.6563.idlsave'],$
;;   nt_mw=425,/plotscatter,/addlabels


;; ; rbecontrails DataA
;; cd,'/home/rutten/data/SST/2008-06-15-RBEs'
;; infiles=['sst/halpha_15Jun2008_strt.pfc.fcube',$  ; first for wav index
;;   'sst/wb_15Jun2008_strt.bcube','sst/ha_dopp_800.fits']
;; nt_mw=212
;; mwspectfiles='sst/spectfile.6563.idlsave'
;; addlabels=1
;; plotscatter=1

;; ; Vasco CRISP+CHROMIS
;; cd,'/media/rutten/SSTNEW/alldata/SST/2018-06-10-vasco'
;; ;; infiles=['sst2rotsdo/hmicont_rot_ip_comb.fits',$
;; ;;  'sst2rotsdo/nb_3950_2018-06-10T12:21:19_scans=0-286_corrected_int_im_rot_ip_comb.icube']
;; infiles=''
;; allsdo=1
;; sdodirs='sst2rotsdo'
;; allmwf=1
;; mwfdirs='sst2rotsdo'
;; mwspectfiles=['chromis/spectfile.3968.idlsave',$
;;                 'crisp/spectfile.6302.idlsave',$
;;                 'crisp/spectfile.6563.idlsave',$
;;               'crisp/spectfile.8542.idlsave']
;; addlabels=1

; ===== standard test
cd,'/home/rutten/data/SST/2016-09-05-demo/'
infiles=''
allmwf=1
mwfdirs='sst2rotsdo'
allspect=1
addlabels=1
allsdo=1
sdodirs='sst2rotsdo'
fitsdirs='sst2rotsdo'
mwspectfiles='spectfiles'

movex_loadfiles,infiles,$
; ----- outputs
  nxfile,nyfile,ntfile,nfiles,files,nwavs,$
  lunfile,facfile,nzfiles,bitpixfile,hofffile,swapendianfile,$
  mwfile,nwavsfile,$
  iwstartfile,iwendfile,iwlcfile,iwinfile,filewav,labelwav,fixzerowav,$
  xcutmin,xcutmax,nxcut,ycutmin,ycutmax,nycut,xyrangecut,$
  tcutmin,tcutmax,ntcut,duplicate,instance=instance,$
; ----- keyword inputs
  nt_mw=nt_mw,mwspectfiles=mwspectfiles,addlabels=addlabels,$
  xrange=xrange,yrange=yrange,trange=trange,$
; ----- special keyword parameters for SDO and SST data
  allsdo=allsdo,sdodirs=sdodirs,$
  allmwf=allmwf,mwfdirs=mwfdirs,$
  allfits=allfits,fitsdirs=fitsdirs,$
  allspect=allspect,spectdirs=spectdirs
  
end
