; file: minmaxcubefile.pro
; init: Jun  6 2017  Rob Rutten  Deil from former minmaxfitscube.pro
; last: May 16 2019  Rob Rutten  Deil

;+

function minmaxcubefile,infile,xrange=xrange,yrange=yrange,trange=trange,$
  trimbox=trimbox,histopt=histopt 

 ;   get min and max of cubefile, or a segment of it, using assoc
 ;
 ; INPUTS:
 ;   infile: string with path/filename
 ; 
 ; OPTIONAL INPUTS:
 ;   xrange, yrange, trange: partial cutout specification (2-element arrays)
 ;   trimbox: usual
 ;   histopt=0 or cutoff value: histo_opt each image first
 ; 
 ; OUTPUTS:
 ;   2-element array with min and max value
 ; 
 ; HISTORY:
 ;   Mar  5 2015 RR: start 
 ;   Jun  6 2017 RR: infile may be fits file or La Palma file
 ;   Jun 24 2022 RR: also SSTRED "5D" fits file
;-

; answer no-parameter query
if (n_params() lt 1) then begin
  sp,minmaxcubefile
  return,-1
endif

; keyword defaults
if (not keyword_set(xrange)) then xrange=[0,-1]
if (not keyword_set(yrange)) then yrange=[0,-1]
if (not keyword_set(trange)) then trange=[0,-1]
if (not keyword_set(trimbox)) then trimbox=-1
if (not keyword_set(histopt)) then histopt=0

; set endian
bigendian=1

; get filetype
dummy=cgrootname(infile,extension=ext) ;RR needs coyotelib

; infile is a FITS file
if (ext eq 'fits') then begin
; get cube geometry and file datatype from the fits header
  header_in=headfits_rr(infile)
  header_insize=(1+fix(n_elements(header_in)/36.))*2880
  bitpix_in=fxpar(header_in,'bitpix')
  nx_in=fxpar(header_in,'naxis1') 
  ny_in=fxpar(header_in,'naxis2')
  nt_in=fxpar(header_in,'naxis3')

; accommodate silly SSTRED 5-dim headers for their 3D files
  ndims=fxpar(header_in,'nxaxis')
  if (ndims eq 5) then begin
    nwavs=fxpar(header_in,'naxis3')
    nstokes=fxpar(header_in,'naxis4')
    ntimesteps=fxpar(header_in,'naxis5')
    nt_in=nwavs*nstokes*ntimesteps
  endif
  
; check
  if (nx_in eq 0 or ny_in eq 0 or nt_in eq 0) then begin
    print,' ###### no proper cubefile since no nx, ny, nt in header '+infile
    return,-1
  endif
endif

; infile is a La Palma file
if (ext eq 'icube' or ext eq 'fcube' or ext eq 'bcube') then begin
; get cube geometry and file datatype from the LP header
  lp_readheader,infile,header=header,datatype=datatype_in,$
    nx=nx_in,ny=ny_in,nt=nt_in,$
    endian=endian_file
; check
  if (nx_in eq 0 or ny_in eq 0 or nt_in eq 0) then begin
    print,$
      ' ##### minmaxcubefile abort: no nx, ny, nt in LaPalma header '+infile
    return,-1
  endif
; set bitpix
  if (datatype_in eq 1) then bitpix_in=8
  if (datatype_in eq 2) then bitpix_in=16
  if (datatype_in eq 4) then bitpix_in=-32
; set header size
  header_insize=512

; reset bigendian
  bigendian=0
endif

; translate trimbox to xrange and yrange
if (trimbox[0] ne -1) then begin
  xrange[0]=trimbox[0]
  xrange[1]=trimbox[2]
  yrange[0]=trimbox[1]
  yrange[1]=trimbox[3]
endif

; xrange and yrange 
xmin=xrange[0]
xmax=xrange[1]
ymin=yrange[0]
ymax=yrange[1]
if (xmax eq -1) then xmax=nx_in-1
if (ymax eq -1) then ymax=ny_in-1

; trange
itstart=trange[0]
itend=trange[1]
if (itend eq -1) then itend=nt_in-1 

; open input file for assoc
get_lun, unit_in
if (bigendian) then openr,unit_in,infile,/swap_if_little_endian $
else openr,unit_in,infile
if (bitpix_in eq -32) then $
  inassoc=assoc(unit_in,fltarr(nx_in,ny_in),header_insize)
if (bitpix_in eq 16) then $
  inassoc=assoc(unit_in,intarr(nx_in,ny_in),header_insize)
if (bitpix_in eq 8) then $
  inassoc=assoc(unit_in,bytarr(nx_in,ny_in),header_insize)

; check input image size
im0=inassoc[0]
sizeim0=size(im0)
nx_im0=sizeim0[1]
ny_im0=sizeim0[2]
if (nx_im0 ne nx_in or ny_im0 ne ny_in) then begin
  print, ' ##### nx or ny in header inequal to first image [nx,ny]' 
  return,-1
endif

; find [minint,maxint]
imstart=inassoc[itstart]
if (histopt ne 0) then imstart=histo_opt_rr(imstart,histopt)
imstart=imstart[xmin:xmax,ymin:ymax]
minint=min(imstart)
maxint=max(imstart)
for it=itstart,itend do begin 
  im=inassoc[it]
  if (histopt ne 0) then im=histo_opt_rr(im,histopt)
  im=im[xmin:xmax,ymin:ymax]
  mmim=minmax(im)
  if (mmim[0] lt minint) then minint=mmim[0]
  if (mmim[1] gt maxint) then maxint=mmim[1]
endfor  

; free the input file
free_lun,unit_in

return,[minint,maxint]
end

; =================== main for testing per IDLWAVE H-c =======================


cd,'/home/rutten/data/SST/2019-08-03_kuridze'
infile='nb_4896_rot_ip.fits' ; nw_in=25 ns_in=1  nt_in=61

print,' ---- minmax = ',minmaxcubefile(infile,$
               xrange=[100,200],trange=[0,3],histopt=1)
end

