; file: maketilefigs.pro.pro
; init: Jun 25 2020  Rob Rutten  Deil = from imagelib/findtiledimshift.pro
; last: Dec 12 2020  Rob Rutten  Deil = former sdo_tilefigs.pro

;+
pro maketilefigs,outdir=outdir,datetime=datetime,wavpair=wavpair,$
  arcsecpx=arcsecpx,shiftchart=shiftchart,arrowsize=arrowsize,$
  plotzonal=plotzonal,showps=showps,verbose=verbose

 ; make tile figs
 ;   
 ; INPUTS:
 ;   requires output file outdir/tileshifts_datetime__wav1_wav2.dat from
 ;   findimshift_tiled.pro
 ;
 ; KEYWORD INPUT PARAMETERS:
 ;   outdir: default /tmp but may overflow
 ;   datetime: only for filenames
 ;   wavpair: 2-elem string array level2 SDO channel names only for filenames
 ;   arcsecpx: arcsec per px, default 0.6 (SDO) 
 ;   shiftchart = 1/0: plot tile vector-shift chart
 ;   arrowsize: magnifier to chart arrow length (default = adapt)
 ;   plotzonal = 1/0: plot zonally-averaged shift sizes + radial (default 0)
 ;   showps = integer/0: open output ps files on screen with mag=opengv
 ;   verbose = 1/0: more output   
 ;   
 ; HISTORY:
 ;   Feb 21 2017 RR: start writing findtiledimshift.pro
 ;   Jan 10 2018 RR: no border/triangle cropping for the tiles
 ;   Jul 31 2018 RR: better vector charts, zonal fit & apply correction
 ;   Jun 16 2020 RR: improved charts
 ;   Jun 25 2020 RR: taken plot part out of findtiledimshift.pro
 ;   Jun 28 2020 RR: > maketilefigs.pro
 ;   Jul 30 2020 RR: applyzonal out to avoid inconsistencies, rerun instead
;-

; answer no-parameter query 
if (n_params(0) gt 0) then begin
  sp,maketilefigs
  return 
endif

; defaults for keywords
if (n_elements(outdir) eq 0) then outdir='/tmp' 
if (n_elements(datetime) eq 0) then datetime='2000.06.01_00:00' ; small r_sun
if (n_elements(wavpair) eq 0) then wavpair=['A','B']
if (n_elements(arcsecpx) eq 0) then arcsecpx=0.6  ;  SDO
if (n_elements(shiftchart) eq 0) then shiftchart=0
if (n_elements(arrowsize) eq 0) then arrowsize=0
if (n_elements(plotzonal) eq 0) then plotzonal=0
if (n_elements(showps) eq 0) then showps=0
if (n_elements(verbose) eq 0) then verbose=0

; set some values
xcen=0.              ; must be the case
ycen=0. 
datetimeut=anytim2utc(datetime,/ccsds)
rsun_obs=get_rb0p(datetimeut,/radius,/quiet)

; read start of tilefile
tilefile=outdir+'/tileshifts_'+datetime+'_'+wavpair[0]+'_'+wavpair[1]+'.dat'
if (file_test(tilefile)) then begin
  openr,42,tilefile
  readf,42,nx,ny,nxtile,nytile,radaslimit,heightdiff
  nx=fix(nx)  ; f**k IDL
  ny=fix(ny)
  nxtile=fix(nxtile)
  nytile=fix(nytile)
  
; define arrays (as in findimshift_tiled.pro)
  ntilesx=fix(nx/nxtile)   
  ntilesy=ntilesx
  ntiles=ntilesx*ntilesy
  xcentiles=fltarr(ntilesx,ntilesy)
  ycentiles=fltarr(ntilesx,ntilesy)
  solvectiles=fltarr(ntilesx,ntilesy)
  xtshifts=fltarr(ntilesx,ntilesy)
  ytshifts=fltarr(ntilesx,ntilesy)

; read rest of tilefile now that xtshifts and ytshifts arrays are defined
  readf,42,xcentiles
  readf,42,ycentiles
  readf,42,solvectiles
  readf,42,xtshifts
  readf,42,ytshifts
  close,42
endif else begin
  print,' ##### maketilefigs.pro abort: no tilefile = '+tilefile
  return
endelse

; check tilefile data sizes
szx=size(xtshifts)
szy=size(ytshifts)
if (szx[1] ne ntilesx or szx[2] ne ntilesy or $
    szy[1] ne ntilesx or szy[2] ne ntilesy) then begin 
  print,' ##### maketilefigs.pro abort: tilefile clash ntilesx or ntilesy'
  STOP
endif

; get arrow lengths
discard=min([[xtshifts],[ytshifts]])
if (discard gt -400) then discard=-500 ; no discarded tiles
veclenarr=fltarr(ntilesx,ntilesy)
veclenarr=sqrt(float(xtshifts)^2+float(ytshifts)^2) 
indexpassed=where(veclenarr lt abs(discard)-1) ; are sqrt(2)*abs(discard)

; adapt arrow length multiplier if arrowsize not specified
if (arrowsize eq 0 and shiftchart ne 0) then begin
  meanlen=avg(veclenarr[indexpassed])
  if (meanlen gt 1) then arrowmult=1./(fix(meanlen)+1) $
  else arrowmult=fix(1./meanlen)  
endif else arrowmult=arrowsize
print,' ----- shiftchart arrow units '+trimd(1./arrowmult)+' px'

; make discard tiles white in veclenarr greyscale image 
imveclen=bytscl(veclenarr<(1.1*max(veclenarr[indexpassed])))

;; ; check
;; ;; imveclen[xcentiles[nxtile/2.+delcenx/arcsecpx],*]=255
;; ;; imveclen[*,ycentiles[nytile/2.+delceny/arcsecpx]]=255
;; imveclen[ntilesx/2+delcenx/nxtile,*]=255
;; imveclen[*,ntilesy/2+delceny/nytile]=255
;; sv,imveclen
;; STOP

; check plots on screen
if (verbose) then begin
  print,' mean veclenarr ='+trimd(mean(veclenarr[indexpassed]))
  cghistoplot,veclenarr[indexpassed],nbins=200,/window,$
    xtitle=' vector length  [px]',title='all tile vector lengths'
  scatcont,xtshifts[indexpassed],ytshifts[indexpassed]
endif

; optional tiles + shift vectors chart
if (shiftchart ne 0) then begin
;RR how to unroll this double loop?
  xv1=fltarr(ntilesx,ntilesy)
  yv1=fltarr(ntilesx,ntilesy)
  xv2=fltarr(ntilesx,ntilesy)
  yv2=fltarr(ntilesx,ntilesy)
  for ixt=0,ntilesx-1 do begin
    for iyt=0,ntilesy-1 do begin
 ;RR arrows from tile centers; excess shifts from final mean
      xv1[ixt,iyt]=ixt+0.5
      yv1[ixt,iyt]=iyt+0.5
      ; zero arrows for misfit outliers
      if (xtshifts[ixt,iyt] le discard+2 or ytshifts[ixt,iyt] le discard+2) $
      then begin ; no arrow length for discard tiles
        xv2[ixt,iyt]=ixt+0.5
        yv2[ixt,iyt]=iyt+0.5
      endif else begin
        xv2[ixt,iyt]=ixt+0.5+arrowmult*(xtshifts[ixt,iyt]) 
        yv2[ixt,iyt]=iyt+0.5+arrowmult*(ytshifts[ixt,iyt]) 
      endelse
    endfor
  endfor
  psfilename=$
    outdir+'/shiftvectors_'+datetime+'_'+wavpair[0]+'_'+wavpair[1]+'.ps'
  plotarea=[0.12,0.12,0.9,0.9] ; must be square
  openpsplot,psfilename,thick=2,fontsize=6,xsize=8.8,ysize=8.8
  xaxtile=indgen(ntilesx)*float(ntilesx)/(ntilesx-1)
  yaxtile=indgen(ntilesy)*float(ntilesy)/(ntilesy-1)
  xtitle='horizontal tile number'
  ytitle='vertical tile number'
  tv,imveclen,plotarea[0],plotarea[1],$
     xsize=plotarea[2]-plotarea[0],ysize=plotarea[3]-plotarea[1],/normal
  contour,imveclen,xaxtile,yaxtile,$
    /nodata,/noerase,/xstyle,/ystyle,position=plotarea,$
    xticklen=-0.03,yticklen=-0.03,xtitle=xtitle, ytitle=ytitle
  arrow,xv1,yv1,xv2,yv2,/data,hsize=-0.4,thick=0.2,color=255
  xyouts,0.12,0.95,/norm,charsize=1.2,$
    datetime+'    '+wavpair[0]+'>'+wavpair[1]+$
    '    hdiff'+trimd(fix(heightdiff+0.5))+$
    '    arrows ['+trimd(1./arrowmult,2,/nospace)+' px]'

; add solar limb circle with center (X,Y)=(0,0)
  draw_circle,ntilesx/2.,ntilesy/2.,$
    rsun_obs/arcsecpx/nxtile,color=128,thick=2

; done
  closepsplot,psfilename,opengv=showps
endif

; ===== optional plot zonal averages to gauge radial component 
if (plotzonal ne 0) then begin
  radpxlimit=radaslimit/arcsecpx
; plot zonal averages veclengths and radial components
  nzones=fix(ntilesx/10.)>3
  averzones=fltarr(nzones)
  radzones=fltarr(nzones)
  zonewidth=radpxlimit/nzones
  for izone=0,nzones-1 do begin
    sumzone=0.
    sumrad=0.
    ninzone=0
;RR how to unroll this double loop?
    for ixt=0,ntilesx-1 do begin
      for iyt=0,ntilesy-1 do begin
        solvecpx=solvectiles[ixt,iyt]/arcsecpx 
        veclength=veclenarr[ixt,iyt] ; vector length shift in px ;
        ysolpx=ycentiles[ixt,iyt]/arcsecpx   ; solar Y of tile center in px
        if (solvecpx ge izone*zonewidth and solvecpx lt (izone+1)*zonewidth $
            and veclength lt abs(discard-2)) then begin
          sumzone=sumzone+veclength
          ninzone=ninzone+1
; radial component 
          alpha=asin(ytshifts[ixt,iyt]/veclength) ; shift angle from X ;
          gamma=asin(ysolpx/solvecpx) ; signed (x,y) limb direction from X ;
          beta=alpha-gamma ; radians project angle shiftvector on radial ;
          delrad=cos(beta)*veclength
          good=where(finite(delrad)) ; there are NaNs from 0/0 ;
          if (good ne -1) then sumrad=sumrad+delrad
        endif
      endfor
    endfor
    averzones[izone]=sumzone/ninzone
    radzones[izone]=sumrad/ninzone
  endfor
  psfilename=$
    outdir+'/zonalshifts_'+datetime+'_'+wavpair[0]+'_'+wavpair[1]+'.ps'
  openpsplot,psfilename,thick=2,fontsize=5.5,xsize=8.8,ysize=8.8/1.62
  pxzones=(indgen(nzones)+0.5)*zonewidth
  rzones=pxzones*arcsecpx/rsun_obs
  rrange=[0.0,1.0]
  yrange=[0,max(averzones)*1.1]
;@@@ add confidence bars for the zone means, needs storage in (ixt,iyt) arrays
  
  plot,rzones,averzones,psym=-4,$
    position=[0.2,0.2,0.8,0.8],$        ; margins ;
    xrange=rrange,yrange=yrange,$
    xstyle=9,ystyle=9,$  ; no axes on top and at right ;
    xtitle=textoidl("r/R_{sun} = sin \theta"),$
    ytitle='mean length + radial part  [px]'
  oplot,rzones,radzones,psym=-5
  
; add km axis at right
  kmrange=[0,max(averzones)*1.1*arcsecpx/rsun_obs*695598.]
  axis,yaxis=1,yrange=kmrange,ystyle=1,$
    ytitle='',$
    ytickname=replicate(' ',60),ytick_get=tempaxticks
  tempticknames=string(tempaxticks,format='(i5)') 
  axis,yaxis=1,yrange=kmrange,ystyle=1,$
    ytitle='mean length, radial part  [km]',ytickname=tempticknames

; add mu axis on top (follow Alfred's plotkopower.pro)
  mutickpos=[1.0,0.9,0.8,0.7,0.6,0.5,0.4,0.0]
  muticknames=['1.0','0.9','0.8','0.7','0.6','0.5','0.4','0.0']
  muticks=n_elements(mutickpos)-1
  rmuticks=sqrt(1.-mutickpos^2) 
  axis,xaxis=1,xticks=muticks,xtickv=rmuticks,xtickname=muticknames,$
    xminor=1,xtitle=textoidl("\mu = cos \theta")

  ; done
  closepsplot,psfilename,opengv=showps
endif

end


; =============== main for testing per IDLWAVE H-c ======================

;; ; Jun 20 2020 parameter-fixing trials
;; datadir='~/data/SDO/2019-11-11-transit/midpointfull/target/cubes_noalign/'
;; ;; datadir='~/data/SDO/2019-11-11-transit/midpointfull/center/cubes/' ; ######
;; datetime='2019-11-11_00:00' ; time fake, doesn't matter 
;; ;; wavpair=['1600','1700']
;; wavpair=['304','magnetogram']

outdir='/tmp'
datetime='2020.01.01_00:00'
wavpair=['1600','1700']

; params
shiftchart=1
plotzonal=1
showps=3
verbose=0
;; arrowsize=10.

maketilefigs,$
  outdir=outdir,datetime=datetime,wavpair=wavpair,arcsecpx=arcsecpx,$
  shiftchart=shiftchart,arrowsize=arrowsize,$
  plotzonal=plotzonal,showps=showps,verbose=verbose

end
