; file: ibis2crispex.pro for <new> fits files from SST/CRISP
; init: Sep 17 2016  Rob Rutten  Deil from crispfits2fits.pro
; last: Sep 17 2016  Rob Rutten  Deil


;+

pro ibis2crispex,infile,outfile,$
  xrange=xrange,yrange=yrange,trange=trange,$
  byt=byt,nx_out=nx_out,ny_out=ny_out,nt_out=nt_out

 ; NAME:
 ;   ibis2crispex
 ; PURPOSE:
 ;   write integer (or byte) single-wav crispex cubefile in old LP format
 ;   from a 4D IBIS fitscube
 ; CALL:
 ;   see above
 ; INPUTS:
 ;   infile = string 'path/ibisfile'
 ;   outfile = string 'path/outfile.fits' 
 ; OPTIONAL INPUTS:
 ;   xrange=xrange,yrange=yrange,trange=trange: limit cube ranges  
 ;   byt=byt: bytscale each image, also before summation 
 ; OUTPUTS:
 ;   integer fitscube file 
 ; OPTIONAL OUTPUTS:
 ;   nx_out, ny_out, nt_out: output cube dimensions for further use
 ; METHOD:
 ;   uses assoc to permit large files
 ; HISTORY:
 ;   Sep 17 2016 RR: start   
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  print,' ibis2crispex,infile,outfile,$'
  print,'   xrange=xrange,yrange=yrange,trange=trange,$'
  print,'   byt=byt,nx_out=nx_out,ny_out=ny_out,nt_out=nt_out]'
  return
endif

; default keywords
if (n_elements(byt) eq 0) then byt=0
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]

; set endian
bigendian=1

; read the input file header to get data type and cube dimensions
header_in=headfits(infile)
bitpix_in=fxpar(header_in,'bitpix')
nx_in=fxpar(header_in,'naxis1')
ny_in=fxpar(header_in,'naxis2')
nt_in=fxpar(header_in,'naxis3')  ; nt, nw reversed wrt CRISP fits 
nw_in=fxpar(header_in,'naxis4')  ; nt, nw reversed wrt CRISP fits 

; define input assoc frame size 
if (bitpix_in eq 8) then inarr=bytarr(nx_in,ny_in)
if (bitpix_in eq 16) then inarr=intarr(nx_in,ny_in)
if (bitpix_in eq -32) then inarr=fltarr(nx_in,ny_in)

; define output assoc frame size 
if (xrange[1] eq -1) then xrange[1]=nx_in-1
if (yrange[1] eq -1) then yrange[1]=ny_in-1
if (trange[1] eq -1) then trange[1]=nt_in-1
nx_out=xrange[1]-xrange[0]+1
ny_out=yrange[1]-yrange[0]+1
nt_out=trange[1]-trange[0]+1

; #### modify nt_out with nwavs, they now get into the 3rd dimension
nt_out=nt_out*nw_in

; open infile for assoc
get_lun, unit_in
openr,unit_in,infile
inassoc=assoc(unit_in,inarr,512)

; output header (either integer or byte, no float!)
header_out=header_in

; modify dimensions (sxaddpar can add but also modify)
sxaddpar,header_out,'naxis1',nx_out
sxaddpar,header_out,'naxis2',ny_out
sxaddpar,header_out,'naxis3',nt_out
sxaddpar,header_out,'naxis4',nw_out

; modify bitpix = word type; default integer, otherwise byte
bitpix_out=16
if (bytscale ne 0) then bitpix_out=8
sxaddpar,header_out,'bitpix',bitpix_out

; size header_out
sizeheader_out=size(header_out)  
; fits header = Nx36 "card images" = Nx2880 bytes
header_outsize=(1+fix(sizeheader_out[1]/36.))*2880

; open outfile for assoc
get_lun, unit_out
if (bigendian) then openw,unit_out,outfile,/swap_if_little_endian $
else openw,unit_out,outfile
;; if (bitpix_out eq -32) then outassoc=assoc(unit_out,fltarr(nx_out,ny_out),$
;;                                        header_outsize)
if (bitpix_out eq 16) then outassoc=assoc(unit_out,intarr(nx_out,ny_out),$
                                      header_outsize)
if (bitpix_out eq 8) then outassoc=assoc(unit_out,bytarr(nx_out,ny_out),$
                                     header_outsize)
if (header_outsize ne 0) then begin
  rec=assoc(unit_out, bytarr(header_outsize))
  rec[0]=byte(header_out)
endif

; loop over images
; ----------------

for it=trange[0],trange[1] do begin
  if (wavdim eq 1) then begin
    image=inassoc[it*nw_in*ns+istokes*nw_in+iwav]
    if (byt eq 1) then image=bytscl(image)
  endif
  if (wavdim eq 2) then begin
    image=inassoc[it*nw_in*ns+istokes*nw_in+iwav[0]]
    if (byt eq 1) then image=bytscl(image)
    for iw=iwav[0]+1,iwav[1] do begin
      nextimage=inassoc[it*nw_in*ns+istokes*nw_in+iw]
      if (byt eq 1) then nextimage=bytscl(nextimage)
      image=image+nextimage
    endfor
    image=image/(iwav[1]-iwav[0])
    if (byt eq 1) then image=bytscl(image)
  endif

  outassoc[it-trange[0]]=image[xrange[0]:xrange[1],yrange[0]:yrange[1]]
endfor

free_lun,unit_in,unit_out

end

; =============== test per IDLWAVE S-c ==============================

path='/home/rutten/data/SST/2011-05-07-test/'
infile='crispex/halpha_1.2011.05.07.strt.algn.icube'
outfile='sstcubes/halc.fits'
ibis2crispex,path+infile,path+outfile,1,3
showex,path+outfile 

end
