; file: flickrr.pro = general 2 image blinker
; init: sometime before 1991, SPO
; last: Nov 25 2017  Rob Rutten  Deil
; note: IDLWAVE shell goes haywire at keyboard input

;+

pro flickrr,input1,input2,$
  duration=duration,it1=it1,it2=it2,delay=delay,mag=mag,$
  sqrtint=sqrtint,logint=logint,nobyt=nobyt,keyboard=keyboard

 ; NAME:
 ;   flickrr
 ; PURPOSE:
 ;   blink two images or movie frames or fitscube frames
 ;   with autofit to screen size 
 ; CALLING SEQUENCE: 
 ;   flickrr,input1,input2 $
 ;     [,it1=it1,it2=it2,delay=delay,mag=mag,nobyt=nobyt]
 ; INPUTS:
 ;   input1, input2 = each a 2D image array or a 3D movie cube 
 ; KEYWORD PARAMETERS:
 ;   duration: quit after duration sec (for no-keyboard IDL sessions)
 ;      default duration=10
 ;      duration=0: run until quit per keyboard (in terminal running IDL)
 ;   it1=it1, it2=it2: select initial time step(s) 3D arrays 
 ;   delay=delay:blink speed in seconds (float, default 1 s)
 ;   mag=mag: magnification factor (float)
 ;     default: image size fitted to screen size
 ;     mag=1: original size
 ;   sqrtint=1/0: show sqrt(int) for both (default 0)
 ;   logint=1/0: show alog10(int) for both (default 0)
 ;   nobyt=1/0: no byte scaling prior to projection
 ;   keyboard=1/0: sense keyboard for on-the-run parameter changes
 ; OUTPUT:
 ;   active blinker, optional keyboard options (don't in IDLWAVE shell):
 ;      space = interrupt/restart
 ;      1 = step it1 down
 ;      2 = step it1 up
 ;      9 = step it2 down
 ;      0 = step it2 up
 ;      5 = step both it1 and it2 down
 ;      6 = step both it1 and it2 up
 ;      [ = decrease blink speed
 ;      ] = increase blink speed
 ;      q = quit   (IDLWAVE: focus on shell, type CR)
 ; EXAMPLE:
 ;   flickrr,ar,ar,it2=50     blink two frames of a movie at it=0, 50
 ; HISTORY:
 ;  <1991: Steve Keil, Sac Peak 
 ;   1991: Herbie Cohl, Sac Peak summer student
 ;  >1991: Rob Rutten
 ;   Aug 10 2014 RR: runtime keyboard control
 ;   Sep 12 2015 RR: added default duration=10 stop (non-keyboard use)
 ;   Sep 19 2017 RR: keyboard sensing optional (I never used them)
 ;   Nov 25 2017 RR: commented out file inputs: better use showex. 
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  print,' flickrr, input1, input2'
  print,' [,duration=duration,it1=it1,it2=it2,delay=delay,mag=mag,$'
  print,'  sqrtint=sqrtint,logint=logint,nobyt=nobyt,keyboard=keyboard]'
  print,'  defaults: duration=10,delay=1.0, rest 0'
  print,'  optional keyboard control in IDL in terminal (not IDLWAVE shell):'
  print,'    space = interrupt/resume'
  print,'    0,1 = step it1 down or up'
  print,'    9,0 = step it2 down or up'
  print,'    5,6 = step both it1 and it2 down or up'
  print,'    [,] = slow down / speed up the blinking'
  print,'    terminate with space, q (IDLWAVE: mouse focus on shell, CR)'
  print,'  inputs = 2D or 3D arrays'
  sp,flickrr
  return
endif

; default parameters
if (n_elements(duration) eq 0) then duration=10
if (n_elements(it1) eq 0) then it1=0
if (n_elements(it2) eq 0) then it2=0
if (n_elements(delay) eq 0) then sdelay=1.0 else sdelay=delay
if (n_elements(nobyt) eq 0) then nobyt=0
if (n_elements(keyboard) eq 0) then keyboard=0

; default values
nt1=1  ; default for loose image rather than [x,y,t] cubes
nt2=1

; find type and dimension of inputs
arr1dim=-1
size1=size(input1)
if (size1[0] eq 0 and size1[1] eq 7) then file1=1 else file1=0
if (size1[0] eq 2) then arr1dim=2
if (size1[0] eq 3) then arr1dim=3 
if (file1 eq 0 and arr1dim eq -1) then begin
  print,' ##### flickrr abort: input1 is not array or string' 
  return
endif
arr2dim=-1
size2=size(input2)
if (size2[0] eq 0 and size2[1] eq 7) then file2=1 else file2=0
if (size2[0] eq 2) then arr2dim=2
if (size2[0] eq 3) then arr2dim=3 
if (file2 eq 0 and arr2dim eq -1) then begin
  print,' ##### flickrr abort: input2 is not array or string' 
  return
endif

; input1 = file
if (file1 eq 1 or file2 eq 1) then begin
 print,' ##### flickrr no longer accepts files; use showex,file1,file2'
 return
endif

;; ; set endian
;;   bigendian=1

;; ; find dimensions and bitpix values file input1
;;   inheader1=headfits_rr(input1)
;;   inheader1size=(1+fix(n_elements(inheader1)/36.))*2880
;;   bitpix1=fxpar(inheader1,'bitpix')
;;   nx1=fxpar(inheader1,'naxis1') 
;;   ny1=fxpar(inheader1,'naxis2') 
;;   nt1=fxpar(inheader1,'naxis3') 
;;   if (nt1 eq 0) then nt1=1

;; ; if 3D fitscube open file input1 for assoc
;;   if (nt1 ne 1) then begin
;;     get_lun, unit1
;;     if (bigendian) then openr,unit1,input1,/swap_if_little_endian $ 
;;     else openr,unit1,input1
;;     if (bitpix1 eq -32) then inass1=assoc(unit1,fltarr(nx1,ny1),inheader1size)
;;     if (bitpix1 eq 16) then inass1=assoc(unit1,intarr(nx1,ny1),inheader1size)
;;     if (bitpix1 eq 8) then inass1=assoc(unit1,bytarr(nx1,ny1),inheader1size)
;;   endif

;; endif

;; ; input2 = fits file
;; if (file2 eq 1) then begin

;; ; set endian
;;   bigendian=1

;; ; find dimensions and bitpix values file input2
;;   inheader2=headfits_rr(input2)
;;   inheader2size=(1+fix(n_elements(inheader2)/36.))*2880
;;   bitpix2=fxpar(inheader2,'bitpix')
;;   nx2=fxpar(inheader2,'naxis1') 
;;   ny2=fxpar(inheader2,'naxis2') 
;;   nt2=fxpar(inheader2,'naxis3') 
;;   if (nt2 eq 0) then nt2=1

;; ; if 3D fitscube open file input2 for assoc
;;   if (nt2 ne 1) then begin
;;     get_lun, unit2
;;     if (bigendian) then openr,unit2,input2,/swap_if_little_endian $ 
;;     else openr,unit2,input2
;;     if (bitpix2 eq -32) then inass2=assoc(unit2,fltarr(nx2,ny2),inheader2size)
;;     if (bitpix2 eq 16) then inass2=assoc(unit2,intarr(nx2,ny2),inheader2size)
;;     if (bitpix2 eq 8) then inass2=assoc(unit2,bytarr(nx2,ny2),inheader2size)
;;   endif
;; endif

; check initial it1 and it2 
if (it1 lt 0) then begin
  print,' ===== flickrr: it1 < 0, set to 0'
  it1=0
endif
if (it2 lt 0) then begin
  print,' ===== flickrr: it2 < 0, set to 0'
  it2=0
endif
if (it1 gt nt1-1) then begin
  print,' ===== flickrr: it1 cut to nt1-1 = '+ntostr(nt1-1)
  it1=nt1-1
endif
if (it2 gt nt2-1) then begin
  print,' ===== flickrr: it2 cut to nt2-1 = '+ntostr(nt2-1)
  it2=nt2-1
endif

initblink=1

; set wall-clock timer (seconds)
timestart=systime(1) 

STARTBLINK:

while get_kbrd(0) eq ''  do begin

; check elapsed time since start against specified duration
timenow=systime(1)
timedone=timenow-timestart
if (duration gt 0 and timedone gt duration) then goto,ENDBLINK

; images are 2D arrays in memory as input
  if (arr1dim eq 2) then im1=input1
  if (arr2dim eq 2) then im2=input2

; images are current it time step of 3D arrays in memory as input
  if (arr1dim eq 3) then im1=reform(input1[*,*,it1])
  if (arr2dim eq 3) then im2=reform(input2[*,*,it2])

;; ; images are 2D image fits files
;;   if (file1 eq 1 and nt1 eq 1) then im1=readfits(input1) 
;;   if (file2 eq 1 and nt2 eq 1) then im2=readfits(input2)

;; ; images are current it time step of 3D fitscube file
;;   if (file1 eq 1 and nt1 gt 1) then im1=inass1[it1]
;;   if (file2 eq 1 and nt2 gt 1) then im2=inass2[it2]

; rescale intensity if specified
  if (keyword_set(sqrtint)) then begin
    im1=sqrt(im1>0.0)
    im2=sqrt(im2>0.0)
  endif

  if (keyword_set(logint)) then begin
    im1=alog10(im1>0.0)
    im2=alog10(im2>0.0)
  endif

; bytscl unless keyword nobyt is specified
  if (nobyt eq 0) then begin
    im1=bytscl(im1)
    im2=bytscl(im2)
  endif

; initialize display the first time 
  if (initblink eq 1) then begin

; get dimensions
    im1size=size(im1)
    im2size=size(im2)

; set default magnification or reduction for screen filling
    screen=get_screen_size()
    xsize=max([im1size[1],im2size[1]])
    ysize=max([im1size[2],im2size[2]])
    if (n_elements(mag) eq 0) then $
      usemag=0.95*min([float(screen[0])/xsize,float(screen[1])/ysize]) $
    else usemag=mag 

; find size after magnify or reduce 
    if (usemag ne 1) then begin
      im1=congrid(im1,fix(usemag*im1size[1]),fix(usemag*im1size[2]))
      im2=congrid(im2,fix(usemag*im2size[1]),fix(usemag*im2size[2]))
      im1newsize=size(im1)
      im2newsize=size(im2)
      xsize=max([im1newsize[1],im2newsize[1]])
      ysize=max([im1newsize[2],im2newsize[2]])
    endif

; open temporary image-size window but not larger than the screen
    border=15 ; seems to be necessary
    if (xsize gt screen[0]-border) then xsize=screen[0]-border
    if (ysize gt screen[1]-border) then ysize=screen[1]-border
    window,/free,xsize=xsize,ysize=ysize,xpos=screen[0]-50-xsize 
      ; on the right to keep IDL shell in view

; end of initialization
    initblink=0
  endif

; magnify or reduce subsequent images
  if (initblink eq 0 and usemag ne 1) then begin
    im1=congrid(im1,fix(usemag*im1size[1]),fix(usemag*im1size[2]))
    im2=congrid(im2,fix(usemag*im2size[1]),fix(usemag*im2size[2]))
  endif

; now blink!

  tv,im1
  wait,sdelay
  tv,im2
  wait,sdelay

endwhile

if (keyboard ne 0) then begin
key=get_kbrd(/key_name)

if (key eq 'q') then goto, ENDBLINK

if (key eq '1' or key eq '5') then begin
  it1=it1-1
  print,'  it1 = '+ntostr(it1,format='(I3)')+$
    '  it2 = '+ntostr(it2,format='(I3)')
; check new it1 non-negative
  if (it1 lt 0) then begin
    print,' ===== flickrr: it1 < 0, set to 0'
    it1=0
  endif
endif

if (key eq '2' or key eq '6') then begin
  it1=it1+1
  print,'  it1 = '+ntostr(it1,format='(I3)')+$
    '  it2 = '+ntostr(it2,format='(I3)')
; check it1 not larger than nt1
  if (it1 gt nt1-1) then begin
    print,' ===== flickrr: it1 cut to nt1-1 = '+ntostr(nt1-1)
    it1=nt1-1
  endif
endif

if (key eq '9' or key eq '5') then begin
  it2=it2-1
  print,'  it1 = '+ntostr(it1,format='(I3)')+$
    '  it2 = '+ntostr(it2,format='(I3)')
; check it2 non-negative
  if (it2 lt 0) then begin
    print,' ===== flickrr: it2 < 0, set to 0'
    it2=0
  endif
endif

if (key eq '0' or key eq '6') then begin
  it2=it2+1
  print,'  it1 = '+ntostr(it1,format='(I3)')+$
    '  it2 = '+ntostr(it2,format='(I3)')
; check it2 not larger than nt2
  if (it2 gt nt2-1) then begin
    print,' ===== flickrr: it2 cut to nt2-1 = '+ntostr(nt2-1)
    it2=nt2-1
  endif
endif

if (key eq '[') then begin
  sdelay=1.2*sdelay
  print, ' ===== flickrr: delay reset to '+$
    ntostr(sdelay,format='(f5.1)')+' sec'
endif

if (key eq ']') then begin
  sdelay=0.8*sdelay
  print, ' ===== xflickrr: delay reset to '+$
    ntostr(sdelay,format='(f5.1)')+' sec'
endif

endif ; end of optional keyboard sensing

goto,STARTBLINK

ENDBLINK:

; delete window
if (!d.window ne -1) then wdelete,!d.window 

;; ; close open files
;; if (file1) then close,unit1
;; if (file2) then close,unit2

end

; ==================== main for testing by IDLWAVE H-c ====================

path='/home/rutten/data/SDO/2014-06-14-small/target/cubes/'
file1=path+'aia304.fits'
file2=path+'aia171.fits'
a304=readfits(path+'aia304.fits')
a171=readfits(path+'aia171.fits')

flickrr,a304,a171    ;; ,delay=0.1,duration=10

end
