; file: findimshift_tiled,pro
; init: Dec 12 2020  Rob Rutten  Deil formerly sdo_tilefindshift.pro
; last: Dec 12 2020  Rob Rutten  Deil

;+
function findimshift_tiled,im1,im2,$
  outdir=outdir,timediff=timediff,heightdiff=heightdiff,$
  arcsecpx=arcsecpx,pxtile=pxtile,tilecut=tilecut,precision=precision,$
  writetileshifts=writetileshifts,wavpair=wavpair,datetime=datetime,$
  shiftchart=shiftchart,minleft=minleft,blink=blink,verbose=verbose

 ; find shift to put im2 on im1 using tiled cross-correlations
 ;   
 ; DESCRIPTION:
 ;  - tesselate images into tiles,
 ;  - optionally adjust for local height-of-formation difference,
 ;  - optionally apply local derotation for time difference,
 ;  - cross-correlate each pair of tiles,
 ;  - iteratively discard outliers, find mean tile shift = image offset,
 ;  - repeat the tile fitting with newly shifted im1 in a loop that should
 ;    converge.
 ;   
 ; INPUTS:
 ;   im1,im2 = image arrays
 ;    - should be (much) larger than pxtile, fulldisk permitted
 ;    - same dimensions and pixel size but may differ in type
 ;    - should be mucked for better alikeness
 ;    - for timediff derotation and heightdiff correction images
 ;      must be (X,Y) oriented with disk center at image center
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   outdir: for download .fits, result .dat, .ps; default /tmp may overflow
 ;   timediff: im1 taken later than im2 in seconds (default 0, for diffderot)
 ; ----- parameters that should be set by sdo_muckimagepair.pro
 ;   heightdiff: "formation height" im1 above im2 in km (default 0)
 ;   arcsecpx: arcsec/px, default 0.6 = AIA and aia_prepped HMI
 ;   pxtile: requested tile x and y size in px (default 50 = 30 arcsec SDO) 
 ;   tilecut: cut outlier multiplier to standard deviation (default 2)
 ;   precision: stop loop at this increment, unit px (default 0.01)
 ; ----- parameters for tile chart production
 ;   writetileshifts = 1/0: write them in /tmp file for maketilefigs.pro 
 ;   datetime: anytim string, only for limb limit and /tmp output filenames
 ;   wavpair: 2-elem string array level2 sdowavs for /tmp output filenames
 ; ----- output options
 ;   shiftchart = 1/0: write results and run maketilefigs.pro > ps shiftchart
 ;   minleft: minimum fraction tiles that must be passed, dfault 0.2
 ;   blink = 1/0: optional shifted-image blinker
 ;   verbose = 1/0   
 ;   
 ; OUTPUTS:
 ;   function result = [shiftx,shifty,confx,confy] for im2 relative to im1
 ;     where confx,confy = +- 95% confidence intervals from tile statistics
 ;   if requested /tmp/tileshifts file for maketilefigs.pro
 ;
 ; HISTORY:
 ;   Feb 21 2017 RR: start findtiledimshift.pro, later sdo_tilefindshift.pro
 ;   Jan 10 2018 RR: no border/triangle cropping for the tiles
 ;   Jul 31 2018 RR: better vector charts, zonal fit
 ;   Jun 16 2020 RR: improved treatment outliers, heightdiff, charts
 ;   Jun 25 2020 RR: timediff, tile figure making > maketilefigs.pro
 ;   Dec 12 2020 RR: > findimshift_tiled.pro for generalization (GONG)
;-

; answer no-parameter query 
if (n_params(0) lt 2) then begin
  sp,findimshift_tiled
  return,-1       ;RR return,-1 for a function
endif

; defaults for keywords (headers below)
if (n_elements(outdir) eq 0) then outdir='/tmp' 
if (n_elements(datetime) eq 0) then datetime='2000.06.01_00:00' ; small r_sun
if (n_elements(wavpair) eq 0) then wavpair=['A','B']
if (n_elements(arcsecpx) eq 0) then arcsecpx=0.6 ; default SDO pixels
if (n_elements(pxtile) eq 0) then pxtile=50
if (n_elements(tilecut) eq 0) then tilecut=2
if (n_elements(precision) eq 0) then precision=0.01
if (n_elements(heightdiff) eq 0) then heightdiff=0
if (n_elements(timediff) eq 0) then timediff=0
if (n_elements(writetileshifts) eq 0) then writetileshifts=0
if (n_elements(shiftchart) eq 0) then shiftchart=0
if (n_elements(minleft) eq 0) then minleft=0.2 
if (n_elements(blink) eq 0) then blink=0
if (n_elements(verbose) eq 0) then verbose=0

; get and check sizes
size1=size(im1)
size2=size(im2)
if (size1[1] ne size2[1] or size1[2] ne size2[2]) then begin
  print, ' ##### findimshift_tiled abort: unequal image sizes'
  return,-1
endif
nx=size1[1]
ny=size1[2]

; get im1 rms 
dummy=moment(im1,sdev=sdevim1)

; adapt tile size to have odd nr tiles and odd px per tile (center=center)
nxtile=pxtile
for istep=0,1000 do begin
  ntilesx=fix(nx/nxtile)
  if (ntilesx/2. ne ntilesx/2 and nxtile/2. ne nxtile/2) then BREAK
  nxtile=nxtile+1
endfor
nytile=nxtile      ; use square tiles
ntilesy=ntilesx    ; square image 
ntiles=ntilesx*ntilesy

; set radpxlimit to value within limb or to image size if smaller
datetimeut=anytim2utc(datetime,/ccsds)
rsun_obs=get_rb0p(datetimeut,/radius,/quiet) ; observed radius in arcsec 
rsunpx=rsun_obs/arcsecpx
rsunkm=696340. 
radaslimit=fix(rsun_obs-0.6*nxtile*arcsecpx) ; radial limit in arcsec
radiusim=sqrt(nx^2+ny^2)/2.*arcsecpx         ; check non-full-disk image
if (radaslimit gt radiusim+1) then radlimit=radiusim+1

; initializations
nlimb=0
nflat=0
xcen=0 ; requirement for input images for timediff diffderot and heightdiff
ycen=0 
discard=-500 ;RR if you change this then also adapt in maketilefigs.pro

; define arrays
xshifttiles=fltarr(ntilesx,ntilesy)
yshifttiles=fltarr(ntilesx,ntilesy)
xcentiles=fltarr(ntilesx,ntilesy)
ycentiles=fltarr(ntilesx,ntilesy)
xfintiles=fltarr(ntilesx,ntilesy)
yfintiles=fltarr(ntilesx,ntilesy)
solvectiles=fltarr(ntilesx,ntilesy)

; xcentiles = solar X at each tile center
; ycentiles = solar Y at each tile center
; solvectiles = arcsec vectors from solar disk center to tile centers
for ixt=0,ntilesx-1 do begin
  for iyt=0,ntilesy-1 do begin
    xcentiles[ixt,iyt]=xcen+arcsecpx*((ixt-ntilesx/2.+0.5)*nxtile)
    ycentiles[ixt,iyt]=ycen+arcsecpx*((iyt-ntilesy/2.+0.5)*nytile)
    solvectiles[ixt,iyt]=sqrt(xcentiles[ixt,iyt]^2+ycentiles[ixt,iyt]^2)
    if (solvectiles[ixt,iyt] ge radaslimit) then nlimb=nlimb+1
  endfor
endfor
ndisk=ntiles-nlimb

; ====== correct im1 per tile for heightdiff and timediff versus im2

; correction loop over all tiles
corrim1=im1 ; remains the original if no corrections get applied
if (heightdiff ne 0 or timediff ne 0) then begin
  nxadd=0
  nyadd=0
  for ixt=0,ntilesx-1 do begin
    for iyt=0,ntilesy-1 do begin

; get tile center parameter values
      solx=xcentiles[ixt,iyt]
      soly=ycentiles[ixt,iyt]
      solvec=solvectiles[ixt,iyt] 

; treat only on-disk tiles
      if (solvec lt radaslimit) then begin 

; cut out larger field per tile to avoid blank border creep-in
; while shifting small images rather than whole every time for speed.
; edge tiles or spill-over shifts still get blank strips - so be it
; no difference for full disk, is this necessary for center cutouts?
        borderextra=10 ; arbitrary, sim heightdiff=4320
        if (nxadd gt nxtile/2) then nxadd=nxtile/2
        if (nyadd gt nytile/2) then nyadd=nytile/2
        nxadd=0
        nyadd=0
        if (ixt eq 0 or ixt eq ntilesx-1) then nxadd=0 else nxadd=borderextra
        if (iyt eq 0 or iyt eq ntilesy-1) then nyadd=0 else nyadd=borderextra
        bigcutim1=corrim1[ixt*nxtile-nxadd:(ixt+1)*nxtile-1+nxadd,$
                          iyt*nytile-nyadd:(iyt+1)*nytile-1+nyadd]
        meanbigcutim1=avg(bigcutim1)

;; ; for showex check below
;;       bigcutim0=bigcutim1
        
; get intensity rms bigtile, check constrast or call it flat and discard
; (maybe blank tile from shift-in on right side?)
        dummy=moment(bigcutim1,sdev=sdevtile)
        if (sdevtile lt 0.1*sdevim1) then begin  ; RR value choice
          nflat=nflat+1
          thisshift=[discard,discard]
          goto, DISCARD
        endif
        
; heightdiff correction: shift im1 tile image back over heightdiff component
        if (heightdiff ne 0) then begin
          sintheta=solvec/rsun_obs           ; r/R value
          delradkm=heightdiff*sintheta       ; projected height difference
          delradpx=delradkm*(rsunpx/rsunkm)  ; in px units
; position angle of sol (X,Y) tile vector from disk center
          gamma=atan(abs(soly)/(abs(solx)>0.01))   ; 0.01 avoids y/0=NaN
          delradx=delradpx*cos(gamma)
          delrady=delradpx*sin(gamma)
; correct quadrant signs
          if (solx lt 0) then delradx=-delradx
          if (soly lt 0) then delrady=-delrady
; shift tile back over radial height-difference component
          if (max(abs([delradx,delrady])) gt 0.01) then $
            bigcutim1=shift_img(bigcutim1,-[delradx,delrady],$
                                missing=meanbigcutim1) ; borders local grey
        endif ; end of heightdiff correction
        
; differentially derotate local im1 tile for im1 sampled later than im2
;RR Dec 11 2020 this may clash with JSOC im_patch tracking but isn't
;RR used anymore in sdo_writeallpairsplines.pro: there timediff=0
;RR but it it can be used for full-disk or non-tracking downloads (as GONG)
        if (timediff ne 0) then begin
          newpos=rot_xy(solx,soly,-timediff,date=datetime)
          diffshift=reform(newpos)-[solx,soly]
          if (max(abs(diffshift)) gt 0.01) then $
            bigcutim1=shift_img(bigcutim1,diffshift,missing=meanbigcutim1)
        endif ; end of derotation correction

       ;; ; check
       ;; if (ixt eq ntilesx/3 and iyt eq ntilesy/3) then begin
       ;;   showex,bigcutim0,bigcutim1
       ;;   STOP
       ;; endif
        
; replace cutim parts of corrim1 with the shift-corrected version 
        szbigcut=size(bigcutim1)
        nxbig=szbigcut[1]
        nybig=szbigcut[2]
        corrim1[ixt*nxtile:(ixt+1)*nxtile-1,$
                iyt*nytile:(iyt+1)*nytile-1] = $
          bigcutim1[nxadd:nxbig-nxadd-1,nyadd:nybig-nyadd-1]
      endif ; end condition tile lies within radaslimit

DISCARD:      
    endfor ; end double correction loop over tiles 
  endfor
endif   ; end of tile corrections for heightdiff and derotation

; ========= iterative all-tile shift determination refinement loop

sumshift=[0.,0.]
nloop=10         ; default maximum if not cut by reaching precision before
shiftim1=corrim1 ; original or corrected im1 at start of loop
szshim=size(shiftim1)
nxsh=szshim[1]
nysh=szshim[2]
tileshiftim1=corrim1      ; file for blink only
meancorrim1=avg(corrim1)  ; for missing in shift_img
for iloop=0,nloop-1 do begin

; measure shifts for all tiles within limb limit (only 2 iterations for speed)
  for ixt=0,ntilesx-1 do begin
    for iyt=0,ntilesy-1 do begin

;;    if (verbose) then print,' ----- next ixt,iyt ='+trimd(ixt)+trimd(iyt)
      
      thisshift=[discard,discard]   ; for excluded tiles

; condition = still on disk 
      solvec=solvectiles[ixt,iyt] ; distance disk center to tile center in px
      if (solvec lt radaslimit) then begin ; exclude limb tiles

; condition = still within image
        if ((ixt+1)*nxtile-1 lt nxsh and (iyt+1)*nytile-1 lt nysh) then begin
          
; cut the tile pair, first time from corrected, later from the latest
          cutim1=shiftim1[ixt*nxtile:(ixt+1)*nxtile-1,$
                          iyt*nytile:(iyt+1)*nytile-1]
          cutim2=im2[ixt*nxtile:(ixt+1)*nxtile-1,$
                     iyt*nytile:(iyt+1)*nytile-1]
          
; find shift in this loop iteration
          thisshift=findimshift_rr(cutim1,cutim2,/subpix,niter=2,filter=10)

; fill current tileshiftim1 for optional blink at the end (will be latest) 
          if (blink ne 0) then $
            tileshiftim1[ixt*nxtile:(ixt+1)*(nxtile)-1,$
                         iyt*nytile:(iyt+1)*(nytile)-1]=cutim1
         endif ; end loop over within-image tiles
      endif  ; end loop over on-disk tiles

; stick in shifttiles
      xshifttiles[ixt,iyt]=thisshift[0]  ; fresh values per tile
      yshifttiles[ixt,iyt]=thisshift[1]  ; end up as variations from mean
    endfor
  endfor

; iteratively remove on-disk outliers beyond cutsdev*rms over all disk tiles
  xtemp=xshifttiles[where(xshifttiles gt discard+1)] ; exclude limb tiles 
  ytemp=yshifttiles[where(yshifttiles gt discard+1)]
; iterate outlier removal, an outlier in one also removes the other valuw
  for itercut=0,2 do begin
    momx=moment(xtemp,sdev=sdevx)
    xtempcut=where(abs(xtemp-momx[0]) gt tilecut*sdevx)
    if (xtempcut[0] ne -1) then remove,xtempcut,xtemp,ytemp
    momy=moment(ytemp,sdev=sdevy)
    ytempcut=where(abs(ytemp-momy[0]) gt tilecut*sdevy)
    if (ytempcut[0] ne -1) then remove,ytempcut,xtemp,ytemp

; set new outliers in this-pass shift arrays to discard value
    xtilescut=where(abs(xshifttiles-momx[0]) gt tilecut*sdevx) ; limb too
    if (xtilescut[0] ne -1) then begin
      xshifttiles[xtilescut]=discard
      yshifttiles[xtilescut]=discard
    endif
    ytilescut=where(abs(yshifttiles-momy[0]) gt tilecut*sdevy) ; limb too
    if (ytilescut[0] ne -1) then begin
      xshifttiles[ytilescut]=discard
      yshifttiles[ytilescut]=discard
    endif
    
  endfor ; ===== end itercut iteration

; check ratio remaining tiles 
  nleft=n_elements(xtemp)
  tilefrac=float(nleft)/ntiles      ; fraction non-discarded tiles
  if (tilefrac lt minleft) then begin
    print,' ##### findimshift_tiled abort: only '+trim(nleft)+' valid tiles'
    STOP
  endif

; put newest valid tile shifts in final fintiles files for vector charts
; last version = tile offsets from final shift solution = local offset pattern
  wherex=where(xshifttiles gt discard+1)
  if (wherex[0] ne -1) then xfintiles[wherex]=xshifttiles[wherex] 
  wherey=where(yshifttiles gt discard+1)
  if (wherey[0] ne -1) then yfintiles[wherey]=yshifttiles[wherey] 

; set current outliers in sum files to discard
  wherex=where(xshifttiles eq discard)
  if (wherex[0] ne -1) then xfintiles[wherex]=discard
  wherey=where(yshifttiles eq discard)
  if (wherey[0] ne -1) then yfintiles[wherey]=discard

; update sumshift with new meanshift  
  meanshift=[mean(xtemp),mean(ytemp)] ; mean shift in this pass
  sumshift=sumshift+meanshift         ; sum of all passes so far

  if (verbose) then print,' -- iloop '+trimd(iloop)+$
    ' itershift ='+trimd(meanshift)+$
    ' sumshift ='+trimd(sumshift)+$
    ' nlimb ='+trimd(nlimb)+' ndisk ='+trimd(ndisk)+$
    ' nflat ='+trimd(nflat)+$   
    ' nleft ='+trimd(nleft)+$
    ' rmsx ='+trimd(sdevx)+' rmsy ='+trimd(sdevy)

; finish this loop passage by shifting corrim1 over current estimate
  shiftim1=shift_img(corrim1,-sumshift,missing=meancorrim1)

; break when good enough 
  if (abs(meanshift[0]) lt precision and $
      abs(meanshift[1]) lt precision) then break

  if (verbose and iloop eq nloop-1) then $
    print,' ##### iloop ='+trimd(iloop)+' but no convergence: BAD case'
  
endfor ; end of shift determination improvement loop

; write optional tile files for maketilefigs.pro
; no writecol since 2D; NB: nxtiles may differ from nytiles (but shouldn't)
if (writetileshifts ne 0 or shiftchart ne 0) then begin
  openw,42,outdir+'/tileshifts_'+datetime+'_'+wavpair[0]+'_'+wavpair[1]+'.dat'
  printf,42,nx,ny,nxtile,nytile,radaslimit,heightdiff
  printf,42,xcentiles
  printf,42,ycentiles
  printf,42,solvectiles
  printf,42,xfintiles
  printf,42,yfintiles
  close,42
endif

; optional shiftchart production (plotzonal too, whyse not)
if (shiftchart ne 0) then maketilefigs,outdir='/tmp',$
  datetime=datetime,wavpair=wavpair,arcsecpx=arcsecpx,$ 
  shiftchart=1,arrowsize=0,plotzonal=1,showps=3,verbose=0

; optional resulting shifted-image blinker (zoom in) 
if (blink ne 0 and heightdiff eq 0) then showex,im1,shiftim1,im2
if (blink ne 0 and heightdiff gt 0) then $
  showex,im1,im2,shiftim1,tileshiftim1,im2

; done: return shifts and 95% confidence levels = 1.96*sigma 
; IDL moment: "sdev" = rms = sqrt(variance) = standard deviation per sample
;   (where variance uses 1/(N-1) rather than 1/N for "unbiased"
; standard deviation of mean: sigma = rms/sqrt(N)
; 95% confidence level (for normal distribution) = +-1.96 sigma 
return,[sumshift,1.96*sdevx/sqrt(nleft),1.96*sdevy/sqrt(nleft)]

end


; =============== main for testing per IDLWAVE H-c ======================

;; datadir='~/data/SDO/2019-11-11-transit/midpointfull/target/cubes_noalign/'
;; datadir='~/data/SDO/2019-11-11-transit/midpointfull/center/cubes/' 
;; datadir=' /media/rutten/RRHOME/alldata/SDO/2019-11-11-transit/full_long/center/cubes/' 
;; datetime='2019-11-11_00:00' ; time fake, doesn't matter 
  ;; im1=readfitscube(datadir+'aia1600.fits',trange=[3,3])
  ;; im2=readfitscube(datadir+'aia1700.fits',trange=[3,3])

; full-disk images in /tmp gotten with sdo_diskfigs.pro
  datetime='2020.01.01_00:00'
  outdir='/tmp'
  im1=readfits(outdir+'/sdo_2020.01.01_00:00_1600.fits',head16)
  im2=readfits(outdir+'/sdo_2020.01.01_00:00_1700.fits',head17)
  wavpair=['1600','1700']
  tim1=fxpar(head16,'t_obs')
  tim2=fxpar(head17,'t_obs')
  timediff=anytim2tai(tim1)-anytim2tai(tim2)

; muck images and set muck parameters
  sdo_muckimagepair,im1,im2,wavpair,$
    muckim1,muckim2,pxtile,tilecut,heightdiff

; other params
  precision=0.01
  blink=0
  writetileshifts=1
  shiftchart=1
  verbose=0

; adapt muck results 
;; heightdiff=0
;; timediff=0
;;  pxtile=240 ; => nxtile = 400, gets modified to 411 px for odd-odd 9 tiles

  shift=$
    findimshift_tiled(muckim1,muckim2,$
      outdir=outdir,$
      timediff=timediff,heightdiff=heightdiff,arcsecpx=arcsecpx,$
      pxtile=pxtile,tilecut=tilecut,precision=precision,$
      writetileshifts=writetileshifts,wavpair=wavpair,datetime=datetime,$
      shiftchart=shiftchart,minleft=minleft,blink=blink,verbose=verbose)
  
  print,' ===== sumshifts, confs = ',shift

end

