; file: findalignimages.pro = align two images per shift, rotate, scale
; init: Aug  1 2014  Rob Rutten  Deil
; last: Aug 12 2022  Rob Rutten  Deil
; site: rridl/imagelib

;+
pro findalignimages,im1,im2,px1,px2,angle,$
  px2asym,shiftfor,shiftrev,nxfor,nyfor,nxrev,nyrev,$
  nxmuckfor=nxmuckfor,nymuckfor=nymuckfor,$
  nxmuckrev=nxmuckrev,nymuckrev=nymuckrev,$
  skipmetcalf=skipmetcalf,maxmetcalf=maxmetcalf,minmetqual=minmetqual,$
  applypx2asym=applypx2asym,$
  trimboxim2=trimboxim2,inshift12=inshift12,$
  smearim1=smearim1,smearim2=smearim2,$
  histopim1=histopim1,histopim2=histopim2,$
  muckdarkim1=muckdarkim1,muckdarkim2=muckdarkim2,$
  muckbrightim1=muckbrightim1,muckbrightim2=muckbrightim2,$
  flatten=flatten,flatim1=flatim1,flatim2=flatim2,$
  metqual=metqual,nmet=nmet,norinse=norinse,$
  blink=blink,checkmetcalf=checkmetcalf,show=show,verbose=verbose

 ; return parameters angle, px2, shifts, frame sizes to precisely align
 ; two unequally-sized, unequally-sampled, shifted images with fairly
 ; well-known pixel scale ratio and rotation angle between them.
 ; Works well if only patterns agree, not pixel detail, over large field.
 ;
 ; COMMENTS: 
 ;   Output both ways: 
 ;     "forward" = to put im1 on im2 (giving frame-filling im2 figures)
 ;     "reverse" = to put im2 on im1 (rotated within im1)
 ;   For output usage see blocks below marked by "@@@@@ demo"
 ;   Image im1 field should be large enough to contain shift, rotation
 ;     and rescaling of im2 in reverse mode.  It may be much larger.
 ;     If they are the same size or if im2 exceeds im1
 ;     selection of a trimboxim2 fitting within im1 is mandatory.
 ;   Either may have the finest sampling; output values are in finest px.
 ;   Pixel size px2 is considered the more unreliable of the two;
 ;     it may even have pixel anisotropy optionally accounted for.
 ;   Image im2 may have flat (black/white/grey) border and/or triangle strips.
 ;   Typical usage:
 ;     im1 = SDO larger-field cutout image (low-res: 0.6 arcsec pixels)
 ;     im2 = hi-res SST/CRISPEX image (hi-res: about 0.06 arcsec pixel)
 ;   The alignment works better for:
 ;     - scenes more alike
 ;     - initial angle (rotation) more precise
 ;     - initial px1/px2 ratio more precise
 ;     - better matching of image centers (smaller scene shift between them)
 ;   The program tends to correct small errors in all of these,
 ;     but check with the Metcalf blinks and result showex comparisons.
 ;   Scale and angle errors interfere and impede Metcalf speed and quality;
 ;     iteration ("metcalving") improves both.  Selecting a trimbox
 ;     increases speed and may improve matching from better scene alikeness.
 ;   The program uses temporary image "mucking" to improve similarity,
 ;     including default smearing and large-scale contrast flattening;
 ;     applying such (with smear, flatten, histop, muckdark, muckbright) 
 ;     may help: inspect the input blinks and try to make scenes more similar.
 ;
 ; DESCRIPTION: 
 ;   uses Tom Metcalf SSW autocorrelation and Rob Rutten rridl programs to:
 ;   - auto-remove flat (dark/grey/white) borders or define your own trimbox
 ;   - smear (default finest to spatial resolution of coarsest, > im1b, im2b)
 ;   - pre-wash: muck one or both images to make the scenes look more similar
 ;   - main-wash = Metcalf iteration loop:
 ;     - congrid coarsest to current px scale of finest
 ;     - rotate im1 over angle, find initial "rotshift" offset, muck
 ;     - rotate, shift, center-cut im1muck 
 ;     - apply Metcalf auto_align_images.pro to these mucks
 ;     - iterate to improve angle and px2 
 ;       (NB: I no longer apply Metcalf matrix setup and inversion because
 ;        in iteration scale and angle corrections to im2 become negligible
 ;        so that only shift determination remains.  Disadvantage over
 ;        single matrix manipulation is that the multiple iteration is slow,
 ;        but the advantage is that Metcalf fits get better along the way.)
 ;   - find the im2 field expansion needed to accommodate reverse rotation
 ;   - rinse wash: correct remaining shifts using central image part
 ;   - show: blink/showex "forward" and "reverse" matches (mucked and result)
 ;
 ; EXAMPLES:
 ;   at end of this file.
 ;
 ; MANDATORY INPUT VARIABLES:
 ;   im1 = image, any 2D array type and size, usually larger than im2 
 ;   im2 = image, any 2D array type and size, larger than im1 needs trimboxim2
 ;   px1 = pixel size im1 in e.g. arcsec, scalar
 ;   px2 = pixel size im2 in the same units, scalar
 ;   angle = scene rotation angle in degrees counterclockwise from im1 to im2
 ;    NB: px2 and angle are returned with improved values
 ;
 ; MANDATORY OUTPUT VARIABLES:
 ;   px2asym = [x,y] asymmetry px2 around the mean (may also be input)
 ;   shiftfor = im1 [xshift,yshift] in finest-px units for im1 > im2
 ;   nxfor,nyfor = central-cut common size in finest px
 ;   shiftrev = im2 [xshift,yshift] in finest-px units for im2 > im1
 ;   nxrev,nyrev = central-cut size in finest px, accommodating im2 rotation
 ;  
 ; OPTIONAL KEYWORD OUTPUTS:
 ;   nxmuckfor,nymuckfor,nxmuckrev,nymuckrev: size mucked trimbox cuts
 ;   metqual: Metcalf iteration "quality" = final scale + angle change 
 ;   nmet = nr Metcalf iterations reached
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   skipmetcalf = 1/0: determine shift only, no scale + rotate correction
 ;   maxmetcalf: set maximum to Metcalf iterations (default 4, don't overdo)
 ;   minmetqual: stop Metcalf iteration at this metqual (default 1E-4)
 ;   applypx2asym = 1/0: apply Metcalf-found px2 asymmetry in the iteration
 ;   norinse = 1/0: do not apply trimcenter shift after Metcalf iteration
 ; "muck" keywords for temporaray image changes to achieve better match
 ;   (judge mucking per blink=1 giving showex after initial shift):
 ;   trimboxim2:  px vector [xmin,ymin,xmax,ymax] defining im2 subfield
 ;     for finding alignment, for example to cut off blank borders
 ;     and derotation edges (otherwise auto-detect is applied).
 ;     or to select a better look-alike part of the scene, or to speed up,
 ;     required if im2 covers a larger solar field than im1
 ;   inshift12: start-off shift of im1 in im1 pixels from im1 center to 
 ;     where im1 shows the feature that is located at the center of im2; 
 ;     measure eg with: showex,im1,/markcenter,/noblock & showex,im2,/noblock 
 ;     (such specification is needed when this offset gets above 10-20 arcsec)
 ;     The program returns an improved value and prints this when verbose=1.
 ;   smearim1,2: in px units (default -1.5 = 1.5*pxratio = finest to coarsest)
 ;   histopim1,2: value for histo_opt_rr (top only)
 ;   muckdarkim1,2: set intensities below threshold (units mean=1) to mean
 ;   muckbrightim1,2: set intensities above threshold (units mean=1) to mean
 ;   flatten: subtract boxcar average from each, value = width in coarsest px
 ;      use eg 5 for granulated scenes with pores or limb
 ;      Nov 15 2021: added flatim12 but kept flatten for existing calls
 ;   flatim1,2: flatten each over value in its own pixels
 ; 
 ; process control keywords:
 ;   blink = flickrr duration (s), 0 = no flicker, default 10, 1 = showex
 ;   checkmetcalf = 1/0: showex final Metcalf result trimboxim2 align pair
 ;   verbose = 1/0: print more or less information along the way 
 ;   show = 0: nothing
 ;        = 1: at conclusion start showex on forward and reverse results
 ;        = 2: measure and print final center shifts instead
 ;        = 3: both 
 ;   
 ; USAGE:
 ;   - use output as in blocks marked by "@@@@@ demo"
 ;   - the output parameters enter directly into reformimage.pro and
 ;       so in reformcubefile.pro etc
 ;   - these programs permit calls by doallfiles.pro for bulk processing
 ;
 ; RESTRICTIONS:
 ;   - the two images should contain comparable scenes (maybe patterns)
 ;   - the field of view of im1 should at least cover im2,
 ;     preferably also after its transformation.  Much larger is fine.
 ; 
 ; WARNINGS:
 ;   sticky parameters needing reset or restart IDL before new run:
 ;     px2, angle, px2asym, shiftfor, shiftrev, inshift12, trimboxim2
 ;   don't overdo maxmetcalf and minmetqual: iterations may run away
 ;
 ; HISTORY:
 ;   Aug  2 2014 RR: start for SDO > SST alignment
 ;   Dec 31 2015 RR: completed complete overhaul
 ;   May  3 2017 RR: iteration within findimshift_rr.pro
 ;   May 11 2017 RR: better mucks, trimboxim2
 ;   May 21 2017 RR: Metcalf iteration, px2asym, flatten
 ;   Dec  7 2017 RR: cleanup, auto border strip, showex results
 ;   Jan 21 2020 RR: correct when im1 finest px (CHROMIS-CRISP)
 ;   Oct 20 2020 RR: correct applypx2asym, trimbox; final rinse; metqual 
 ;   Feb 28 2022 RR: correct negative smear
 ;   Aug 11 2022 RR: extensive checks, rinse default, unrolled shift_img_rr
;-

; answer a no-parameter query
if (n_params() lt 12) then begin
  sp,findalignimages
  return
endif

; keyword defaults
if (n_elements(skipmetcalf) eq 0) then skipmetcalf=0
if (n_elements(maxmetcalf) eq 0) then maxmetcalf=4
if (n_elements(minmetqual) eq 0) then minmetqual=1E-4
if (n_elements(px2asym) eq 0) then px2asym=[1.0,1.0]
if (n_elements(applypx2asym) eq 0) then applypx2asym=0
if (n_elements(trimboxim2) eq 0) then trimboxim2=-1
if (total([trimboxim2]) eq 0) then trimboxim2=-1  ; ?? why this?
if (n_elements(inshift12) eq 0) then inshift12=[0.,0.]
if (n_elements(smearim1) eq 0) then smearim1=-1.5 ; NB default
if (n_elements(smearim2) eq 0) then smearim2=-1.5 ; NB default
if (n_elements(histopim1) eq 0) then histopim1=0
if (n_elements(histopim2) eq 0) then histopim2=0
if (n_elements(muckdarkim1) eq 0) then muckdarkim1=0 
if (n_elements(muckdarkim2) eq 0) then muckdarkim2=0 
if (n_elements(muckbrightim1) eq 0) then muckbrightim1=0 
if (n_elements(muckbrightim2) eq 0) then muckbrightim2=0 
if (n_elements(flatten) eq 0) then flatten=0 ; not default=5 anymore
if (n_elements(flatim1) eq 0) then flatim1=0 
if (n_elements(flatim2) eq 0) then flatim2=0 
if (n_elements(norinse) eq 0) then norinse=0
if (n_elements(blink) eq 0) then blink=10  ; NB default
if (n_elements(checkmetcalf) eq 0) then checkmetcalf=0
if (n_elements(show) eq 0) then show=0
if (n_elements(verbose) eq 0) then verbose=0

; set wall-clock timer (seconds)
timestart=systime(1) 

; print pacifier
if (verbose ne 0) then print,' ----- findalignimages starts'

; check inshift12
if (n_elements(inshift12) ne 2) then begin
  print,' ##### inshift12 not 2 elements'
  return
endif

; check flatten
if (flatten ne 0 and (flatim1 ne 0 or flatim2 ne 0)) then begin
  print,' ##### clash flatten set and flatim1 or flatim2 set'
  return
endif

; initial pixel ratio
pxratio=float(px1)/px2    ; > 1 means im2 finer sampled (as 1=SDO, 2=SST)

; initial image sizes
size1=size(im1)
nx1=size1[1]
ny1=size1[2]
size2=size(im2)
nx2=size2[1]
ny2=size2[2]

; define smear if requested (abs of negative value = multiplier to pxratio)
if (smearim1 lt 0) then $
  if (pxratio ge 1.0) then smear1=abs(smearim1) $
  else smear1=abs(smearim1)/pxratio
if (smearim2 lt 0) then $
  if (pxratio ge 1.0) then smear2=abs(smearim2)*pxratio $
  else smear2=abs(smearim2)

; set flatten values, since Nov 15 2021 may also be done per image 
flatten1=flatim1
flatten2=flatim2

; old flatten still called in other programs; unit = coarsest px
if (flatten ne 0) then begin
  ; px_im2 < px_im1 (as im1=SDO > im2=SST)  
  if  (pxratio gt 1.0) then begin 
    flatten1=flatten
    flatten2=flatten*pxratio
  endif else begin
  ; px_im2 > px_im1 (as im1=SDO > im2=GONG)
    flatten1=flatten/pxratio
    flatten2=flatten
  endelse 
endif

; set trimbox to borders of (auto-)stripped im2 if no trimboxim2 specified
trimbox=trimboxim2
if (trimbox[0] eq -1) then begin
  reformimage,im2,im2dummy,/croptriangles,cropbox=cropbox
  trimbox=cropbox
  if (verbose eq 1) then $
    print,' ----- (nx2,ny2) = ('+trim(nx2)+','+trim(ny2)+')'+$
    '    trimbox for im2 after auto-stripping ='+trimd(trimbox)
endif

; set starting value rotshiftpx2 = rotated and scaled inshift12
rotshiftpx2=rotatevec(-inshift12*pxratio,angle) ; from im1 px to im2 px

;; ; check 
;; print,' ----- rotated inshift12 = '+trimd(rotshiftpx2)

; add im2 trimbox as shift to starting value rotshiftpx2 (tricky, all px im2)
; (the +1 is to get nr elements, alternative is (nx2-1) => axis lengths)
if (trimbox[0] ne -1) then rotshiftpx2=rotshiftpx2-[trimbox[0],trimbox[1]] $
  +[(nx2-(trimbox[2]+1-trimbox[0]))/2.,(ny2-(trimbox[3]+1-trimbox[1]))/2.]

; pre-wash: muck images as requested; trimbox im2 
reformimage,im1,im1muck,$
  histopttop=histopim1,smear=smear1,$ 
  muckdark=muckdarkim1,muckbright=muckbrightim1,flatten=flatten1,splinip=1
reformimage,im2,im2muck,trimbox=trimbox,$
  histopttop=histopim2,smear=smear2,$
  muckdark=muckdarkim2,muckbright=muckbrightim2,flatten=flatten2,splinip=1

;; ; check
;; sv,im1 & sv,im1muck
;; sv,im2 & sv,im2muck
;; STOP
;; ;; wdelall

; ===== main wash: start Metcalf scale-rotation refinement iteration loop 

px2d=[px2,px2]    ; start value for determining bi-axial pixel size im2
if (skipmetcalf eq 1) then nmetcalf=1 else nmetcalf=maxmetcalf
for itermetcalf=1,nmetcalf do begin  ; ===== Metcalf loop starting at 1

  if (skipmetcalf eq 0 and verbose ne 0) then begin
    print,' ----- start Metcalf iteration nr '+trimd(itermetcalf)+$
      '  !! takes long'
    print,' ----- startoff angle='+trimd(angle,2)+$
      '  px2 ='+trimd(px2,4) 
    if (itermetcalf eq nmetcalf) then begin
      print,' ##### Metcalf iteration seems slow '+$
        'or no convergence: check co-location, improve flatten, histop, muck?'
      print,' ##### itermetcalf=nmetcalf: Metcalving quits after this one'
    endif
  endif

; (re-)set scales
  px2d=px2*px2asym          ; refresh 2D version to newest Metcalfed px2
  pxratio=float(px1)/px2    ; refresh to newest Metcalfed px2, >1 for 1>2
  pxrat2d=float(px1)/px2d 
  pxscale1=[1.0,1.0]
  pxscale2=[1.0,1.0]
  if (pxratio lt 0.999) then begin ; px1 < px2 (non-standard, CHROMIS > CRISP)
  ; for regridding im2 to finer pixels of im1 including non-square correction
    if (applypx2asym eq 0) then pxscale2=1./[pxratio,pxratio] $
    else pxscale2=1./pxrat2d     ; pxscale1 remains [1,1]
  endif
  if (pxratio gt 1.001) then begin ; px1 > px2 (standard SDO > SST)
  ; regrid im1 to finer pixels of im2; apply non-square correction to im2
    ; rescale im1 to mean im2 pixel size 
    pxscale1=[pxratio,pxratio]  
    if (applypx2asym eq 1) then pxscale2=px2asym
    ; rescale im2 only with the relative im2 pixel asymmetry
    ; otherwise pxscale2 remains [1,1]
  endif

; error check
  if (min([pxscale1,pxscale2]) lt 0) then begin
    print,' ##### findalignimages abort: pixel scale < 0, sets metqual=0.1'
    metqual=0.1
    goto, ERROREND
  endif
  
; resample coarsest to finest (again) 
  reformimage,im1muck,im1a,congridfactor=pxscale1,splinip=1
  reformimage,im2muck,im2a,congridfactor=pxscale2,splinip=1
  
; initially rescale rotshift if im1 finest px; rotshiftiter = in finest px
  if (itermetcalf eq 1) then $
    if (pxratio lt 1.0) then rotshiftiter=rotshiftpx2*pxscale2 else $
      rotshiftiter=rotshiftpx2
  
;; ; check 
;; sv,im1a & sv,im2a
;; STOP
;; ;; wdelall

; get resampled image sizes
  sizeim1a=size(im1a)
  nx1a=sizeim1a[1]
  ny1a=sizeim1a[2]
  sizeim2a=size(im2a)
  nx2a=sizeim2a[1]
  ny2a=sizeim2a[2]

; ----- determine fine-px shift 1 to 2 after rotation and current rotshiftiter

; rotate im1a, then shift over current rotshiftiter, cut to im2a size
  reformimage,im1a,im1b,shift=rotshiftiter,rotate=angle,$ 
    cutcentralx=nx2a,cutcentraly=ny2a,splinip=1,missingvalue=-1

;RR below was pre Aug 11 2022 unrolling shift_img_rr.pro
;;   reformimage,im1a,im1x,rotate=angle,splinip=1
;;   reformimage,im1x,im1b,shift=rotshiftiter,$
;;     cutcentralx=nx2a,cutcentraly=ny2a,splinip=1

;; ; check these images 
;; showex,im1b,im2a,/blink
;; STOP
;RR image nomenclature within Metcalf iteration
;     im1,im2 are originals
;     im1muck, im2muck mucked smeared originals with im2muck trimboxed
;     im1a, im2a current finest-px congrid resamples of im1muck, im2muck
;     im1b is current rotated and shifted cutout of im1a with size im2a
;     im1d idem refined per findimshift_rr
;     im1a, im2a, im1d get updated with new px2 and px2asym in Metcalving

  im1d=im1b

; find new rotshiftiter = finest px in im2 orientation from im1b to im2a 
  if (verbose ne 0) then $
    print,' ----- findimshift_rr for rotshiftiter (finest px)'
  corshift=findimshift_rr(bytscl(im2a),bytscl(im1b),/subpix,$
                          filter=10,croptriangles=0,cropborders=1,$
                          verbose=verbose)
  rotshiftiter=rotshiftiter+corshift  ; in finest px

; check and correct inshift12 (px of im1 and im1 orientation) for new call
  corinshift=rotatevec(corshift/pxscale1,-angle)
  if (max(abs(inshift12)) gt 10 and $
      max(abs(corinshift)) gt max(abs(inshift12))) then $
        print,' !!!!! looks as runaway divergence !!!!!'
  inshift12=inshift12-corinshift

; rotate im1a again, apply corrected shift, cut again (undoing new borders)
  reformimage,im1a,im1d,shift=rotshiftiter,rotate=angle,$
    cutcentralx=nx2a,cutcentraly=ny2a,splinip=1
  
;RR below was pre Aug 11 2022 unrolling shift_img_rr.pro
  ;; reformimage,im1a,im1x,rotate=angle,splinip=1
  ;; reformimage,im1x,im1d,shift=rotshiftiter,$
  ;;   cutcentralx=nx2a,cutcentraly=ny2a,splinip=1

  if (verbose eq 1) then print,$
    ' ----- current rotshiftiter ='+trimd(rotshiftiter,2)+$
    ' improved inshift12 ='+trimd(inshift12,2)

; inspect by blinking (also when verbose=0: these should always be inspected) 
  if (blink gt 0) then begin
    if (itermetcalf eq 1) then begin
      print,' !!!!! findalignimages: initial blink shifted mucked input'
      if (skipmetcalf eq 0) then $
        print,' ----- match must be close except slight scale, rotate'
      if (skipmetcalf eq 1) then print,' ----- match must be excellent!'
      print,' ?? if not then quit and improve inshift12, px_stx, angle_stx'
      print,' ?? try different trimboxstx, flatten, histop, muck, smear?'
    endif
    if (itermetcalf eq 2) then print,' ----- Metcalf iteration: improvement?'
    if (itermetcalf gt 2) then print,' ----- Metcalf iteration: refinement?'
    if (blink gt 1) then flickrr,im1d,im2a,duration=blink $
    else showex,im1d,im2a,/blink,frame_speed=4     
  endif

; ------------ Metcalf refinement = determine <shift, rotate, scale>

  if (skipmetcalf eq 0) then begin

; run Metcalf on this input pair
    pin=[[0,0],[1.0,0]]
    qin=[[0,1.0],[0,0]]
;RR Metcalf "scale" parameter must be larger for worse starting guesses
    im1m=auto_align_images(im1d,im2a,pin,qin,pmet,qmet,/quiet)

; get rss = "rotation-shift-scale", printed by pq2rss if verbose
    if (verbose eq 1) then begin
      print,' ----- check Metcalf results below, rotation small, scales 1?'
      quiet=0
    endif else quiet=1
    pq2rss,pmet,qmet,erot,exscl,eyscl,exshft,eyshft,enrss,nx2a,ny2a,/center,$
      quiet=quiet,/rotfirst

; update angle and im2 pixel values
    angle=angle+erot
    rescale=[exscl,eyscl]
    px2d=px2d/rescale
    px2=avg(px2d)
    px2asym=px2d/px2
    metqual=abs(avg(rescale)-1.0)+abs(erot/10.)
    nmet=itermetcalf
    
; add metqual to the pq2rss printout
    if (verbose eq 1) then $
      print,'   metqual = '+strcompress(string(float(metqual)))
    
; update rotshiftiter (in newest finest px optionally including asymmetry) 
    if (applypx2asym eq 0) then $
      rotshiftiter=rotshiftiter*avg(rescale)+[exshft,eyshft] else $
        rotshiftiter=rotshiftiter*rescale+[exshft,eyshft]

; break Metcalf iteration when corrections beget negligible  
    if (metqual le minmetqual) then begin    
      if (verbose ne 0) then $
        print,' ----- metqual < minmetqual: break Metcalf iteration'
      break
    endif
  endif

endfor ; ================ end of Metcalf angle and scale refinement loop ;

; inspect Metcalf result for  trimboxim2 of the pair per showex for zoom-in
; (this may replace blink per Metcalf step)
if (checkmetcalf eq 1) then begin
  print,' ----- showex final Metcalf result trimboxim2 pair'
  showex,im1d,im2a,/blink,/plotscatter
endif

; update scales to final values (@@@@@ demo = copy for pxscale12)
px2d=px2*px2asym          ; refresh 2D version to newest Metcalfed px2
pxratio=float(px1)/px2    ; refresh to newest Metcalfed px2, >1 for 1>2
pxrat2d=float(px1)/px2d 
pxscale1=[1.0,1.0]
pxscale2=[1.0,1.0]
if (pxratio ge 1.0) then begin ; px1 > px2 (standard SDO > SST)
  pxscale1=[pxratio,pxratio]   ; pxscale2 remains [1,1]
  if (applypx2asym eq 1) then pxscale2=px2asym
endif
if (pxratio lt 1.0) then begin ; px1 < px2 (non-standard CHROMIS > CRISP)
  if (applypx2asym eq 0) then pxscale2=1./[pxratio,pxratio] $
  else pxscale2=1./pxrat2d     ; pxscale1 remains [1,1]
endif

; get im2 muck sizes in finest px (assume size didn't change in last)
nxmuckfor=nx2a
nymuckfor=ny2a

; for rev include rotated corners (used in sdo_stx_findalignfiles)
rot1=rotatevec([nxmuckfor,nymuckfor],angle)
rot2=rotatevec([nxmuckfor,-nymuckfor],angle)
nxmuckrev=fix(max([abs(rot1[0]),abs(rot2[0])])+0.5)+1
nymuckrev=fix(max([abs(rot1[1]),abs(rot2[1])])+0.5)+1

; find output nx, ny im2 
nxfor=fix(nx2*mean(pxscale2)+0.5)
nyfor=fix(ny2*mean(pxscale2)+0.5)

;; ; check
;; print,' ===== nxfor, nyfor ='+trimd(nxfor)+' '+trimd(nyfor)+$
;;       '  nxmuckfor, nyxmuckfor ='+trimd(nxmuckfor)+' '+trimd(nymuckfor)

; for rev include rotated corners
rot1=rotatevec([nxfor,nyfor],angle)
rot2=rotatevec([nxfor,-nyfor],angle)
nxrev=fix(max([abs(rot1[0]),abs(rot2[0])])+0.5)+1  ; +1 = have corners inside
nyrev=fix(max([abs(rot1[1]),abs(rot2[1])])+0.5)+1

;; ; shifts in finest px after rotate im1; undo asymmetry of rotshiftiter 
if (applypx2asym eq 0) then shiftfor=rotshiftiter else $
  shiftfor=rotshiftiter/px2asym
shiftrev=-rotatevec(shiftfor,-angle)

; correct shifts centers trimbox to original im2 frame 
; @@@@@ demo = emulate content for application to field outside full im2
; Mar 12 2022: usage when for what?
;              needs trimbox=trimboxim2 and nx2,ny2 = dims orig im2
if (trimbox[0] ne -1) then begin
;  trimbox offset in original im2 pixels (?? not changed in forward)
  trimcorfor=[trimbox[0],trimbox[1]]-$
    [(nx2-(trimbox[2]+1-trimbox[0]))/2.,(ny2-(trimbox[3]+1-trimbox[1]))/2.]
  trimcorfor=trimcorfor*pxscale2
  trimcorrev=-rotatevec(trimcorfor,-angle)
  shiftfor=shiftfor+trimcorfor
  shiftrev=shiftrev+trimcorrev
  ;; ; check
  ;; print,' -----'+$
  ;;   ' trimcorfor ='+trimd(trimcorfor,2)+$
  ;;   ' trimcorrev ='+trimd(trimcorrev,2)+$
  ;;   '    shiftfor ='+trimd(shiftfor,2)+$
  ;;   '    shiftrev ='+trimd(shiftrev,2)
endif 

; define central image part as trimbox for rinse and shift measurements
trimrforcenter=fix([nxfor/4.,nyfor/4.,nyfor*3./4,nyfor*3./4])
trimrevcenter=fix([nxrev/4.,nyrev/4.,nyrev*3./4,nyrev*3./4])

; ----- rinse wash = final shift improvements per central trimbox
; Aug  9 2022: no longer finalrinse > 1 calling findimshift_tiled.pro
;   because that has no trimbox option (is only for full-field/full-disk).
;   I ran many even/odd/odditize/evenize tests to undo the need for this
;   final shift-align (emrecord SST 2020-08-13-AR-5D) but did not succeed.
; Aug 11 2022: maybe due to angle-shift problems now solved by
;   unrolling rotate-scale-shift in reformimages.pro but let be and
;   even/odd px leftover of order 0.5 px be removed by rinse, now default.

;  find forward rinse shift: reform as demo but muck as above 
if (norinse eq 0) then begin
  reformimage,im1,im1for,congridfactor=pxscale1,$
    shift=shiftfor,rotate=angle,$
    nxlarge=nxfor,nylarge=nyfor,missingvalue=-1,$
    cutcentralx=nxfor,cutcentraly=nyfor,splinip=1,$
    histopttop=histopim1,smear=smear1,$
    muckdark=muckdarkim1,muckbright=muckbrightim1,flatten=flatten1
  reformimage,im2,im2for,congridfactor=pxscale2,$
    nxlarge=nxfor,nylarge=nyfor,missingvalue=-1,$ 
    cutcentralx=nxfor,cutcentraly=nyfor,splinip=1,$
    histopttop=histopim2,smear=smear2,$
    muckdark=muckdarkim2,muckbright=muckbrightim2,flatten=flatten2
  rinsefor=findimshift_rr(im2for,im1for,/subpix,$
                          filter=10,trimbox=trimforcenter,$
                          croptriangles=0,cropborders=1)

; find reverse rinse shift: reform as demo but muck as above 
  reformimage,im1,im1rev,congridfactor=pxscale1,$
    shift=-shiftrev,$ 
    nxlarge=nxrev,nylarge=nyrev,missingvalue=-1,$ 
    cutcentralx=nxrev,cutcentraly=nyrev,splinip=1,$
    histopttop=histopim1,smear=smear1,$ 
    muckdark=muckdarkim1,muckbright=muckbrightim1,flatten=flatten1
  reformimage,im2,im2rev,congridfactor=pxscale2,$
    rotate=-angle,$
    nxlarge=nxrev,nylarge=nyrev,missingvalue=-1,$
    cutcentralx=nxrev,cutcentraly=nyrev,splinip=1,$
    histopttop=histopim2,smear=smear2,$
    muckdark=muckdarkim2,muckbright=muckbrightim2,flatten=flatten2
  rinserev=findimshift_rr(im2rev,im1rev,/subpix,$
                          filter=10,trimbox=trimrevcenter,$
                          croptriangles=0,cropborders=1)

; update shifts with rinse corrections - tricky, took all Aug 10 2022
;; Aug 10 version happy bedtime before unrolling shift_img_rr.pro had:
;; ;; shiftfor=shiftfor+rotatevec(rinsefor,-angle) 
  shiftfor=shiftfor+rinsefor
  shiftrev=shiftrev-rinserev
  inshift12=inshift12+rotatevec(rinsefor/pxscale1,-angle)
;RR small remaining forward offset for case B (im1 smaller px)

; rinse wash done
  if (verbose ne 0) then $
    print,' ----- rinse corrections:  rinsefor ='+trimd(rinsefor)+$
    '  rinserev ='+trimd(rinserev)

endif ; end final rinse shifting

; ##### wash done - in 500 arduous lines of IDL

; ----- optionally show results

if (show ne 0) then begin

; start showex blink nonmucked forward (@@@@@ demo = copy content) 
  reformimage,im1,im1for,congridfactor=pxscale1,$
    shift=shiftfor,rotate=angle,$
    nxlarge=nxfor,nylarge=nyfor,missingvalue=-1,$
    cutcentralx=nxfor,cutcentraly=nyfor,splinip=1
  if (applypx2asym eq 0 and pxratio gt 1.0) then im2for=im2 else $
    reformimage,im2,im2for,congridfactor=pxscale2,$
    nxlarge=nxfor,nylarge=nyfor,missingvalue=-1,$ 
    cutcentralx=nxfor,cutcentraly=nyfor,splinip=1 
  if (show eq 1 or show eq 3) then showex,im1for,im2for,/blink
  
; start showex blink nonmucked reverse (@@@@@ demo = copy content) 
;   reform im2 with latest values but no shift; instead shift im1
;   to center the rotated im2rev within a larger im1 field
  reformimage,im1,im1rev,congridfactor=pxscale1,$
    shift=-shiftrev,$ 
    nxlarge=nxrev,nylarge=nyrev,missingvalue=-1,$ 
    cutcentralx=nxrev,cutcentraly=nyrev,splinip=1
  reformimage,im2,im2rev,congridfactor=pxscale2,$
    rotate=-angle,$
    nxlarge=nxrev,nylarge=nyrev,missingvalue=-1,$
    cutcentralx=nxrev,cutcentraly=nyrev,splinip=1
  if (show eq 1 or show eq 3) then showex,im1rev,im2rev,/blink
endif

; optionally measure and print final center shifts (ideally zero,
; may be way off if mucking is necessary since these are nonmucked)
if (show ge 2) then begin
  if (verbose ne 0) then $
    print,' ----- patience: measuring results center shifts in result px'
  resultforshift=findimshift_rr(im2for,im1for,/subpix,$
                                filter=10,trimbox=trimforcenter)
  resultrevshift=findimshift_rr(im2rev,im1rev,/subpix,$
                                filter=10,trimbox=trimrevcenter)
  print,' ----- measured final shifts (near zero? NB: nonmucked centers): '+$
    'forshift:'+trimd(resultforshift)+$
    ' revshift:'+trimd(resultrevshift)
endif

; optionally print final results 
if (verbose ne 0 and skipmetcalf eq 0) then begin
  print,' ----- results:'+$
    '  nmet ='+trimd(nmet)+$
    '  metqual = '+trim(metqual,'(E8.2)')+$
    '  angle ='+trimd(angle,2)+$
    '  px2 ='+trimd(px2,4)+$
    '  px2asym ='+trimd(px2asym,4)+$
    '  shiftfor ='+trimd(shiftfor,1)+$
    '  shiftrev ='+trimd(shiftrev,1)+$
    '  nxfor,nyfor ='+trimd(nxfor)+' '+trimd(nyfor)+$
    '  nxrev,nyrev ='+trimd(nxrev)+' '+trimd(nyrev)+$
    '  inshift12 ='+trimd(inshift12,1)
endif

; if verbose print elapsed time
if (verbose ne 0) then begin 
  timedone=systime(1)
  timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
  print,' ----- findalignimages took '+timelaps+' minutes'
endif

ERROREND:

end ; of program



; ======================== test per IDLWAVE H-c =====================

print,' !!!!! run in new IDL session for non-sticky keywords'

; Below is a much more stringent test then when I developed this program
; putting SDO to SST because the SDO pixels are 10x larger.
; This test is a rich SST scene that gets deformed for self-align.
; You can run this test yourself since the infile is available, see
;   https://robrutten.nl/rrweb/sdo-demo/instruction.html

cd,'/media/rutten/RRHOME/alldata/SST/2014-06-24-solnet/sst'
infile='width_kr_6563_0.9_cut.fits'  ; Kevinesq detailed scene
incube=readfits(infile)          ; int [932, 920, 53] 
im1=incube[*,*,0:0]              ; full cube use in test selfaligncube_v2.pro
missing=avg(im1)                 ; border color for showex greyscaling

; startup
px1=1.
maxmetcalf=5
minmetqual=1E-5

; 200-border cutout im2 to admit scale and rotation within im1
xrange=[200,732]
yrange=[200,720]

; deform second image considerably
shifts=[6,3]  ; in im2 px
rotate=3      ; in degrees
rotate=60     ; was needed to fix the rinse shifts 

; ##### deform px scales: choose a case below by in/uncommenting

; ##### Case A: no px2asym, px2 < px1  = classic SDO > SST case
scale=[1.05,1.05]  
applypx2asym=0

;; ; ##### Case B: no px2asym, px2 > px1 = e.g., SDO > GONG case
;; scale=[0.95,0.95]  
;; applypx2asym=0
;; ;RR still small scale problems

;; ; ##### Case C: px2asym with mean px2 < px1 = SDO > DST/IBIS case
;; scale=[1.02,1.08] ; asymmetry around 1.05
;; applypx2asym=1

;; ; ##### Case D: px2asym with mean px2 > px1 (not yet encountered)
;; scale=[0.98,0.92] ; asymmetry around 0.95
;; applypx2asym=1 

; deform image 
reformimage,im1,im2,xrange=xrange,yrange=yrange,$
  shift=shifts,rotate=rotate,scale=scale,$
  missingvalue=missing,splinip=1

; select trimbox im2 for smaller field, avoid activity;  smaller=faster
;; showex,im2
;; STOP
trimboxim2=[100,100,350,350] ; classic, quiet near center
trimboxim2=[25,298,240,444] ; Mar 12 2022: upper-left corner
trimboxim2=-1 ; im2 full cutout frame, auto rotation triangle trimming

; set close startoff parameters 
px2=1.02/avg(scale)   ; 2% off
angle=rotate-0.5      ; 0.5 deg off

;; ; muck parameters: none here except trimbox since scenes are identical
flatten=0.
smearim1=-10
smearim2=-10

; output choices
blink=0
checkmetcalf=0
show=3
verbose=1

; run the program
findalignimages,im1,im2,px1,px2,angle,px2asym,$
  shiftfor,shiftrev,nxfor,nyfor,nxrev,nyrev,$    
  nxmuckfor=nxmuckfor,nymuckfor=nymuckfor,$
  nxmuckrev=nxmuckrev,nymuckrev=nymuckrev,$
  skipmetcalf=skipmetcalf,maxmetcalf=maxmetcalf,minmetqual=minmetqual,$
  applypx2asym=applypx2asym,$
  trimboxim2=trimboxim2,inshift12=inshift12,$
  smearim1=smearim1,smearim2=smearim2,$
  histopim1=histopim1,histopim2=histopim2,$
  muckdarkim1=muckdarkim1,muckdarkim2=muckdarkim2,$
  muckbrightim1=muckbrightim1,muckbrightim2=muckbrightim2,$
  flatten=flatten,flatim1=flatim1,flatim2=flatim2,$
  metqual=metqual,nmet=nmet,norinse=norinse,$
  checkmetcalf=checkmetcalf,blink=blink,show=show,verbose=verbose

end


