; file: diffdesolrotate.pro
; init: Jul 12 2020  Rob Rutten  Deil
; last: Nov 17 2020  Rob Rutten  Deil

;+
function diffdesolrotate,image,arcsecpx,xcenim,ycenim,timage,tother,$
  dedesolrot=dedesolrot,orientang=orientang

 ;  differentially derotate large solar image bacǩ over time interval
 ;  (i.e., differential solar rotation over time, not image rotation)
 ;   
 ; DESCRIPTION:
 ;  - finds differential solar derotations on a coarse image-covering grid
 ;  - applies these per image pixel after congridding to image size
 ;  - optionally dedesolrots already field-center-derotated image
 ;    (RR-pipe-made SDO fitscube, correlation-tracking imagery as SST)
 ;  - only large field and long time interval have effect
 ;
 ; CALLED BY:
 ;  - /diffdesolrot option in sdo_getdata_rr.pro etc., reformcubefile.pro
 ; 
 ; INPUTS:
 ;   image: 2D array
 ;   arcsecpx: in arcsec (SDO "RR level2": 0.6)
 ;   xcenim,ycenim: solar (X,Y) at image center (arcsec)
 ;   timage = anytim datetime string for this image
 ;   tother = anytim datetime string for start of interval
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   dedesolrot = 1/0: first de-derotate when field-center tracked
 ;   orientang: clockwise angle of image from solar (X,Y) in deg (def 0)
 ;
 ; OUTPUT:
 ;   differentially derotated image
 ;
 ; METHOD:
 ;  - rotate to (X,Y) orientation when image isn't (keyword orientang)
 ;  - define differential de-rotation values on coarse grid covering image
 ;  - congrid grid to actual image dimensions and shift each pixel
 ;  - optionally dedesolrot if image was already derotated tracking center
 ;  - rotate back if image orientation wasn't (X,Y)
 ;
 ; EXAMPLE: 
 ;   all SDO target/cubes produced my way having center-field derotation
 ;   using reformcubefile.pro per doallfiles.pro:
 ;     doallfiles,'.','.','.fits','_diff.fits','reformcubefile',$
 ;     /diffdesolrot,/dedesolrot
 ;
 ; PROBLEMS:
 ;   extreme limb and off-limb suffer big-pixelation and remain unmodified
 ;
 ; HISTORY:
 ;   Jul 12 2020: Rob Rutten start   
;-

; answer no-parameter query 
if (n_params(0) lt 6) then begin
  sp,diffdesolrotate
  return,-1
endif

; keyword default
if (n_elements(dedesolrot) eq 0) then dedesolrot=0
if (n_elements(orientang) eq 0) then orientang=0

; do nothing when timage=tother (as first image of RR SDO cubefile)
if (anytim2tai(timage) eq anytim2tai(tother)) then begin
  outim=image
  goto, DONE
endif

; image dimensions
szim=size(image)
nx=szim[1]
ny=szim[2]

; unstick image; rotate counterclockwise to (X,Y) (terrestrial North up) 
if (orientang ne 0) then reformimage,image,imt1,rotate=orientang $
else imt1=image

; unstick xcenim, ycenim; find xcen,ycen non-tracked image at tother
; (0 = other, 1 = image because usually t0 before t1)
xcent1=xcenim 
ycent1=ycenim
t0pos=rot_xy(xcent1,ycent1,tstart=timage,tend=tother)
t0pos=reform(t0pos)
xcent0=t0pos[0]
ycent0=t0pos[1]
shift10=[xcent1,ycent1]-t0pos

; shiftdedesolrot in arcsec when /dedesolrot = image was center-tracked 
if (dedesolrot eq 1) then shiftdedesolrot=shift10 else shiftdedesolrot=[0,0]

;; ; check disk-center value
;; print,' ===== center shift in px',shift10/arcsecpx

; set coarse-grid px size (RR choice, rectangular "big" pixels)
xbigpx=min([50./arcsecpx,nx/3])   ; 60 arcsec = 100 SDO px
ybigpx=min([100./arcsecpx,ny/3])  ; y shifts 10x smaller

; define coarse-grid "big-pixel images" to fill with shifts
nxrotim=fix(nx/xbigpx)
nyrotim=fix(ny/ybigpx)
shxim=fltarr(nxrotim,nyrotim)
shyim=shxim
for iy=0,nyrotim-1 do begin
  for ix=0,nxrotim-1 do begin

; get solar (X,Y) of center of big rotim in arcsec
      xsolbig=xcent1+arcsecpx*((ix-nxrotim/2.+0.5)*xbigpx)
      ysolbig=ycent1+arcsecpx*((iy-nyrotim/2.+0.5)*ybigpx)

; get rotation shifts for big px
    newpos=rot_xy(xsolbig,ysolbig,tstart=timage,tend=tother)
    newpos=reform(newpos)
    if (min(newpos) lt -3000) then goto,NEXTBIGPX ; off limb => 0

; now fill shift images with shift values in float px units
    rotshift=(newpos-[xsolbig,ysolbig]+shiftdedesolrot)/arcsecpx
    shxim[ix,iy]=rotshift[0]
    shyim[ix,iy]=rotshift[1]

;; ; check at center
;; if (ix eq nxrotim/2 and iy eq nyrotim/2) then STOP
    
NEXTBIGPX:    
  endfor ; end ix loop
endfor   ; end iy loop

;; ; check (NB: showex assumes square pixels)
;; showex,shxim,shyim
;; STOP

; congrid to original image grid and round to integers
shximfull=fix(congrid(shxim,/center,nx,ny,1,cubic=-0.5)+0.5)
shyimfull=fix(congrid(shyim,/center,nx,ny,1,cubic=-0.5)+0.5)

;; ; check 
;; showex,shximfull,shyimfull
;; STOP

; refill derotim with shifted pixels
derotim=imt1  ; image stays same where shift=0 as off-limb
for iy=0,ny-1 do begin
  for ix=0,nx-1 do begin
    if (ix-shximfull[ix,iy] gt 0 and iy-shyimfull[ix,iy] gt 0) $
      and (ix-shximfull[ix,iy] lt nx and iy-shyimfull[ix,iy] lt ny) then $
        derotim[ix,iy]=imt1[ix-shximfull[ix,iy],iy-shyimfull[ix,iy]]
  endfor ; end ix loop
endfor   ; end iy loop

; rotate result back clockwise if it wasn't (X,Y) oriented
if (orientang ne 0) then reformimage,derotim,outim,rotate=-orientang $
  else outim=derotim

DONE:
return,outim
end


; =============== main for testing per IDLWAVE H-c ======================

; standard RR SDO cubefile with center-tracked images
cd,'/home/rutten/data/SDO/2014-06-14-longer/target/cubes'
cube=readfits('hmicont.fits',head)
starttim=fxpar(head,'starttim')
tother=anytim2tai(starttim) ; back to cube start
nt=fxpar(head,'naxis3')
cadence=fxpar(head,'cadence')
image=cube[*,*,nt-1]
timage=tother+(nt-1)*cadence
imother=cube[*,*,0]
xcenim=fxpar(head,'xcen')  
ycenim=fxpar(head,'ycen')  
orientang=fxpar(head,'crota2')
dedesolrot=1
print,' -----  timage-tother in min =',(nt-1)*cadence/60. 

;; ; aia-prepped level2 image file
;; cd,'/media/rutten/RRHOME/alldata/SDO/2019-11-11-transit/full_long/target'
;; ;; tother=anytim2tai('2019.11.11_18:00')    ;; 4 hr earlier (t1 = 22:00)
;; tother=anytim2tai('2019.11.11_11:00')     ;; 11 hr earlier
;; xcent0=0
;; ycent0=0
;; ; get last level2 image 
;; files=file_search('level2/*_0193.fits*')
;; nfiles=n_elements(files)
;; spawn,'funpack -D '+files[nfiles-1]
;; files=file_search('level2/*_0193.fits') ; uncompressed ones
;; nfiles=n_elements(files) 
;; read_sdo,files[nfiles-1],index,image,/uncomp_delete
;; timage=anytim2tai(index.t_obs) ; 2019-11-11T21:59:29.84Z
;; im=sdo_intscale(index,image,'193',clipmin=-1,clipmax=-1)

;; ; use /tmp images collected by sdo_diskfigs.pro
;; image=readfits('/tmp/sdo_2020.01.01_00:00_1600.fits',index1)
;; imother=readfits('/tmp/sdo_2020.01.01_00:00_1700.fits',index2)
;; image=histo_opt_rr(image,1E-3) ; only for showex below
;; imother=histo_opt_rr(imother,1E-3)
;; timage=fxpar(index2,'t_obs')
;; xcenim=fxpar(index1,'xcen') ; 0 for full-disk
;; ycenim=fxpar(index1,'ycen') ; 0 for full-disk
;; orientang=fxpar(index1,'crota2') ; should be 0
;; tother=fxpar(index1,'t_obs')  ; 34 sec later = 0.1 arcsec disk center
;; ;; tother='2019.12.31_23:00'   ; test canonical 9 arcsec/hr at center
;; dedesolrot=0

; act
derotim=diffdesolrotate(image,0.6,xcenim,ycenim,timage,tother,$
                        dedesolrot=dedesolrot,orientang=orientang)

;; showex,image,derotim
       ; zoom in to eg, center and measure pixel shift A-B
showex,imother,derotim,/blink

end
