; file: deforepix.pro
; init: Jul 18 2020  Rob Rutten  Deil
; last: Nov 17 2020  Rob Rutten  Deil

;+
function deforepix,image,xcen,ycen,timage,$
  orientang=orientang,reverse=reverse

 ;  undo limbward foreshortening by adding pixels in that direction
 ;  (square angular pixels sample 1/mu more surface in limb direction;
 ;   they get shrunk to geometrically square as viewed from above)
 ;
 ; INPUTS:
 ;   image: 2D image array
 ;   xcen, ycen: solar (X,Y) of image center (arcsec)
 ;   timage: anytim date-time string for image (for sun radius arcsec)
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   orientang: clockwise orientation angle from solar (X,Y) in deg (def 0)
 ;   reverse = 1/0: (re-)foreshorten = expand square pixels limbward 
 ;
 ; OUTPUT:
 ;   same image with shrunk pixels = more in the limb direction;
 ;   center-field pixels become geometrically square, with disk-center size
 ;
 ; METHOD:
 ;  - rotate image to (X,Y) orientation when image isn't (keyword orientang)
 ;  - de- (or re-) foreshorten to viewpoint "above" image center
 ;  - rotate back
 ;
 ; EXAMPLE:
 ;   called as /deforepix option in reformcubefile.pro permitting use as:
 ;     doallfiles,'.','.','.fits','_defor.fits','reformcubefile',$
 ;     /deforepix,orientang=66.6
 ;
 ; NOTE:
 ; - only for small fields since image center value used for whole field
 ; - doesn't solve shadowing obscurations by assuming flat image plane
 ;   e.g., granules appaer lengthened limbward from blocking lines behind)
 ; - along (X,Y) diagonals the scene remains the same, with shrunk pixels
 ;
 ; HISTORY:
 ;   Jul 23 2020: RR start   
;-

; answer no-parameter query 
if (n_params(0) lt 4) then begin
  sp,deforepix
  return,-1
endif

; keyword default
if (n_elements(orientang) eq 0) then orientang=0
if (n_elements(reverse) eq 0) then reverse=0

; check not disk center
if (xcen eq 0 and ycen eq 0) then begin
  print,' ----- no deforepix at disk center'
  return,image
endif

; rotate image to (X,Y) when needed
if (orientang ne 0) then reformimage,image,im1,rotate=orientang $
else im1=image

; image dimensions
szim=size(im1)
nx=szim[1]
ny=szim[2]

; radius in arcsec
utdatetime=anytim2utc(timage,/ccds) ; Slater needs proper UT format 
rsun=get_rb0p(utdatetime,/radius,/quiet)

; radial size increase scales with 1/mu, reverse decrease scales with mu
vec=sqrt(float(xcen)^2+float(ycen)^2) ; also arcsec
if (vec lt 0.99*rsun) then mu=cos(asin(vec/rsun)) else begin
  print,' ----- deforepix.pro does not act beyond 0.99 R_sun'
  return,image
endelse
if (reverse eq 0) then reldelvec=(vec/mu-vec)/vec else $
  reldelvec=(vec-vec*mu)/vec

; components for nx and ny field sizes in (X,Y) orientation
posang=atan(abs(float(xcen))/abs(float(ycen)))
;RR ?? no quadrant business needed?
if (reverse eq 0) then begin
  nxnew=fix(nx+sin(posang)*nx*reldelvec+0.5)
  nynew=fix(ny+cos(posang)*ny*reldelvec+0.5)
endif else begin
  nxnew=fix(nx-sin(posang)*nx*reldelvec+0.5)
  nynew=fix(ny-cos(posang)*ny*reldelvec+0.5)
endelse 

; congrid image to (de-)stretched version
imnew=congrid(im1,/center,nxnew,nynew,1,cubic=-0.5)

; rotate back over orientang
if (orientang ne 0) then reformimage,imnew,imout,rotate=-orientang $
else imout=imnew

return,imout
end


; =============== main for testing per IDLWAVE H-c ======================

cd,'/media/rutten/RRHOME/alldata/SDO/2019-11-11-transit/full_long/target'

; get last level2 image 
files=file_search('level2/*_cont.fits*')
nfiles=n_elements(files)
spawn,'funpack -D '+files[nfiles-1]
files=file_search('level2/*_cont.fits') ; uncompressed ones
nfiles=n_elements(files) 
read_sdo,files[nfiles-1],indexfull,imfull,/uncomp_delete
timage=anytim2tai(indexfull.t_obs) ; 2019-11-11T21:59:29.84Z
xcenfull=indexfull.xcen
ycenfull=indexfull.ycen

; ##### parameters: set (X,Y) for cut center and cut size in arcsec
xcencut=0
ycencut=-700
cutsize=50
orientang=0

; this is AIA
arcsecpx=0.6
xcenfullpx=xcenfull/arcsecpx
ycenfullpx=ycenfull/arcsecpx
xcencutpx=xcencut/arcsecpx
ycencutpx=ycencut/arcsecpx

; cut the piece
fullsize=size(imfull)
nxfull=fullsize[1]
nyfull=fullsize[2]
imcut=imfull[nxfull/2-xcenfullpx+xcencutpx-cutsize/2:$
             nxfull/2-xcenfullpx+xcencutpx+cutsize/2,$
             nyfull/2-ycenfullpx+ycencutpx-cutsize/2:$
             nyfull/2-ycenfullpx+ycencutpx+cutsize/2]

; show the cut
imfull[nxfull/2-xcenfullpx+xcencutpx-cutsize/2:$
             nxfull/2-xcenfullpx+xcencutpx+cutsize/2,$
             nyfull/2-ycenfullpx+ycencutpx-cutsize/2:$
             nyfull/2-ycenfullpx+ycencutpx+cutsize/2]=0

; act on this piece
imdefor=deforepix(imcut,xcencut,ycencut,timage,orientang=orientang)

; and back as check
imback=deforepix(imdefor,xcencut,ycencut,timage,$
                    orientang=orientang,reverse=1)

; print dimensions
nxcut=cutsize+1
nycut=cutsize+1
szdef=size(imdefor)
print,' ----- nxcut,nycut,nxdefor,nydefor: ',nxcut,nycut,szdef[1],szdef[2]

; show 
wdelall
sv,imfull
sv,imcut
sv,imdefor
sv,imback

; zooms
;; showex,imdefor
;; showex,imcut,imback

end
