; file: addfitscube2crispex.pro
; init: Feb 11 2015  Rob Rutten  Deil
; last: Feb 12 2015  Rob Rutten  Deil

;+

pro addfitscube2crispex,fitscube,crispinfile,crispinwavs,crispoutfile,$
  bytscale=bytscale,fitslabel=fitslabel

 ; NAME:
 ;   addfitscube2crispex
 ; PURPOSE:
 ;   add a fitscube to a "classical-format La Palma" multi-wav crispex file
 ; CALL:
 ;   see above
 ; INPUTS:
 ;   fitscube = string 'path/filename' of input cube file
 ;   crispinfile = string 'path/filenam' of crispex input file
 ;   crispinavs = nr of wavelengths (channels) in crispinfile
 ;   crispoutfile = string 'path/filename' of extended crispex output file
 ; OPTIONAL KEYWOD PARAMETERS:
 ;   bytscale=0: if crispinfile is a bcube: bytscale each image
 ;   bytscale=1: idem: bytscale with min and max of first image as cutoffs
 ;   bytscale=2: idem: bytscale with min and max of whole fitscube as cutoffs
 ;   fitslabel = string with identifying overplot label (if bytscaled)
 ; RESTRICTIONS:
 ;   fitscube must have the same [nx,ny,nt] size as one crispex channel   
 ;   (but datatypes may differ)
 ; HISTORY:
 ;   Feb 11 2015 RR: start
;-

; answer no-parameter query
if (n_params() lt 4) then begin
  print,'  addfitscube2crispex,fitscube,crispinfile,crispinwavs,$'
  print,'     crispoutfile,bytscale=bytscale,fitslabel=fitslabel' 
  return
endif

; print pacifier
print,' ==== addfitscube2crispex starts  #### takes long'

; defaults for keywords
if (n_elements(fitslabel) eq 0) then fitslabel=''
if (n_elements(bytscale) eq 0) then bytscale=0

; fitscube: read header info
header_in=headfits(fitscube)
header_insize=(1+fix(n_elements(header_in)/36.))*2880
bitpix_in=fxpar(header_in,'bitpix')
nx_in=fxpar(header_in,'naxis1') 
ny_in=fxpar(header_in,'naxis2') 
nt_in=fxpar(header_in,'naxis3') 
if (nx_in eq 0 or ny_in eq 0 or nt_in eq 0) then begin
  print,' ###### no proper fitscube since no nx, ny, nt in header '+infile
  return
endif
nt_cout=nt_in

; open fitscube for assoc
bigendian=1 ;RR alas, fits files are bigendian; intel machines are little
get_lun, unit_in
if (bigendian) then openr,unit_in,fitscube,/swap_if_little_endian $
else openr,unit_in,fitscube
if (bitpix_in eq -32) then $
  assoc_in=assoc(unit_in,fltarr(nx_in,ny_in),header_insize)
if (bitpix_in eq 16) then $
  assoc_in=assoc(unit_in,intarr(nx_in,ny_in),header_insize)
if (bitpix_in eq 8) then $
  assoc_in=assoc(unit_in,bytarr(nx_in,ny_in),header_insize)

; set bytscale extrema for fitscube
if (bytscale ne 0) then begin
  min_in=1.e10
  max_in=-1.e10
  if (bytscale eq 1) then ntminmax=1
  if (bytscale eq 2) then ntminmax=nt_in
  for it=0,ntminmax-1 do begin
    image=assoc_in[it]
    if (min(image) lt min_in) then min_in=min(image)
    if (max(image) gt max_in) then max_in=max(image)
  endfor
; maintain zero value for magnetograms and Doppler maps
  if (min_in lt -0.5*max_in) then begin
    minzero=min([min_in,-max_in])
    maxzero=max([max_in,-min_in])
    limit=max([abs(minzero),maxzero])
    max_in=limit
    min_in=-limit
  endif
endif

; crispinfile: read header info
crispex_read_header,crispinfile,header=header_cin,$
  stokes=stokes_cin,ns=ns_cin,datatype=data_cin, $
  dims=dims_cin,nx=nx_cin,ny=ny_cin,nt=nt_cin,endian=endian_cin
nw_cin=crispinwavs
nt_cin=fix(0.1+nt_cin/nw_cin) ;RR real nt, accommodate small mistake
nt_cin=nt_cin/ns_cin          ;RR real nt in case of 4 Stokes samplings
nw_cout=nw_cin+1              ; add one wav

; check cube equality
if (nx_cin ne nx_in or ny_cin ne ny_in or nt_cin ne nt_in) then begin
  print,' ##### FATAL ERROR: dimensions unequal'
  print,'       fitscube     nx,ny,nt = '+ntostr(nx_in,ny_in.nt_in)
  print,'       crispinfile  nx,ny,nt = '+ntostr(nx_cin,ny_cin,nt_cin)  
  return
endif

; open crispinfile for assoc 
if (data_cin eq 1) then arr_cin=bytarr(nx_cin,ny_cin)
if (data_cin eq 2) then arr_cin=intarr(nx_cin,ny_cin)
if (data_cin eq 4) then arr_cin=fltarr(nx_cin,ny_cin)
get_lun, unit_cin
openr,unit_cin,crispinfile
assoc_cin=assoc(unit_cin,arr_cin,512)

; define crispoutfile header
im0=assoc_cin[0]  ; first crispinfile image sets size and datatype
header_cout=make_lp_header(im0,nt=nt_cout*nw_cout)
header_cout=str_replace(header_cout,' datatype',$
                        'stokes=[I], ns=1 :  datatype')

; open crispoutfile for assoc (same image size and type as crispinfile)
bigendian=0   ;RR nonfits SST files are little endian (Oslo, linux) 
get_lun, unit_cout
if (bigendian) then $
  openw,unit_cout,crispoutfile,/swap_if_little_endian $
else openw,unit_cout,crispoutfile
assoc_cout=assoc(unit_cout,arr_cin,512)

; write header_cout
rec=assoc(unit_cout, bytarr(512))
rec[0]=byte(header_cout)

; set fitslabel overlay
if (fitslabel ne '') then begin
  blank=bytarr(nx_in,ny_in)
  while !d.window ne -1 do wdelete,!d.window 
  window,xsize=nx_in,ysize=ny_in,/pixmap,retain=2
  !p.font=-1
  tv,blank
  xyouts,ny_in/30,ny_in/30,fitslabel,color=255,$
    /device,charsize=sqrt(ny_in)/10.,charthick=sqrt(ny_in)/10.
  imdiagname=tvrd()
endif

; finally the big loop over it and iw
for it=0,nt_in-1 do begin
  for iw=0,nw_cin-1 do begin
    image_cin=assoc_cin[it*nw_cin+iw]
    assoc_cout[it*nw_cout+iw]=image_cin
  endfor
  image_in=assoc_in[it]
  if (data_cin eq 1) then begin 
    if (bytscale eq 0) then image_in=bytscl(image_in) $
    else image_in=bytscl(image_in,min=min_in,max=max_in)
  endif
  if (fitslabel ne '') then image_in[where(imdiagname gt 1)]=255
  assoc_cout[it*nw_cout+nw_cout-1]=image_in
endfor

; print success
print,' ==== wrote file '+crispoutfile

; close files
free_lun,unit_in
free_lun,unit_cin
free_lun,unit_cout

end 

; ==================== test per IDLWAVE super C ======================

path='/home/rutten/data/SST/2011-05-07-test/'
fitscube=path+'sstcubes/fecont.fits'
fitslabel='SST FeI cont'
crispinfile=path+'sdo2sst/sdo_crispex-7.bcube'
crispinwavs=7
crispoutfile=path+'sdo2sst/sdo_crispex-8.bcube'
addfitscube2crispex,fitscube,crispinfile,crispinwavs,crispoutfile,$
  bytscale=1,fitslabel=fitslabel

; inspect
crispex,crispoutfile,single_cube=8

end


