; file: shiftfitscube.pro
; init: Nov 24 2013  Rob Rutten  Sac Peak
; last: Sep  4 2015  Rob Rutten  Freiburg
; note: superseded by reformfitscube.pro

pro shiftfitscube,infile,outfile,shiftsfile=shiftsfile

;+
 ; NAME:
 ;   shiftfitscube.pro 
 ; PURPOSE:
 ;   shift a cube in a fitsfile
 ; CALL:
 ;   shiftfitscube,infile,outfile,shiftsfile
 ; INPUTS:
 ;   infile: string with path/filename for the input fitsfile
 ;   outfile: string with path/filename for the output fitsfile
 ;   shiftsfile=shiftsfile: ASCII file with shifts as 2 values or 2 columns
 ; OUTPUTS:
 ;   shifted cube of the same type as the input cube in the outfile   
 ; METHOD:
 ;   uses assoc to permit larger cubes than memory size
 ;   making shiftsfile a keyword parameter permits call by doallfiles.pro
 ; HISTORY:
 ;   Nov 24 2013 RR: start
 ;   Jan 17 2014 RR: added single values and this info header
 ;   Jul 28 2014 RR: made suited for call by doallfiles.pro
;-

; set endian
bigendian=1

; get cube geometry and file datatype from the fits header
inheader=headfits_rr(infile)
inheadersize=(1+fix(n_elements(inheader)/36.))*2880
nx=fxpar(inheader,'naxis1') 
ny=fxpar(inheader,'naxis2') 
nt=fxpar(inheader,'naxis3') 
bitpix=fxpar(inheader,'bitpix')

; open input file for assoc
get_lun, unit_in
if (bigendian) then openr,unit_in,infile,/swap_if_little_endian $
else openr,unit_in,infile
if (bitpix eq -32) then inassoc=assoc(unit_in,fltarr(nx,ny),inheadersize)
if (bitpix eq 16) then inassoc=assoc(unit_in,intarr(nx,ny),inheadersize)
if (bitpix eq 8) then inassoc=assoc(unit_in,bytarr(nx,ny),inheadersize)

; copy input header to output header
outheader=inheader
outheadersize=inheadersize

; open output file for assoc, write header
get_lun, unit_out
if (bigendian) then openw,unit_out,outfile,/swap_if_little_endian $
else openw,unit_out,outfile
if (bitpix eq -32) then outassoc=assoc(unit_out,fltarr(nx,ny),outheadersize)
if (bitpix eq 16) then outassoc=assoc(unit_out,intarr(nx,ny),outheadersize)
if (bitpix eq 8) then outassoc=assoc(unit_out,bytarr(nx,ny),outheadersize)
if (outheadersize ne 0) then begin
  rec=assoc(unit_out, bytarr(outheadersize))
  rec[0]=byte(outheader)
endif

; read the shifts
readcol,shiftsfile,shiftx,shifty

; check whether single values or values per time step
nshifts=n_elements(shiftx)
if (nshifts eq 1) then begin
  shiftxarr=fltarr(nt)+shiftx[0]
  shiftyarr=fltarr(nt)+shifty[0]
  shiftx=shiftxarr
  shifty=shiftyarr
endif

; check array sizes
nshifts=n_elements(shiftx)
if (nshifts ne nt) then begin
  print,' ---  nt cube = '+ntostr(nt)+$
     ' not equal to nr shifts = '+$
    ntostr(nshifts),'; reset to smalllest'
  if (nt gt nshifts) then nt=nshifts
endif

; do the shifting
for it=0,nt-1 do begin
  imin=inassoc[it]
  imout=shift_img(imin,[shiftx[it],shifty[it]])
  outassoc[it]=imout
endfor

; free the input and output files
free_lun,unit_in,unit_out

end
