; file: selfalignfitscube.pro
; init: Oct 26 2020  Rob Rutten  Deil from selfaligncube_v2.pro
; last: Jan 29 2021  Rob Rutten  Deil

;+
pro selfalignfitscube,infile,outfile,$
  naverref=naverref,itbest=itbest,tselect=tselect,$
  smear=smear,flatten=flatten,applypx2asym=applypx2asym,$
  alignshift=alignshift,alignmetcalf=alignmetcalf,actsticky=actsticky,$
  maxmetcalf=maxmetcalf,minmetqual=minmetqual,cutmetqual=cutmetqual,$
  finalrinse=finalrinse,$
  trimbox=trimbox,transdatafile=transdatafile,plotpath=plotpath,$
  show=show

 ; self-align a fitscube using assoc to permit large files;
 ; either only find and apply shifts per timestep or use
 ; <slow> Metcalf iteration per time step using findalignimages.pro 
 ; 
 ; INPUTS:
 ;   infile: fitscube file (image sequence in classical fits format)
 ;  
 ; OPTIONAL KEYWORD INPUTS:
 ;   naverref: (tricky, much depending on data quality and duration)
 ;      0: successive pairwise alignment (default)
 ;      1: align all to image at itbest
 ;      > 1: align each to average last NN already aligned images before
 ;     -1: align all to first image of sequence (bad at evolution)
 ;     -2: align all to nt/2 mid-sequence image (bad at evolution)
 ;     -3: align all to sequence time-average (smear likely useful)
 ;      < -3: align each to temporal |naverref| box-car time-smoothed 
 ;      NB: runaway likely for naverref = 0, also but less for > 1 and < -3
 ;   itbest: it of reference image, must be present for naverref=1
 ;   tselect: it index array of images to do, others remain unaffected
 ;   smear: smooth mucked align pair by boxcar [px] smearing 
 ;   flatten: subtract boxcar-smeareed from align pair, boxcar size in px
 ;   applypx2asym: as in findalignimages.pro but not to ref image (may be bad)
 ;   alignshift = 0: do not self-align by shifting only 
 ;                1: self-align by shifting only with findimshift_rr.pro
 ;                > 1: by findimshift_tiled.pr with pxtile=alignshift 
 ;   alignmetcalf = 1/0: self-align per Metcalf iteration (SLOW!)
 ;   actsticky = 1/0: start findalignimages with params last good one
 ;   maxmetcalf: max number Metcalf iterations (default 4)
 ;   minmetqual: stop Metcalf iteration at this metqual value (1E-5)
 ;   cutmetqual: for higher metqual repeat previous aligned image (1E-2)
 ;   trimbox: subarea to self-align [xmin,ymin,xmax,ymax] (eg disk center)
 ;   finalrinse: apply shift-together after Metcalf iteration
 ;      0: don't (default)
 ;      1: do using findimshift_rr.pro
 ;      > 1: do using findimshift_tiled.pro with value = pxtile
 ;   transdatafile: filestring writecol results, default '' = none
 ;   plotpath: path/filestart results graphs, default '' = none
 ;   show = 2/0: showex result every time step
 ;
 ; OUTPUTS:
 ;   - outfile: same size and type as infile
 ;   - transdatafile with transform values for application to parallel files
 ;   - figures with shifts(t), (plus scales(t), rotate(t) when Metcalf)
 ;
 ; REMARKS:
 ;   possibly derotate first (for trimbox center), then rotate result
 ;   for shift-align only use selfshiftalignfitscube.pro, much faster
 ; 
 ; HISTORY:
 ;   Oct 26 2020: RR: start = convert selfaligncube_v2.pro to assoc
;-

; answer no-parameter query 
if (n_params(0) lt 2) then begin
  sp,selfalignfitscube
  return  
endif

; defaults for keywords
if (n_elements(smear) eq 0) then smear=0
if (n_elements(flatten) eq 0) then flatten=0
if (n_elements(applypx2asym) eq 0) then applypx2asym=0
if (n_elements(naverref) eq 0) then naverref=0
if (n_elements(itbest) eq 0) then itbest=-1
if (n_elements(tselect) eq 0) then tselect=-1 
if (n_elements(alignshift) eq 0) then alignshift=0
if (n_elements(alignmetcalf) eq 0) then alignmetcalf=0
if (n_elements(actsticky) eq 0) then actsticky=0
if (n_elements(maxmetcalf) eq 0) then maxmetcalf=4    ;NB default
if (n_elements(minmetqual) eq 0) then minmetqual=1E-5 ;NB default
if (n_elements(cutmetqual) eq 0) then cutmetqual=1E-2 ;NB default   
if (n_elements(finalrinse) eq 0) then finalrinse=0
if (n_elements(trimbox) eq 0) then trimbox=-1
if (n_elements(transdatafile) eq 0) then transdatafile=''
if (n_elements(plotpath) eq 0) then plotpath=''
if (n_elements(show) eq 0) then show=0

; check inputs
if (alignshift eq 0  and alignmetcalf eq 0) then begin
  print,' ##### selfalignfitscube abort: no shift nor metcalf align'
  return
endif

if (alignshift gt 0  and alignmetcalf eq 1) then begin
  print,' ##### selfalignfitscubes abort: no shift and metcalf both'
  return
endif

if (naverref eq 1 and itbest eq -1) then begin
  print,' ##### selfalignfitscube abort: naverref=1 but no itbest'
  return
endif

; ----- standard infile, outfile assoc setups (from boxcarfitscube.pro)

; set endian
bigendian=1

; get fitscube geometry and file datatype from the fits header
header=headfits_rr(infile)
headersize=(1+fix(n_elements(header)/36.))*2880
bitpix=fxpar(header,'bitpix')
nx=fxpar(header,'naxis1') 
ny=fxpar(header,'naxis2') 
nt=fxpar(header,'naxis3') 

; check dimensions
if (nx lt 3 or ny lt 3 or nt lt 3) then begin
  print,' ###### infile incorrect nx, ny, nt, no sequence?'+infile
  return
endif

; open input file for assoc
get_lun, unit_in
if (bigendian) then openr,unit_in,infile,/swap_if_little_endian $
else openr,unit_in,infile
if (bitpix eq -32) then $
  inassoc=assoc(unit_in,fltarr(nx,ny),headersize)
if (bitpix eq 16) then $
  inassoc=assoc(unit_in,intarr(nx,ny),headersize)
if (bitpix eq 8) then $
  inassoc=assoc(unit_in,bytarr(nx,ny),headersize)

; check input image size
im0=inassoc[0]
sizeim0=size(im0)
nx_im0=sizeim0[1]
ny_im0=sizeim0[2]
if (nx_im0 ne nx or ny_im0 ne ny) then begin
  print, ' ##### nx or ny in header inequal to first image [nx,ny]' 
  return
endif

; open output file for assoc, write header
get_lun, unit_out
if (bigendian) then openw,unit_out,outfile,/swap_if_little_endian $
else openw,unit_out,outfile
if (bitpix eq -32) then $
  outassoc=assoc(unit_out,fltarr(nx,ny),headersize)
if (bitpix eq 16) then $
  outassoc=assoc(unit_out,intarr(nx,ny),headersize)
if (bitpix eq 8) then $
  outassoc=assoc(unit_out,bytarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_out, bytarr(headersize))
  rec[0]=byte(header)
endif

; ----- end standard assoc setups

; reference image initialization
im_ref=inassoc[0]
if (naverref eq 1) then im_ref=inassoc[itbest]
if (naverref eq -2) then im_ref=inassoc[nt/2]
if (naverref eq -3) then im_ref=getmeanfitscube(infile)
if (naverref lt -3) then begin
  smoothfile='/tmp/selfalignfitscube_smooth.fits'
  boxcarfitscube,infile,smoothfile,boxcar=[1,1,ans(naverref)]
endif

; define output arrays for stickies, plots, writecol
arrpx2=fltarr(nt)+1.0
arrangle=fltarr(nt)
arrinshift12=fltarr(nt,2)
arrnmet=intarr(nt)
arrmetqual=fltarr(nt)+1.E-5
arrshiftx=fltarr(nt)
arrshifty=fltarr(nt)
arrscalex=fltarr(nt)+1.0
arrscaley=fltarr(nt)+1.0

; initial value of parameters that get changed in findalignimages.pro
px1=1
px2=1
angle=0
inshift12=[0,0]

; store for non-sticky
px2_in=1 
angle_in=0
inshift12_in=[0,0]
lastokay=0

; ----- large loop over it

for it=0,nt-1 do begin

  ; im1 = present image to get aligned to im_ref
  im1=inassoc[it]
  
  ; set missingvalue for shift-in blank edges
  if (it eq 0) then begin
    centerim=im1[nx/3:nx*2/3,ny/3:ny*2/3]
    missingval=avg(centerim)
  endif
  
  ; return input image if it not in tselect
  if (tselect[0] ne -1) then begin
    if (min(abs(tselect-it)) gt 0.1) then begin
      outassoc[it]=im1
      goto,SKIPTHIS
    endif
  endif

  if (naverref eq 1 and it eq itbest) then begin
    outassoc[it]=im1
    lastokay=it
    px2=1
    angle=0
    inshift12=[0,0]
    goto,SKIPTHIS
  endif
  
  if (it eq 0 and naverref gt -2 and naverref ne 1) then begin
    outassoc[0]=im1
    goto,SKIPTHIS
  endif

  ; naverref < -3 = use time-smoothed cube sample as reference
  if (naverref lt -3) then im_ref=readfitscube(smoothfile,trange=[it,it])
  
; shift-align only (no rotate, scale)
  if (alignshift ne 0) then begin
    if (alignshift eq 1) then $
      shiftit=findimshift_rr(bytscl(im_ref),bytscl(im1),/subpix,filter=10)
    if (alignshift gt 1) then begin
      shiftfour=findimshift_tiled(bytscl(im_ref),bytscl(im1),$
                                  timediff=0,heightdiff=0,pxtile=alignshift)
      shiftit=shiftfour[0:1]
    endif
    if (show ne 0) then $
      print,' ----- selfalignfitscube finds shifts it ='+trimd(it)+$
      '  shift ='+trimd(shiftit)
    arrshiftx[it]=shiftit[0]
    arrshifty[it]=shiftit[1]
    reformimage,im1,im1_al,shift=shiftit,splinip=1,$
      missingvalue=missingval   ; NB: shift positive = 2nd to first = ref 
    outassoc[it]=im1_al
  endif ; end shift-only mode

  ; ---------- Metcalf iteration per findalignimages.pro

  if (alignmetcalf eq 1) then begin
    
; if no trimbox make im2 from im_ref smaller than im1 for findalignimages.pro
    if (trimbox[0] eq -1) then $
      reformimage,im_ref,im2,cutcentralx=nx-30,cutcentraly=ny-30 $
    else im2=im_ref

; run findalign: "forward" meaning im1 on im2 = current im_ref (unusual!)
    findalignimages,bytscl(im1),bytscl(im2),px1,px2,angle,px2asym,$
      shiftfor,shiftrev,nxfor,nyfor,nxrev,nyrev,$    
      nxmuckfor=nxmuckfor,nymuckfor=nymuckfor,$
      nxmuckrev=nxmuckrev,nymuckrev=nymuckrev,$
      skipmetcalf=skipmetcalf,maxmetcalf=maxmetcalf,minmetqual=minmetqual,$
      applypx2asym=applypx2asym,$
      trimboxim2=trimbox,inshift12=inshift12,$
      smearim1=smear,smearim2=smear,$
      histopim1=histopim1,histopim2=histopim2,$
      muckdarkim1=muckdarkim1,muckdarkim2=muckdarkim2,$
      muckbrightim1=muckbrightim1,muckbrightim2=muckbrightim2,$
      flatten=flatten,metqual=metqual,nmet=nmet,finalrinse=finalrinse,$
      blink=0,show=0,verbose=0

; copy @@@@@ demo = set scales but only incomplete copy, no scaling to finest
    pxratio=float(px1)/px2    ; refresh to newest Metcalfed px2, >1 for 1>2
    pxscale1=[pxratio,pxratio] 
    if (applypx2asym eq 1) then pxscale1=pxscale1/px2asym ; not to im2 here

; store results for sticky, writecol, plots
    arrpx2[it]=px2
    arrangle[it]=angle
    arrinshift12[it,*]=inshift12
    arrnmet[it]=nmet
    arrmetqual[it]=metqual
    arrshiftx[it]=shiftfor[0]
    arrshifty[it]=shiftfor[1]
    arrscalex[it]=pxscale1[0]
    arrscaley[it]=pxscale1[1]

; if metqual above cutmetqual use last good image; blank if first
    if (metqual ge cutmetqual) then begin
      nmet=-nmet
      if (it eq 0) then begin
        outassoc[it]=im1*0+avg(im1)
        px2=px2_in
        angle=angle_in
        inshift12=inshift12_in
      endif else begin
        outassoc[it]=outassoc[lastokay]
        if (actsticky eq 1) then begin
          px2=arrpx2[lastokay]
          angle=arrangle[lastokay]
          inshift12=arrinshift12[lastokay,*]
        endif
      endelse
    endif
    
; metqual okay: proceed
    if (metqual lt cutmetqual) then begin
      lastokay=it
      
; copy @@@@@ demo forward mode (but nxfor,nyfor > nx,ny; no px2asym to im2)
      reformimage,im1,im1for,$
        congridfactor=pxscale1,shift=shiftfor,rotate=angle,$
        nxlarge=nx,nylarge=ny,cutcentralx=nx,cutcentraly=ny,$
        missingvalue=missingval,splinip=1

; stick in file
      outassoc[it]=im1for

    endif  ; end treatment when metqual OK
    
; optionally inspect latest align pair
    if (show eq 2) then showex,im1for,im_ref,/blink

;; ; insert for checking variables here
;;   if (it eq 6) then STOP
    
; always print results per timestep (lullaby because so slow)
    print,' ===== it ='+trimd(it)+'/'+trim(nt-1)+$
      ':  nmet ='+trimd(nmet)+'  metqual ='+trimd(metqual,6)+$
      '  scale ='+trimd([arrscalex[it],arrscaley[it]],4)+$
      '  angle ='+trimd(angle,2)+$
      '  shiftx ='+trimd(arrshiftx[it],2)+$
      '  shifty ='+trimd(arrshifty[it],2)
    
; reset input values for next when nonsticky
    if (actsticky eq 0) then begin
      px2=px2_in
      angle=angle_in
      inshift12=inshift12_in
    endif

  endif ; end Metcalf mode

; positive naverref options for next im_ref
  ; pairwise next one to this one
  if (naverref eq 0) then im_ref=im1for

  ; next one to average last preceding ones
  if (naverref ge 2) then begin 
    naver=min([it+1,naverref])
    imsum=0.*im1for
    for itav=it-naver+1,it do imsum=imsum+outassoc[itav]
    im_ref=imsum/naver
    ;; ; check
    ;; print,'naver, it-naver+1, it =', naver,it-naver+1,it
    ;; showex,im_ref,outassoc[it],/blink
  endif

SKIPTHIS:
endfor ; end large loop over it 

; optionally writecol alignment results
if (transdatafile ne '') then begin
  if (alignshift ne 0) then $
    writecol,transdatafile,arrshiftx,arrshifty,fmt='(2F10.3)'
  if (alignmetcalf eq 1) then writecol,transdatafile,$
    arrnmet,arrmetqual,arrshiftx,arrshifty,arrscalex,arrscaley,arrangle,$
    fmt='(1I3,1F10.6,5F10.4)'
endif

; ----- optionally plot alignment results to show time variation

if (plotpath ne '') then begin
  if (alignmetcalf eq 1) then begin

; Metcalf quality plot  
    psfilename=plotpath+'metqual.ps'
    axrat=1.62 ; golden ratio
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
    xtitle='time step file 1'
    ytitle='log(metqual) > -5'
    xrange=[0-0.2*nt,nt-1+0.2*nt]
    qual=alog10(arrmetqual>1.E-5) ; avoid log(0)
    mima=minmax(qual)
    range=mima[1]-mima[0]
    yrange=[mima[0]-0.1*range,mima[1]+0.1*range]
    plot,indgen(nt),qual,$
      position=[0.2,0.2,0.95,0.95],$        ; margins
      xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
      psym=1,symsize=1.3,xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
    plots,[-10,nt+10],[alog10(cutmetqual),alog10(cutmetqual)],$
      noclip=0,linestyle=2
    closepsplot,psfilename,opengv=0

; Metcalf scales plot (only scale1)
    psfilename=plotpath+'scales.ps'
    axrat=1.62 ; golden ratio
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
    xtitle='time step'
    ytitle='scale factors  (x=+  y=0)'
    xrange=[0-0.2*nt,nt-1+0.2*nt]
    mima=minmax([arrscalex,arrscaley])
    range=mima[1]-mima[0]
    yrange=[mima[0]-0.1*range,mima[1]+0.1*range]
    plot,indgen(nt),arrscalex,$
      position=[0.2,0.2,0.95,0.95],$        ; margins
      xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
      psym=1,symsize=1.3,xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
    oplot,indgen(nt),arrscaley,psym=6,symsize=1.3
    closepsplot,psfilename,opengv=0

; Metcalf rotate plot  
    psfilename=plotpath+'rotate.ps'
    axrat=1.62 ; golden ratio
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
    xtitle='time step'
    ytitle='rotation angle  [deg]'
    xrange=[0-0.2*nt,nt-1+0.2*nt]
    mima=minmax(arrangle)
    range=mima[1]-mima[0]
    yrange=[mima[0]-0.1*range,mima[1]+0.1*range]
    plot,indgen(nt),arrangle,$
      position=[0.2,0.2,0.95,0.95],$        ; margins
      xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
      psym=1,symsize=1.3,xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
    closepsplot,psfilename,opengv=0
  endif  

; shifts plot  
  psfilename=plotpath+'shifts.ps'
  axrat=1.62 ; golden ratio
  openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
  xtitle='time step'
  ytitle='shifts  [px]  (x=+  y=0)'
  xrange=[0-0.2*nt,nt-1+0.2*nt]
  mima=minmax([arrshiftx,arrshifty])
  range=mima[1]-mima[0]
  yrange=[mima[0]-0.1*range,mima[1]+0.1*range]
  plot,indgen(nt),arrshiftx,$
    position=[0.2,0.2,0.95,0.95],$        ; margins
    xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
    psym=1,symsize=1.3,xtitle=xtitle,ytitle=ytitle,$
    xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
  oplot,indgen(nt),arrshifty,psym=6,symsize=1.3
  closepsplot,psfilename,opengv=0

endif ; end plot production

end ; end of program



; =============== main for testing per IDLWAVE H-c ======================

; similar heavy-deform test as under findalignimages.pro, test that there

; sharp SST scene from web demo 
cd,'/home/rutten/rr/web/sdo-demo/2014-06-24/sst'
infile='width_kr_6563_0.9_cut.fits'   ; Kevinesq detail ; int [932, 920, 53]
nt=10
selfile='/tmp/selfalignfitscube_sel.fits'
selectfile='/tmp/selfalignfitscube_select.fits'
reformcubefile,infile,selfile,trange=[0,nt-1]
boxcarfitscube,selfile,selectfile,boxcar=[2,2,5]  ; make it easier

; set trimbox, avoid SST rotation triangles, smaller means faster
trimbox=[200,500,400,700] ; nice 200x200 area upper left

; use trimbox area to define missing for blank borders in deformation
firstim=readfitscube(selectfile,trange=[0,0])
missing=avg(firstim[200:400,500:700])

; randomly deform input cube ("put SST on boat in Stockholm")
; (NB: program aborts on crop problem when deformed too much)
headsel=headfits(selectfile)
nt=fxpar(headsel,'naxis3')
seed=1 ; always same for repeated tests
r1=randomn(seed,nt)
r2=randomn(seed,nt) ; continues sampling the same series via same seed
r3=randomn(seed,nt)
r4=randomn(seed,nt)
r5=randomn(seed,nt)
shiftsall=3.*[[r1],[r2]]
;; shiftsall=1.*[[r1],[r2]]
angleall=0.5*r3
;; scaleall=[[1.0+0.005*r4],[1.0+0.005*r4]] ; random, no px2asym
scaleall=[[1.0+0.002*r4],[1.0+0.002*r4]] ; random, no px2asym, small
smearall=5.*abs((r5+1)>1)-4 ; add very bad seeing moments above 1 px 
;; smearall=2.*abs((r5+1)>1)-1 ; add bad seeing moments above 1 px, better 

; choose mode
;; alignshift=100  ; small SST px so large tile
alignshift=0
alignmetcalf=1
actsticky=0
finalrinse=100  ; small SST px so large tile

; deform with only shifts for alignshift mode
if (alignshift ne 0) then begin
  angleall=0
  scaleall=0*scaleall+1.
endif

; add very bad seeing for it=5 to be cut by cutmetqual
;; smearall[5]=50.

; initial reference image no deformation for aligning to it=0 as precise check
shiftsall[0,0:1]=0
angleall[0]=0
scaleall[0,0:1]=[1.0,1.0]

; deform the selected cube
deformfile='/tmp/selfalignfitscube_deform.fits'
reformcubefile,selectfile,deformfile,shift=shiftsall,rotate=angleall,$
  scale=scaleall,smear=smearall,missingvalue=missing,splinip=1

; parameters
naverref=-1  ; align all to first
naverref=-3  ; to time average
itbest=5
smear=5    ; useful?
flatten=50 ; useful?

maxmetcalf=6
applypx2asym=0
show=0
;; tselect=[5,7]

; output files 
transdatafile='/tmp/selfalignfitscube_transform.dat'
plotpath='/tmp/selfalignfitscube_'

; run program
outfile='/tmp/selfalign_out.fits'
selfalignfitscube,deformfile,outfile,$
  naverref=naverref,itbest=itbest,tselect=tselect,$
  smear=smear,flatten=flatten,applypx2asym=applypx2asym,$
  alignshift=alignshift,alignmetcalf=alignmetcalf,actsticky=actsticky,$
  maxmetcalf=maxmetcalf,minmetqual=minmetqual,cutmetqual=cutmetqual,$
  finalrinse=finalrinse,$
  trimbox=trimbox,transdatafile=transdatafile,plotpath=plotpath,$
  show=show

; inspect result (use showex first play button A versus B)
; zoom to center, check the granules under the fast (white) RBEs
showex,outfile,deformfile,selectfile  ; not blink but play

end


