; file: readfitscube.pro = read (part of) a cube fitsfile into memory 
; init: Aug 12 2014  Rob Rutten  Oslo
; last: Jan  9 2021  Rob Rutten  Deil

;+
function readfitscube,infile,$
  xrange=xrange,yrange=yrange,trange=trange,$
  byt=byt,nxout=nxout,nyout=nyout,ntout=ntout,headerin=headerin
 ; 
 ; read [partial] cube fitsfile into cube in memory
 ;
 ; INPUTS:
 ;   infile = string with name of fitscube [x,y,t] file
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   xrange=xrange,yrange=yrange,trange=trange,   [input]: limit ranges  
 ;   byt=byt         [input]: return bytescaled cube to conserve memory 
 ;
 ; OUTPUTS:
 ;   cube(nx,ny,nt), bytescaled if requested
 ;
 ; OPTIONAL KEYWORD OUTPUTS:
 ;   nxout, nyout, ntout  [output]: output cube dimensions for further use
 ;   headerin: infile header for furher use
 ;
 ; HISTORY:
 ;   Aug 12 2014 RR: started = copy from readsstcube.pro
 ;   Oct  9 2015 RR: corrected for endian misfit input
;-

; no-parameter answer to x=readfitscube() query
if n_params() eq 0 then begin
  print,' cube=readfitscube(infile [, options])'
  print,' options: xrange=xrange,yrange=yrange,trange=trange,'
  print,'          byt=byt,nxout=nxout,nyout=nyout,ntout=ntout,'
  print,'          headerin=headerin'
  return,-1           ; function must return something
endif

; default keywords
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(byt) eq 0) then byt=0

; set endian
bigendian=1

; get cube geometry and file datatype from the fits header
headerin=headfits_rr(infile)
headerinsize=(1+fix(n_elements(headerin)/36.))*2880
bitpix=fxpar(headerin,'bitpix')
nxin=fxpar(headerin,'naxis1') 
nyin=fxpar(headerin,'naxis2') 
ntin=fxpar(headerin,'naxis3') 

; define output cube size and type
if (xrange[1] eq -1) then xrange[1]=nxin-1
if (yrange[1] eq -1) then yrange[1]=nyin-1
if (trange[1] eq -1) then trange[1]=ntin-1
nxout=xrange[1]-xrange[0]+1
nyout=yrange[1]-yrange[0]+1
ntout=trange[1]-trange[0]+1

; open input file for assoc
get_lun, unit_in
if (bigendian) then openr,unit_in,infile,/swap_if_little_endian $
else openr,unit_in,infile
if (bitpix eq -32) then inassoc=assoc(unit_in,fltarr(nxin,nyin),headerinsize)
if (bitpix eq 16) then inassoc=assoc(unit_in,intarr(nxin,nyin),headerinsize)
if (bitpix eq 8) then inassoc=assoc(unit_in,bytarr(nxin,nyin),headerinsize)

; fill the (sub-)cube
if (bitpix eq -32) then cube=fltarr(nxout,nyout,ntout)
if (bitpix eq 16) then cube=intarr(nxout,nyout,ntout)
if (bitpix eq 8) then cube=bytarr(nxout,nyout,ntout)
for it=trange[0],trange[1] do begin
  image=inassoc[it]
  cube[*,*,it-trange[0]]=image[xrange[0]:xrange[1],yrange[0]:yrange[1]]
endfor

; close unit
free_lun,unit_in

; bytescale the whole cube (not the individual images) if requested
if (byt eq 1) then cube=bytscl(cube)

; done
return,cube

end 

; ============== test for IDLWAVE S-c ==========================


path='/home/rutten/data/SDO/2014-06-14-small/target/cubes/'
infile='hmicont.fits'
cube=readfitscube(path+infile,trange=[0,25])
showex,cube
end

