; file: makefitsflowcubes = vx, vy, divergence/convergence, vorticity
; init: Nov 11 2013  Rob Rutten  Sac Peak from dotlib/makeflowcubefiles.pro
; last: Dec  9 2013  Rob Rutten  Sac Peak
; note: Uses assoc to read input and write output cubes as files.
;       After Bob van Veelen using old Roberto Molowny Horas programs
;       via Pit Suetterlin.  These (odd.pro, flowvect.pro,
;       divergence.pro, vorticity.pro) sit in dotlib.
  
function limit,val,min,max
return,max([min,min([val,max])])
end

pro makefitsflowcubes,cubepath,infile,$
  winlength=winlength,fwhm=fwhm,cutsigma=cutsigma,$
  itstart=itstart,itend=itend

;+
 ; NAME:
 ;   makefitsflowcubes.pro 
 ; PURPOSE:
 ;   determine flow fields from an image sequence in a fitsfile cube
 ; DESCRIPTION:
 ;   uses ancient Roberto Molowny codes revised by Pit Suetterlin
 ; CALL:
 ;   makefitsflowcubes,cubepath,infile,$
 ;     [winlength=winlength,fwhm=fwhm,itstart=itstart,itened=itend]
 ; INPUTS:
 ;   cubepath = string with path of the directory for input and output files
 ;   infile = string with the name of the input fits file
 ; KEYWORDS:
 ;   winlength: temporal evaluation window in time steps (minimum 2 = default)
 ;   fwhm: spatial smooth area in pixels (default 2)
 ;   cutsigma: replace outliers with abs(value) > mean(abs) + cutsigma*sigma
 ;     (measured from first image) by zero; e.g. cutsigma=3 (default none) 
 ;   itstart, itend: select segment of the image sequence
 ; OUTPUTS:
 ;   files _flox.fits, _floy.fits, _dive.fits, _vort.fits' 
 ; HISTORY:
 ;   Nov 11 2013 RR: distilled from dotlib/makeflowcubefiles.pro   
 ;   Aug 24 2014 RR: add cutsigma
;-

; answer a no-parameter query
if (n_params() lt 2) then begin
  print,' makefitsflowcubes,cubepath,infile,'
  print,'  [winlength=winlength,fwhm=fwhm,cutsigma=cutsigma,$'
  print,'   itstart=itstart,itend=itend]'
  return
endif

; defaults for keywords
if (n_elements(winlength) eq 0) then winlength=2
if (n_elements(fwhm) eq 0) then fwhm=2
if (n_elements(cutsigma) eq 0) then cutsigma=0 
if (n_elements(itstart) eq 0) then itstart=0
if (n_elements(itend) eq 0) then itend=0

; check wheter input file  is a fits file
fits=strpos(infile,'.fits')
if (fits eq -1) then begin
  print,' ##### OOPS, infile is not a fitsfile'
  return
endif

; set output names
floxfile=str_replace(infile,'.fits','_flox.fits')
floyfile=str_replace(infile,'.fits','_floy.fits')
divfile=str_replace(infile,'.fits','_dive.fits')
vorfile=str_replace(infile,'.fits','_vort.fits')

; get cube geometry and file datatype (telescope-dependent)
inheader=headfits_rr(cubepath+infile)
nx=fxpar(inheader,'naxis1') 
ny=fxpar(inheader,'naxis2') 
nt=fxpar(inheader,'naxis3') 
if (itend eq 0) then itend=nt-1
bitpix=fxpar(inheader,'bitpix')
if (bitpix eq 8) then datatype='byte'
if (bitpix eq 16) then datatype='integer'
if (bitpix eq -32) then datatype='float'
mkhdr,header,abs(bitpix)/8,[nx,ny,itend-itstart+1]
sizeheader=size(header)  ; fits header = Nx36 "card images" = Nx2880 bytes
headersize=(1+fix(sizeheader[1]/36.))*2880
bigendian=1

; open input file for assoc
get_lun, unit_in
if (bigendian) then openr,unit_in,cubepath+infile,/swap_if_little_endian $
else openr,unit_in,cubepath+infile
if (datatype eq 'float') then $
  inassoc=assoc(unit_in,fltarr(nx,ny),headersize)
if (datatype eq 'integer') then $
  inassoc=assoc(unit_in,intarr(nx,ny),headersize)
if (datatype eq 'byte') then $
  inassoc=assoc(unit_in,bytarr(nx,ny),headersize)

; open flow x output file for assoc, write header
get_lun, unit_flox
if (bigendian) then openw,unit_flox,cubepath+floxfile,/swap_if_little_endian $
else openw,unit_flox,cubepath+floxfile
floxassoc=assoc(unit_flox,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_flox, bytarr(headersize))
  rec[0]=byte(header)
endif

; open flow y output file for assoc, write header
get_lun, unit_floy
if (bigendian) then openw,unit_floy,cubepath+floyfile,/swap_if_little_endian $
else openw,unit_floy,cubepath+floyfile
floyassoc=assoc(unit_floy,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_floy, bytarr(headersize))
  rec[0]=byte(header)
endif

; open divergence output file for assoc, write header 
get_lun, unit_div
if (bigendian) then openw,unit_div,cubepath+divfile,/swap_if_little_endian $
else openw,unit_div,cubepath+divfile
divassoc=assoc(unit_div,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_div, bytarr(headersize))
  rec[0]=byte(header)
endif

; open vorticity output file for assoc, write header 
get_lun, unit_vor
if (bigendian) then openw,unit_vor,cubepath+vorfile,/swap_if_little_endian $ 
else openw,unit_vor,cubepath+vorfile
vorassoc=assoc(unit_vor,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_vor, bytarr(headersize))
  rec[0]=byte(header)
endif

; define array cubeseg 
if (datatype eq 'float') then cubeseg=fltarr(nx,ny,winlength)
if (datatype eq 'integer') then cubeseg=intarr(nx,ny,winlength)
if (datatype eq 'byte') then cubeseg=bytarr(nx,winlength)

; start big loop over timesteps it
scalefac=0
for it=itstart,itend do begin 
  print,' ===== time step ',it

; define start and end of timecut depending on odd or even length
  if (fix(winlength/2) eq winlength/2) then begin
    segstart=limit(fix(it-fix(0.5*winlength)),0,nt-winlength-1)
    segend=segstart+winlength
  endif else begin
    segstart=limit(fix(it-floor(0.5*winlength)),0,nt-winlength)
    segend=segstart+winlength-1
  endelse

; fill cubeseg for this timestep
  for iseg=0,winlength-1 do cubeseg[*,*,iseg]=inassoc[segstart+iseg]

; compute flow map at this timestep
  flowvect,cubeseg,vx,vy,fwhm=fwhm

; test vx and vy for NaN values and set these to zero
  wnanvx=where(~finite(vx)) ; NaN values in vx?
  wnanvy=where(~finite(vy)) ; NaN values in vy?
  szwnanvx=size(wnanvx)
  if (szwnanvx[0] gt 0) then begin 
    print,' ##### nr NaN values in vx: ',szwnanvx[1]
    index=fltarr(nx,ny)       ; where gives 1D vector, need array 
    vx[wnanvx]=index[wnanvx]
  endif
  szwnanvy=size(wnanvy)
  if (szwnanvy[0] gt 0) then begin 
    print,' ##### nr NaN values in vy: ',szwnanvy[1]
    index=fltarr(nx,ny)       ; where gives 1D vector, need array 
    vy[wnanvy]=index[wnanvy]
  endif

; if cutsigma set: replace outliers by zero
if (cutsigma ne 0) then begin
  if (it eq itstart) then begin
    momvx=moment(abs(vx),sdev=sigvx)
    cutsig=momvx[0]+cutsigma*sigvx
  endif
  wexcvx=where(abs(vx) gt cutsig)
  wexcvy=where(abs(vy) gt cutsig)
  szwexcvx=size(wexcvx)
  if (szwexcvx[0] gt 0) then begin 
    print,'  ##### nr excess values in vx: ',szwexcvx[1]
    index=fltarr(nx,ny)       ; where gives 1D vector, need array 
    vx[wexcvx]=index[wexcvx]
  endif
  szwexcvy=size(wexcvy)
  if (szwexcvy[0] gt 0) then begin 
    print,'  ##### nr excess values in vy: ',szwexcvy[1]
    index=fltarr(nx,ny)       ; where gives 1D vector, need array 
    vy[wexcvy]=index[wexcvy]
  endif
endif    

; compute divergence and vorticity maps at this timestep
  div=divergence(vx,vy)
  vor=vorticity(vx,vy) 

; determine 10^x scale factor fitting integers < 32767 limit
; test scalefac=0 for presence of identical images at start
  if (scalefac eq 0) then begin
    vxclip=histo_opt_rr(vx,1E-4)  ; cut extreme outliers
    clipmom=moment(vxclip,sdev=sdevclip,/double)
    scalefac=10.^(4-fix(alog10(clipmom[0]+5*sdevclip)))   ;RR oops, was 10
    if ((clipmom[0]+5*sdevclip)*scalefac gt 32000) then scalefac=scalefac/10
    if ((clipmom[0]+5*sdevclip)*scalefac gt 32000) then scalefac=scalefac/10
    print,' ==== scalefactor: all outputs multiplied by ',scalefac
  endif
   ; NB: subsequently use imin, imax in ximovie for display clipping

; output integer values after multiplication with scalefactor
  floxassoc[it-itstart]=fix(vx*scalefac)
  floyassoc[it-itstart]=fix(vy*scalefac)
  divassoc[it-itstart]=fix(div*scalefac)
  vorassoc[it-itstart]=fix(vor*scalefac)

; if (scalefac ne 0) then STOP

; end big loop over timesteps it
endfor

; free the input and output files
free_lun,unit_in,unit_flox,unit_floy,unit_div,unit_vor

print,' === makefitsflowcubes properly completed'
end

; ======================= test on SST test data = EB2b ====================

;; path='/home/rutten/data/SST/2011-05-07-test/sstcubes/'
;; infile='fecont.fits'    ;; INT Array[924, 934, 123] 

path='/home/rutten/data/SolO/2020-05-30-first/sdo/target/cubes/'
infile='hmicont.fits'
makefitsflowcubes,path,infile,$
  winlength=8,fwhm=10,cutsigma=5 ;; $,itstart=0,itend=30

; check, with blank-edge cutout (such produce very bad values)
;; showex,path+'hmicont_flox.fits' ;; ,pos=[100,800,100,800]
;; showex,path+'hmicont_dive.fits'  ;; ,pos=[100,800,100,800]
showex,path+'hmicont_dive.fits',path+'hmicont_vort.fits',path+'aia171.fits'

end

