l; file: makefitsflowarrowcube.pro = cube arrow images from vx & vy flow cubes
; init: Nov 11 2013  Rob Rutten  Sac Peak from dotlib/makeflowarrowcubefile
; last: Aug 24 2014  Rob Rutten  Deil

pro makefitsflowarrowcube,cubepath,vxfile,vyfile,$
  trial=trial,ngrid=ngrid,nsmooth=nsmooth,arrowsize=arrowsize,$
  itstart=itstart,itend=itend

;+
 ; NAME:
 ;   makefitsflowarrowcube.pro 
 ; PURPOSE:
 ;   make a cube with small arrows outlining flows 
 ; CALL:
 ;   makefitsflowarrowcube,cubepath,vxfile,vyfile,$
 ;   trial=trial,ngrid=ngrid,nsmooth=nsmooth,arrowsize=arrowsize,$
 ;   itstart=itstart,itend=itend
 ; INPUTS:
 ;   cubepath = string with path to cubes dir (end with /) 
 ;   vxfile,vyfile: strings with input _flox.fits and _floy.fits file names
 ; OPTIONAL KEYWORD INPUTS:
 ;   trial=0/1: show the first images in a window
 ;   ngrid: vector x and y distances n pixels, default 20
 ;   nsmooth: optional smoothing (nr pixels), default 0   ### seems wrong
 ;   arrowsize: size of arrows, default 0.2
 ;   tstart, itend: select time segment (eg for tests)
 ; OUTPUTS:
 ;   fitsfile named as infile but _flox replaced by _vect
 ; METHOD:
 ;   uses assoc to permit large files
 ; RESTRICTIONS:
 ;   smooth seems wrong
 ; HISTORY:
 ;   Nov 11 2013 RR: start, from dotlib/makeflowarrowcubefile
 ;   Aug 24 2014 RR: polished
;-

; answer a no-parameter query
if (n_params() lt 3) then begin
  print,'  makefitsflowarrowcube,cubepath,vxfile,vyfile,$'
  print,'  [trial=trial,ngrid=ngrid,nsmooth=nsmooth,arrowsize=arrowsize,$]'
  print,'  itstart=itstart,itend=itend'
  return
endif

; defaults for keywords
if (n_elements(trial) eq 0) then trial=0
if (n_elements(ngrid) eq 0) then ngrid=20
if (n_elements(nsmooth) eq 0) then nsmooth=0
if (n_elements(arrowsize) eq 0) then arrowsize=0.02
if (n_elements(itstart) eq 0) then itstart=0
if (n_elements(itend) eq 0) then itend=0


; specify output filename
vecfile=str_replace(vxfile,'_flox','_vect')

; get cube geometry and file datatype (telescope-dependent)
inheader=headfits_rr(cubepath+'/'+vxfile)
nx=fxpar(inheader,'naxis1') 
ny=fxpar(inheader,'naxis2') 
nt=fxpar(inheader,'naxis3') 
if (itend eq 0) then itend=nt-1
bitpix=fxpar(inheader,'bitpix')
if (bitpix eq 8) then datatype='byte'
if (bitpix eq 16) then datatype='integer'
if (bitpix eq -32) then datatype='float'
mkhdr,header,abs(bitpix)/8,[nx,ny,itend-itstart+1]
sizeheader=size(header)  ; fits header = Nx36 "card images" = Nx2880 bytes
headersize=(1+fix(sizeheader[1]/36.))*2880
bigendian=1

; open input files for assoc
get_lun, unit_vx
if (bigendian) then openr,unit_vx,cubepath+'/'+vxfile,/swap_if_little_endian $
else openr,unit_vx,cubepath+'/'+vxfile
if (datatype eq 'float') then vxarr=fltarr(nx,ny)
if (datatype eq 'integer') then vxarr=intarr(nx,ny)
if (datatype eq 'byte') then vxarr=bytarr(nx,ny)
vxassoc=assoc(unit_vx,vxarr,headersize)

get_lun, unit_vy
if (bigendian) then openr,unit_vy,cubepath+'/'+vyfile,/swap_if_little_endian $
else  openr,unit_vy,cubepath+'/'+vyfile
if (datatype eq 'float') then vyarr=fltarr(nx,ny)
if (datatype eq 'integer') then vyarr=intarr(nx,ny) 
if (datatype eq 'byte') then vyarr=bytarr(nx,ny)
vyassoc=assoc(unit_vy,vyarr,headersize)

; open output file for assoc, write header 
get_lun, unit_vec
if (bigendian) then openw,unit_vec,cubepath+vecfile,/swap_if_little_endian $
else openw,unit_vec,cubepath+vecfile
vecassoc=assoc(unit_vec,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_vec, bytarr(headersize))
  rec[0]=byte(header)
endif

; start big loop over timesteps it
vx=intarr(nx,ny)
vy=intarr(nx,ny)
for it=itstart,itend do begin 
  print,' ==== time step ',it

; read vx and vy
  vx=vxassoc[it]
  vy=vyassoc[it]
  
; smooth and select grid values
  if (nsmooth ne 0) then begin
    vx=smooth(vx,nsmooth,/edge_truncate)
    vy=smooth(vx,nsmooth,/edge_truncate)
  endif
  nxsample=nx/ngrid
  nysample=ny/ngrid
  xsample=fltarr(nxsample,nysample)
  ysample=fltarr(nxsample,nysample)
  for ix=0,nxsample-1 do xsample[ix,*]=float(ix*ngrid)
  for iy=0,nysample-1 do ysample[*,iy]=float(iy*ngrid)

; define arrow anchor and tip coordinates
  vxsample=fltarr(nxsample,nysample)
  vysample=fltarr(nxsample,nysample)
  for ix=0,nxsample-1 do begin
    for iy=0,nysample-1 do begin
      vxsample[ix,iy]=vx[ix*ngrid,iy*ngrid]
      vysample[ix,iy]=vy[ix*ngrid,iy*ngrid]
    endfor
  endfor
  vxsample=xsample+vxsample*arrowsize
  vysample=ysample+vysample*arrowsize

; make vector arrow image (via tvrd since Tosh = 16-bit color)
  window,xsize=nx,ysize=ny,/pixmap,retain=2
  blank=bytarr(nx,ny)
  tv,blank
  arrow,xsample/nx,ysample/ny,vxsample/nx,vysample/ny,$
    /norm,hsize=-0.4,thick=0.1,color=255
  arrowimage=tvrd()
  
;; ; show for check and arrow parameter selection
  if (trial and it lt itstart+4) then begin
    sv,arrowimage
    return
  endif

; end big loop over timesteps it
  vecassoc[it-itstart]=fix(arrowimage)
endfor 

; free the input and output files
free_lun,unit_vx,unit_vy,unit_vec

print,' === makefitsflowarrowcube wrote ', vecfile

end


; ============ test per IDLWAVE S-c on SST test files =========

path='/home/rutten/data/SST/2011-05-07-test/sstcubes/'
vxfile='fecont_flox.fits' 
vyfile='fecont_floy.fits' 
makefitsflowarrowcube,path,vxfile,vyfile,$
  /trial,ngrid=20,arrowsize=0.02;,nsmooth=2

end
