; file: make4panelmovie.pro
; init: Feb 14 2014  Rob Rutten  Eindhoven, from fitscube2mpeg.pro
; last: Dec 30 2020  Rob Rutten  Deil

;+
pro make4panelmovie,cubefiles,moviefile,$
  xrange=xrange,yrange=yrange,trange=trange,$
  rebin=rebin,cutxga=cutxga,sharpen=sharpen,$
  sqrtint=sqrtint,logint=logint,$
  bytscale=bytscale,cutmin=cutmin,cutmax=cutmax,$
  quietbox=quietbox,wrappercent=wrappercent,wraprms=wraprms,$
  channelident=channelident,$
  datetime=datetime,clock=clock,location=location,sundisk=sundisk,$
  gap=gap,overlay=overlay,overcolor=overcolor,verbose=verbose

 ; make a four-panel mpeg movie from four SDO fitscubes
 ;
 ; INPUTS:
 ;   cubefiles = 4-element vector of strings path/fitscube names, clockwise 
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   xrange, yrange, trange = partial cutout specification (2-element vectors)
 ;   rebin = multiplier to nx and ny for rebinning (larger or smaller image)
 ;   cutxga = 1/0: 4p frame size 1024x768 pixels (= beamer size XGA)
 ;      sets rebin for smaller than XGA field size
 ;   sharpen = 1/0: sharpen with fixed ImageMagick parameters (for talks)
 ;   sqrtint =  4-element vector values 1/0 display sqrt(int)
 ;   logint = 4-element vector values 1/0: display alog10(int)
 ;   bytscale:   4-element vector options for bytscaling that movie panel
 ;     bytscale = 0 don't bytscale the data (should be bytes)
 ;     bytscale = 1 use (min,max) of first image for whole movie
 ;     bytscale = 2 use (min,max) of the whole image sequence (default)
 ;     bytscale = 3 bytscale every movie frame individually (not for mag, dop)
 ;     bytscale = 4 use cutmin and cutmax values for this movie
 ;     cutmin = 4-element vector with minimum value for bytscale=4
 ;       negative = - percentage fraction of min of the whole sequence
 ;       for mag and dop cutmin is set to -cutmax to maintain zero
 ;     cutmax = 4-element vector with maximum value for bytscale=4
 ;       negative = - percentage fraction of max of the whole sequence
 ;   quietbox = trimbox quiet area for mean+rms in wraprms 
 ;   wrappercent =  4-element vector wrap mean + (value/100)*mean ([0,0,0,0])
 ;   wraprms =  4-element vector wraparound at mean + value*rms ([0,0,0,0])
 ;   channelident = 1/0:  insert SDO channel identifier (0)
 ;   channelident = 4-element string array: explicit labels
 ;   datetime = 1/0: insert running date-time banner (0) 
 ;   clock = 1/0: insert running clock (0)
 ;   location = 1/0: insert location banner (0) 
 ;   sundisk = 1/0: insert solar disk with field of view  (0)
 ;     NB: SDO-style keywords needed in header 3rd cubefile for time and place
 ;   overlay = (nx,ny) frame or (nx,ny,nt) cube with overlay
 ;   overcolor = 0,1,2,3 = B/W, red, green blue
 ;   gap = nr frames of dark grey at end (to mark start of replay)
 ;   verbose = 1/0: print progress
 ;
 ; OUTPUTS: 
 ;   moviefile = string with path+filename including .mpg
 ;
 ; RESTRICTIONS:
 ;   cuts spatial extent to multiples of 16 pixels (cutxga: 512x384 pixels)
 ;
 ; METHOD:
 ;   uses assoc to accommodate large fitscube files
 ;
 ; HISTORY:
 ;   Feb 20 2014 RR: start
 ;   Aug 22 2014 RR: options cutmin, cutmax 
 ;   Sep  5 2014 RR: options gap, overlay
 ;   Apr 20 2016 RR: backhand out
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  sp,make4panelmovie
  return
endif

; set wall-clock timer (seconds)
timestart=systime(1)

; keyword defaults
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(rebin) eq 0) then rebin=1
if (n_elements(cutxga) eq 0) then cutxga=0
if (n_elements(sharpen) eq 0) then sharpen=0
if (n_elements(sqrtint) eq 0) then sqrtint=[0,0,0,0]
if (n_elements(logint) eq 0) then logint=[0,0,0,0]
if (n_elements(bytscale) eq 0) then bytscale=[2,2,2,2]
if (n_elements(cutmin) eq 0) then cutmin=0
if (n_elements(cutmax) eq 0) then cutmax=0
if (n_elements(quietbox) eq 0) then  quietbox=-1
if (n_elements(wrappercent) eq 0) then wrappercent=[0,0,0,0]
if (n_elements(wraprms) eq 0) then wraprms=[0,0,0,0]
if (n_elements(channelident) eq 0) then channelident=0
if (n_elements(datetime) eq 0) then datetime=0
if (n_elements(clock) eq 0) then clock=0
if (n_elements(location) eq 0) then location=0
if (n_elements(sundisk) eq 0) then sundisk=0
if (n_elements(gap) eq 0) then gap=0
if (n_elements(overlay) eq 0) then overlay=0
if (n_elements(overcolor) eq 0) then overcolor=0
if (n_elements(verbose) eq 0) then verbose=0

; checks
if (n_elements(overlay) eq 0) then overcolor=0

; print pacifier
print,' '
print,' --- make4panelmovie starts on '+moviefile+'; takes long'

; declarations
bigendian=1
if (cutxga ne 0) then begin
  nxp=512
  nyp=384
endif
rannr=intarr(5)
strrannr=strarr(5)

; get metadata from the first fits header and define params
inheader=headfits_rr(cubefiles[0])
if (n_elements(inheader) eq 1) then begin
  print,' ##### fitscube file not found: ',cubefiles[0]
  wait,5
  return
endif
inheadersize=(1+fix(n_elements(inheader)/36.))*2880
nxfile=fxpar(inheader,'naxis1') 
nyfile=fxpar(inheader,'naxis2') 
ntfile=fxpar(inheader,'naxis3') 

; set dimension ranges as requested, rename to not muck input params 
xmin=xrange[0]
xmax=xrange[1]
ymin=yrange[0]
ymax=yrange[1]
itstart=trange[0]
itend=trange[1]
if (xmax eq -1) then xmax=nxfile-1
if (ymax eq -1) then ymax=nyfile-1
if (itend eq -1) then itend=ntfile-1
nx=xmax-xmin+1
ny=ymax-ymin+1
nt=itend-itstart+1

; check requested cutout size
if (nx gt nxfile or ny gt nyfile or nt gt ntfile $
    or xmax gt nxfile-1 or ymax gt nyfile-1 or itend gt ntfile-1) then begin
  print,' ### ERROR: xrange or yrange or trange excess'
  print,' ###        nx, ny, nt first fitscube file = ',$
    ntostr([nxfile,nyfile,ntfile])
  return
endif

; get and check size overlay
sizeoverlay=size(overlay)
if (sizeoverlay[0] gt 0) then begin
  if (sizeoverlay[0] eq 2 and (sizeoverlay[1] ne nxfile or $
                               sizeoverlay[2] ne nyfile)) then begin
    print,' ##### overlay dimensions differ from image'
    return
  endif
  if (sizeoverlay[0] eq 3 and $
      (sizeoverlay[1] ne nxfile or $
       sizeoverlay[2] ne nyfile or sizeoverlay[3] ne ntfile)) then begin
    print,' ##### overlay dimensions differ from movie'
    return
  endif
endif

; define panel geometry
PANELSIZE:
if (cutxga eq 0) then begin
  nxp=(nx*rebin/16)*16
  nyp=(ny*rebin/16)*16
endif
if (nxp gt nx*rebin or nyp gt ny*rebin) then begin
  rebin=rebin+1
  if (verbose ne 0) then print,' --- increasing rebin to '+ntostr(rebin) 
  goto, PANELSIZE
endif
excessx=max([(nx*rebin-nxp)/2,0])
excessy=max([(ny*rebin-nyp)/2,0])
xoff=[0,nxp,nxp,0]   ; 4 panels clockwise
yoff=[nyp,nyp,0,0]

; define blank for insert overlays
blank=bytarr(nxp,nyp)

; get axes overlay image
while !d.window ne -1 do wdelete,!d.window 
window,xsize=nxp,ysize=nyp,/pixmap,retain=2
xaxisarr=0.6/rebin*indgen(nxp)*float(nxp)/(nxp-1)     ; add 1 for pixelation
yaxisarr=0.6/rebin*indgen(nyp)*float(nyp)/(nyp-1)     ; add 1 for pixelation
contour,blank,xaxisarr,yaxisarr,/nodata,xstyle=1,ystyle=1,$ 
  position=[0,0,(nxp-1.)/nxp,(nyp-1.)/nyp],$ ;RR shift outer borders in frame
  xticklen=0.01,yticklen=0.01*384/512,$
  xtickinterval=10,ytickinterval=10,color=255
imaxes=tvrd()

; define unique temporary file identifier
for irann=0,4 do begin
  rannr[irann]=fix(abs(randomn(seed))*10000)
  strrannr[irann]=strmid(string(rannr[irann]+1E5,format='(i6)'),2)
endfor

; start large loop over four input cubes
; ----------------------------------------
for icube=0,3 do begin

; pacifier
  if (verbose ne 0) then print,' --- start on '+cubefiles[icube]

; define temporary files for optional sharpening
  pnginfile='~/tmp/raw'+strrannr[icube]+'.png'
  pngoutfile='~/tmp/sharp'+strrannr[icube]+'.png'

; get file datatype from the fits header
  inheader=headfits_rr(cubefiles[icube])
  if (n_elements(inheader) eq 1) then begin
    print,' ##### fitscube file not found: '+cubefiles[icube]
    wait,5
    return
  endif
  bitpix=fxpar(inheader,'bitpix')

; get channel ident 
  channel=''
  if (n_elements(channelident) eq 1) then $
    channel=strtrim(fxpar(inheader,'channel'),2)
  if (n_elements(channelident) eq 4) then $
    channel=channelident[icube]
  
; set zeromid
  zeromid=(channel eq 'HMI magnetogram' or channel eq 'HMI Dopplergram')

; open input file for assoc
  get_lun, unit_in
  if (bigendian) then openr,unit_in,cubefiles[icube],$
    /swap_if_little_endian $
  else openr,unit_in,cubefiles[icube]
  if (bitpix eq -32) then inassoc=assoc(unit_in,fltarr(nxfile,nyfile),$
                                        inheadersize)
  if (bitpix eq 16) then inassoc=assoc(unit_in,intarr(nxfile,nyfile),$
                                       inheadersize)
  if (bitpix eq 8) then inassoc=assoc(unit_in,bytarr(nxfile,nyfile),$
                                      inheadersize)

; get min, max of first image 
  minint=min(inassoc[0])
  maxint=max(inassoc[0])

; bytscale > 1: get min and max of whole sequence whole field
  if (bytscale[icube] gt 1) then begin
    for it=itstart,itend do begin 
      mima=minmax(inassoc[it])
      if (mima[0] lt minint) then minint=mima[0]
      if (mima[1] gt maxint) then maxint=mima[1]
    endfor  
    if ((bytscale[icube] eq 4 and n_elements(cutmin) ne 4) or $
        (bytscale[icube] eq 4 and n_elements(cutmax) ne 4)) then begin
      print,' ##### bytscale = 4 requires 4-vectors cutmin and cutmax'
      return
    endif
    if (verbose eq 1) then print,' ===== icube '+ntostr(icube)+$
      '  minint = '+ntostr(minint)+'  maxint = '+ntostr(maxint)
  endif 

; if wrap requested find corresponding maxint, full field or quietbox
  if (wrappercent[icube] ne 0 or wraprms[icube] ne 0) then begin
    bytscale[icube]=0
    summean=0.
    sumrms=0.
    for it=itstart,itend do begin
      im=inassoc[it]
      if (quietbox[0] ne -1) then $
        im=im[quietbox[0]:quietbox[2],quietbox[1]:quietbox[3]]
      momim=moment(im)
      summean=summean+momim[0]
      sumrms=sumrms+sqrt(momim[1])
    endfor
    avermean=summean/(itend-itstart+1) 
    averrms=sumrms/(itend-itstart+1)
    if (wrappercent[icube] ne 0) then $
      maxint=avermean+wrappercent[icube]/100.*avermean
    if (wraprms[icube] ne 0) then maxint=avermean+wraprms[icube]*averrms
  endif

; set max and min intensity equal for magnetograms and Dopplergrams
  if (zeromid) then begin
    zeromax=max([abs(minint),abs(maxint)]) 
    if (minint lt 0) then minint=-zeromax
    if (maxint gt 0) then maxint=zeromax
  endif

; adapt max and min for sqrt rescaling
  if (sqrtint[icube]) then begin
    if (zeromid) then minint=-sqrt(abs(minint)) else minint=sqrt(minint)
    maxint=sqrt(maxint)
  endif

; adapt max and min for log rescaling
  if (logint[icube]) then begin
    if (zeromid) then minint=-alog10(abs(minint)) else minint=alog10(minint)
    maxint=alog10(maxint) 
  endif

; get channel identifier if requested 
  if (channel ne '') then begin
    while !d.window ne -1 do wdelete,!d.window 
    window,xsize=nxp,ysize=nyp,/pixmap,retain=2
    tv, blank

  ; add specification of intensity scale operator
  ;; actually EUV are log (except 304 = sqrt) but don't specify, nobody cares
  ;;      if (strpos(channel,'171') gt 0) then channel='log '+channel  
    if (sqrtint[icube]) then channel='sqrt '+channel
    if (logint[icube]) then channel='log '+channel

  ; set window position for the channel identifier around the 4-panel center
    if (icube eq 0) then begin
      xpos=nxp-nyp/40
      ypos=nyp/50
      alignpos=1
    endif
    if (icube eq 1) then begin
      xpos=nyp/40
      ypos=nyp/50
      alignpos=0
    endif
    if (icube eq 2) then begin
      xpos=nyp/40
      ypos=0.95*nyp
      alignpos=0
    endif
    if (icube eq 3) then begin
      xpos=nxp-nyp/40
      ypos=0.95*nyp
      alignpos=1
    endif

  ; get the channel insert image
    xyouts,xpos,ypos,channel,color=255,$
      /device,alignment=alignpos,charsize=2.0,charthick=1.5
    imchannel=tvrd()
  endif

; for the 3rd cube distill time and location info from SDO-style  keywords
  if (icube eq 2) then begin 

; initiate datetime values
    if (clock ne 0 or datetime ne 0) then begin
      t0start=fxpar(inheader,'starttim')
      cadence=fxpar(inheader,'cadence')
      t0time=anytim2tai(t0start)
      starttime=anytim2utc(t0time+itstart*cadence,/ccsds)
      date=strmid(starttime,0,10)
      hh=strmid(starttime,11,2) 
      mm=strmid(starttime,14,2) 
      ss=strmid(starttime,17,2)
      starttimetai=anytim2tai(starttime)
    endif

; define time sequence for the clock
    if (clock eq 1) then begin
      tt=findgen(nt)*cadence+hh*3600.+mm*60.+ss
      hr=fix(tt/3600.) mod 24
      mn=fix((tt mod 3600)/60)
      sc=fix(tt mod 60)
      tt=nnumber(hr,2)+":"+nnumber(mn,2)+":"+nnumber(sc,2)
    endif

; get location banner and/or sundisk overlay image
    if (location or sundisk) then begin

  ; get field of view coordinates
  ;RR cutout center is maintained in rebinning and cutting here
      xcen=fxpar(inheader,'xcen')+0.6*((xmin+xmax)/2-nxfile/2)
      ycen=fxpar(inheader,'ycen')+0.6*((ymin+ymax)/2-nyfile/2) 
      rsun=fxpar(inheader,'rsun_obs')
      if (rsun eq 0) then $
        print,' ##### no rsun_obs in header 3rd input cubefile'
      pxsizex=fxpar(inheader,'cdelt1')
      pxsizey=fxpar(inheader,'cdelt2')
      if (pxsizex ne pxsizey) then begin
        print,' ##### OOPS px size x not equal to px size y'
        return
      endif
      xfov=fix(nxp*pxsizex/rebin+0.5)
      yfov=fix(nyp*pxsizey/rebin+0.5)
      mu=sqrt(1.-(xcen^2+ycen^2)/rsun^2)

  ; get locbanner
      if (location) then begin
        while !d.window ne -1 do wdelete,!d.window 
        window,xsize=nxp,ysize=nyp,/pixmap,retain=2
        tv, blank
        locbanner='('+$
          ntostr(xcen/rsun,format='(f5.2)')+','+$
          ntostr(ycen/rsun,format='(f5.2)')+')  mu='+$
          ntostr(mu,format='(f5.2)')+'  FOV='+$
          ntostr(xfov)+'"'+'x'+ntostr(yfov)+'"  '
        xyouts,nyp/50,nyp/50,locbanner,color=255,$
          /device,alignment=0,charsize=1.4,charthick=1.3
        imlocbanner=tvrd()
      endif

  ; get sundisk
      if (sundisk) then begin
        while !d.window ne -1 do wdelete,!d.window 
        window,xsize=nxp,ysize=nyp,/pixmap,retain=2
        tv, blank
    ; draw sun cicle
        disksize=(nyp/8)/16*16 ; same as clock
        radius=max([disksize,48])/2
        xc=nyp/40+radius
        yc=nyp/15+radius
        draw_circle,xc,yc,radius,npts=200,/device
    ; insert field of view
        cxmin=fix(xc+radius*(xcen-xfov/2)/rsun)
        cxmax=fix(xc+radius*(xcen+xfov/2)/rsun)+1
        cymin=fix(yc+radius*(ycen-yfov/2)/rsun)
        cymax=fix(yc+radius*(ycen+yfov/2)/rsun)+1
        polyfill,[cxmin,cxmax,cxmax,cxmin],$
          [cymin,cymin,cymax,cymax],color=255,/device
        imdisk=tvrd()
      endif
    endif
  endif

; start loop over timesteps it = frame by frame processing this cube
; ------------------------------------------------------------------
  for it=itstart,itend do begin 

; print progress 
    if (verbose ne 0) then $
      print,' --- icube = '+ntostr(icube)+' timestep ', ntostr(it-itstart) $
      +' of ',ntostr(itend-itstart)

; get image 
    image=inassoc[it]

; cut as requested
    image=image[xmin:xmax,ymin:ymax]

; rebin if requested
    if (rebin ne 1) then image=rebin(image,nx*rebin,ny*rebin)

; cut size to (nxp,nyp) panel size, maintain center
    image=image[excessx:nxp+excessx-1,excessy:nyp+excessy-1]

; check multiple of 16 px
    sizeim=size(image)
    if (sizeim[1] ne nxp) then message,' ### error sizex not equal to nxp'
    if (sizeim[2] ne nyp) then message,' ### error sizey not equal to nyp'

; check for size excess
    if (nxp gt 2048 or nyp gt 2048) then $
      message, ' ### fatal errror: requested mpeg frame exceeds 2048x2048'

; rescale intensity if requested
    if (sqrtint[icube]) then begin
      if (zeromid) then begin 
        amp=sqrt(abs(float(image))) 
        negim=where(image lt 0)
        image=amp
        image[negim]=-amp[negim]
      endif else image=sqrt(float(image))
    endif

    if (logint[icube]) then begin
      if (zeromid) then begin 
        amp=alog10(abs(float(image))) 
        negim=where(image lt 0)
        image=amp
        image[negim]=-amp[negim]
      endif else image=alog10(float(image))
    endif

; wrap if requested
    if (wrappercent[icube] ne 0 or wraprms[icube] ne 0) then $
      image=byte((float(image)-minint)/(maxint-minint)*255)

; bytscale as requested
    if (bytscale[icube] eq 1 or bytscale[icube] eq 2) then $
      image=bytscl(image,min=minint,max=maxint)
    if (bytscale[icube] eq 3) then image=bytscl(image)
    if (bytscale[icube] eq 4) then begin
      if (cutmax[icube] lt 0) then $
        uppercut=float(-cutmax[icube])/100.*maxint $
      else uppercut=cutmax[icube]
      if (cutmin[icube] lt 0) then $
        lowercut=float(-cutmin[icube])/100.*abs(minint) $
      else lowercut=cutmin[icube]
      if (zeromid) then lowercut=-uppercut
      image=bytscl(image,min=lowercut,max=uppercut)
    endif

; sharpen if requested  (fixed ImageMagick parameter choice);
; won't work before bytscale when negative values (SDO EUV mag, dop);
; doing after bytscale may give black wraps for sharpened brightest,
; but I like that, makes overexposure clearer than blanked cutoff
    if (sharpen ne 0) then begin
      write_png,pnginfile,image
      spawn,'convert -unsharp 3.0x1.0+1.5+0 '+pnginfile+' '+pngoutfile  
      image=read_png(pngoutfile)
    endif

; insert axes 
    annot=where(imaxes gt 1)
    background=avg(image[annot])
    insertcolor=0+255*(background lt 180)
    image[annot]=insertcolor

; insert the channel identifier
    if (channel ne '') then begin
      annot=where(imchannel gt 1)
      background=avg(image[annot])
      insertcolor=0+255*(background lt 180)
      image[annot]=insertcolor
    endif

; insert location and timing specifiers only into the lower-right panel
; NB: this cube must have proper SDO-style keywords
    if (icube eq 2) then begin

; insert running clock
      if (clock) then begin
        while !d.window ne -1 do wdelete,!d.window 
        window,xsize=nxp,ysize=nyp,/pixmap,retain=2
        tv, blank
        clocksize=(nyp/8)/16*16 ; multiple of 16 px for mpeg 
        clocksize=max([clocksize,48])
        clock,tt[it-itstart],/dev,size=clocksize,$
          pos=[nxp-nyp/40-clocksize,nyp/15]
        imclock=tvrd()
        annot=where(imclock gt 1)
        background=avg(image[annot])
        insertcolor=0+255*(background lt 180)
        image[annot]=insertcolor
      endif

; insert running date-time banner
      if (datetime) then begin
        while !d.window ne -1 do wdelete,!d.window 
        window,xsize=nxp,ysize=nyp,/pixmap,retain=2
        tv, blank
        tittai=starttimetai+(it-itstart)*cadence
        dtbanner=strmid(anytim2utc(tittai,/ccsds),0,19)
        dtbanner=strtrim(str_replace(dtbanner,'T','  '))
        xyouts,nxp-nyp/50,nyp/50,dtbanner,color=255,$
          /device,alignment=1,charsize=1.4,charthick=1.3
        imdtbanner=tvrd()
        annot=where(imdtbanner gt 1)
        background=avg(image[annot])
        insertcolor=0+255*(background lt 180)
        image[annot]=insertcolor
      endif

; insert location
      if (location) then begin
        annot=where(imlocbanner gt 1)
        background=avg(image[annot])
        insertcolor=0+255*(background lt 180)
        image[annot]=insertcolor
      endif

; insert sundisk
      if (sundisk) then begin
        annot=where(imdisk gt 1)
        background=avg(image[annot])
        insertcolor=0+255*(background lt 180)
        image[annot]=insertcolor
      endif

; end of inserts into lower-right panel
    endif 

; do the overlay insert if requested
    if (overlay ne 0) then begin
      if (sizeoverlay[0] eq 2) then overim=overlay $
      else overim=reform(overlay[*,*,it])

;; ; only treat non-empty overlay frames
;;     im3=image
;;     if (max(overim) ne min(overim)) then begin

; manipulate overlay to same size as image
      overim=overim[xmin:xmax,ymin:ymax]
      if (rebin ne 1) then overim=rebin(overim,nx*rebin,ny*rebin)
      if (cutxga ne 0) then $
        overim=overim[excessx:nx*rebin-excessx-1,excessy:ny*rebin-excessy-1]
      overim=overim[0:nxp-1,0:nyp-1]
      while !d.window ne -1 do wdelete,!d.window 
      window,xsize=nxp,ysize=nyp,/pixmap,retain=2
      tv,blank
      tv,overim
      overimpx=tvrd()
      
; apply the overlay
      if (sizeoverlay[0] ne 0) then begin
        if (overcolor eq 0) then begin
          image[where(overimpx ge 1)]=insertcolor 
;RR need RGB 0-255 > YUV 16-235 format conversion, yak
          image=byte(fix(image*((235.-16.)/255.)+16.))
        endif
        if (overcolor gt 0) then begin
          image3col=[[[image]],[[image]],[[image]]]
          im3=transpose(image3col,[2,0,1])
          for icolor=0,2 do begin
            im=im3[icolor,*,*]
            im=reform(im)
            if (max(overimpx) gt min(overimpx)) then begin 
              if (icolor eq 0 or icolor eq 2) then im[where(overimpx ge 1)]=0
              if (icolor eq 1) then im[where(overimpx ge 1)]=255 ; green
            endif
;RR need RGB 0-255 > YUV 16-235 format conversion, yak
            im=byte(fix(im*((235.-16.)/255.)+16.))
            im3[icolor,*,*]=im 
          endfor
        endif
      endif
    endif

; write panel frame as ~/tmp/png with unique sequence name
    filename='mpegim'+strrannr[icube]+$
      strmid(string(it-itstart+1e6,format='(i7)'),1,6)
    if (overcolor eq 0) then write_png,'~/tmp/'+filename+'.png',image
    if (overcolor gt 0) then write_png,'~/tmp/'+filename+'.png',im3

  endfor ; end of the big loop over images > panel frames

; free the input file
  free_lun,unit_in

; end of the large loop over 4 input cubes
endfor 

; now start making the movie

if (verbose ne 0) then print,' --- start (lengthy) movie assembly '+moviefile

; read the 4 panels and assemble them into movie frames
frame=bytarr(nxp*2,nyp*2)
frame3=[[[frame]],[[frame]],[[frame]]]
frame3=transpose(frame3,[2,0,1])
for it=itstart,itend do begin
  framefile='mpegim'+strrannr[4]+$
    strmid(string(it-itstart+1e6,format='(i7)'),1,6)
  for icube=0,3 do begin
    infile='mpegim'+strrannr[icube]+$
      strmid(string(it-itstart+1e6,format='(i7)'),1,6)
    read_png,'~/tmp/'+infile+'.png',image
    if (overcolor eq 0) then begin 
      frame[0+xoff[icube]:nxp+xoff[icube]-1,0+yoff[icube]:nyp+yoff[icube]-1]=image[*,*]
      write_png,'~/tmp/'+framefile+'.png',frame
    endif
    if (overcolor gt 0) then begin 
      for icolor=0,2 do $
        frame3[icolor,0+xoff[icube]:nxp+xoff[icube]-1,0+yoff[icube]:nyp+yoff[icube]-1]=image[icolor,*,*]
      write_png,'~/tmp/'+framefile+'.png',frame3
    endif
  endfor
endfor

; add gap to mark repeat if requested
if (gap ne 0) then begin
  gapframe=byte(intarr(nxp*2,nyp*2)+50) ;RR nice restful dark grey 
  for it=itend+1,itend+abs(gap) do begin
    framefile='mpegim'+strrannr[4]+$
      strmid(string(it-itstart+1e6,format='(i7)'),1,6)
    write_png,'~/tmp/'+framefile+'.png',gapframe
  endfor
endif

; construct the mpeg
;RR avconv doesn't work in UU science.staff computer so back to ffmpeg
if (verbose ne 0) then print, ' --- producing mpeg'
spawn,'ffmpeg -f image2 -i ~/tmp/mpegim'+strrannr[4]+'%06d.png -r 24 -b 65536k -y -v quiet '+moviefile

; delete ~/tmp files for this movie
for ran=0,4 do spawn,'rm -f ~/tmp/mpegim'+strrannr[ran]+'* '
spawn,'rm -f '+pnginfile
spawn,'rm -f '+pngoutfile

; done: print elapsed time
if (verbose ne 0) then begin
  print,' === make4panelmovie wrote '+moviefile
  timedone=systime(1)
  timelaps=ntostr((timedone-timestart)/60.,format='(F10.2)')
  print,' === took '+timelaps+' minutes'
endif

end


; ================= main = test per IDLWAVE H-c ==========================

;; ; QUICK TEST
;; ; this test dir has a304, aia171, aia1600, aia1700, mag, cont, dop
;; ; fitscube files, small size = 241x241x11

;; ; make moving overlay circle for test
;; nx=241
;; ny=241
;; nt=11
;; overlay=bytarr(nx,ny,nt)
;; for it=0,nt-1 do begin
;;   window,xsize=nx,ysize=ny,/pixmap,retain=2
;;   tv,bytarr(nx,ny)
;;   disksize=(ny/8)/16*16 ; same as clock
;;   radius=max([disksize,48])/2
;;   xc=ny/2+it*2
;;   yc=ny/2+it*2
;;   draw_circle,xc,yc,radius,npts=100,/device
;;   overlay[*,*,it]=tvrd()
;; endfor

;; cubedir='~/data/SDO/2014-01-10-test/cubes/'            
;; cubefiles=[cubedir+'hmimag.fits',cubedir+'aia1700.fits',$
;;            cubedir+'aia1600.fits',cubedir+'aia171.fits'] ; clockwise
;; moviefile='~/data/SDO/2014-01-10-test/mpegs/4panel.mpg'
;; make4panelmovie,cubefiles,moviefile,$
;;   /cutxga,/clock,/datetime,xrange=[40,240],yrange=[30,180],$
;;   bytscale=[4,4,4,4],$
;;   cutmin=[-100,-100,-100,-100],cutmax=[7500,10000,15000,-105],$
;;   /channelident,/location,/sundisk,$
;;   /verbose,trange=[5,10],overlay=overlay,overcolor=1
;; spawn,'play '+moviefile

; run on SDO + SST data 2014-06-21-quiet = hot blob plus Ha fibril
;; cubedir='/home/rutten/data/SDO/2014-06-21-quiet/cubes_alsst/'
;; cubefiles=['hacont.fits','halc.fits','aia304_alcnt.fits','aia171_alcnt.fits']
;; moviefile=cubedir+'4panel.mpg'
;; make4panelmovie,cubedir,cubefiles,moviefile,$
;;   /cutxga,xrange=[250,800],yrange=[300,730],$
;; ;;  trange=[80,85],$  ; for trials
;; trange=[50,150],$ ; for bullet movie
;;   bytscale=[4,4,4,4],$
;;   cutmin=[-400,-150,-200,-150],cutmax=[-90,-80,-100,-100],$
;;   channelident=['SST Ha wing','SST Ha core','AIA He 304','AIA Fe 171'],$
;;   /location,/sundisk,/clock,/datetime,$
;;   /sharpen,/verbose
;; spawn,'play '+moviefile

;; datadir='/media/rutten/RRDATA/alldata/SST/2014-06-21-quiet/'
;; moviefile='/home/rutten/rr/wrk/SST/2014-06-21-quiet/mpegs/4panel-kis.mpg' 
;; overlay=readfits(datadir+'bullet/contour.fits')
;; cubefiles=[datadir+'sstcubes/ha_3.fits',datadir+'sstcubes/ha_lc.fits',$
;;   datadir+'sstcubes/ca_lc.fits',datadir+'sstcubes/ca_8.fits']
;; make4panelmovie,cubefiles,moviefile,$
;;   /cutxga,xrange=[300,850],yrange=[300,700],$
;; ;; trange=[70,76],$  ; for quick trials
;; trange=[40,150],$ ; for bullet scene shown in talk
;; ;; trange=[50,140],$
;;   bytscale=[4,4,4,4],$
;;   cutmin=[-400,-200,-150,-400],cutmax=[-90,-80,-90,-90],$
;;   channelident=['SST Ha blue wing','SST Ha line center',$
;;     'SST CaII 8542 line center','SST CaII 8542 inner blue wing'],$
;; ;; /location,/sundisk,/clock,/datetime,$ ; no proper keywords in 3rd panel
;;   /sharpen,gap=0,/verbose,overlay=overlay,overcolor=1
;; print,' '
;; spawn,'play '+moviefile

;; datadir='/media/rutten/RRHOME/alldata/SST/2014-06-21-quiet/'
;; moviefile='/home/rutten/rr/wrk/SST/2014-06-21-quiet/mpegs/4panel-HaCa.mpg' 
;; overlay=readfits(datadir+'bullet/contour.fits')
;; cubefiles=[datadir+'sstcubes/ha_3.fits',datadir+'sstcubes/ha_lc.fits',$
;;   datadir+'sstcubes/ca_lc.fits',datadir+'sstcubes/ca_8.fits']
;; make4panelmovie,cubefiles,moviefile,$
;;   /cutxga,xrange=[300,850],yrange=[300,700],$
;; trange=[70,76],$  ; for quick trials
;; ;; trange=[40,150],$ ; for bullet scene shown in talk
;; ;; trange=[50,140],$
;;   bytscale=[4,4,4,4],$
;;   cutmin=[-400,-200,-150,-400],cutmax=[-90,-80,-90,-90],$
;;   channelident=['SST Ha blue wing','SST Ha line center',$
;;     'SST CaII 8542 line center','SST CaII 8542 inner blue wing'],$
;; ;; /location,/sundisk,/clock,/datetime,$ ; no proper keywords in 3rd panel
;;   /sharpen,gap=20,/verbose

cd,'/media/rutten/RRHOME/alldata/SST/2017-09-22-peb/target'
cubefiles=['cubes/hmimag.fits','cubes/aia1700.fits','cubes/aia1600.fits',$
             'cubes/aia1617ebs.fits']
moviefile='/tmp/4panel-ebs.mpg'
trange=[0,50]
bytscale=[4,2,2,2]  ;RR FIXED SCALES !!!!
cutmin=[-7500,0,0,0]
cutmax=[7500,0,0,0]
;; wrappercent=[0,800,1200,8000]   ;RR FIXED SCALES !!!! 
wraprms=[0,8,8,8]
channelident=1
cutxga=1
datetime=1
clock=1
location=1
sundisk=1
verbose=1
quietbox=[140,5,216,216] ; from showex

make4panelmovie,cubefiles,moviefile,$
  xrange=xrange,yrange=yrange,trange=trange,$
  rebin=rebin,cutxga=cutxga,sharpen=sharpen,$
  sqrtint=sqrtint,logint=logint,bytscale=bytscale,$
  cutmin=cutmin,cutmax=cutmax,$
  quietbox=quietbox,wrappercent=wrappercent,wraprms=wraprms,$
  channelident=channelident,$
  datetime=datetime,clock=clock,location=location,sundisk=sundisk,$
  gap=gap,overlay=overlay,overcolor=overcolor,verbose=verbose

; show
spawn,'play '+moviefile

end
