; file: getmeanfitscube.pro 
; init: Feb 13 2015  Rob Rutten  Eindhoven
; last: Oct 30 2020  Rob Rutten  Deil

function getmeanfitscube,infile,cutmin=cutmin,$
  trange=trange,xrange=xrange,yrange=yrange,trimbox=trimbox

;+
 ;   return mean image from fitscube or part of it, per assoc
 ;
 ; INPUTS:
 ;   infile = string with path/filename of 3D input cube in a fits file
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   cutmin: threshold value, pass only values above
 ;   xrange, yrange = subfield specification (2-element arrays)
 ;   trange: cut time span [it1,it2]
 ;   trimbox = 4-elem px box [xmin,ymin,xmax,ymax] for (x,y) cutting
 ;
 ; OUTPUT:
 ;   outfile = mean image, same data type as fitscube, maybe smaller
 ;
 ; HISTORY:
 ;   Feb 13 2015 RR: start
 ;   Oct 30 2020 RR: subcube options
;-

; answer no-parameter query
if (n_params() lt 1) then begin
  sp,getmeanfitscube
  return,-1
endif

; keyword defaults
if (n_elements(cutmin) eq 0) then cutmin=-1.E10  ; nonzero default, why?
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(trimbox) eq 0) then trimbox=-1

; set endian
bigendian=1

; get fitscube geometry and file datatype from the fits header
header_in=headfits_rr(infile)
header_insize=(1+fix(n_elements(header_in)/36.))*2880
inbitpix=fxpar(header_in,'bitpix')
nx_in=fxpar(header_in,'naxis1') 
ny_in=fxpar(header_in,'naxis2') 
nt_in=fxpar(header_in,'naxis3') 
if (nx_in eq 0 or ny_in eq 0 or nt_in eq 0) then begin
  print,' ###### no proper fitscube since no nx, ny, nt in header '+infile
  return,0
endif

; open input file for assoc
get_lun, unit_in
if (bigendian) then openr,unit_in,infile,/swap_if_little_endian $
else openr,unit_in,infile
if (inbitpix eq -32) then $
  inassoc=assoc(unit_in,fltarr(nx_in,ny_in),header_insize)
if (inbitpix eq 16) then $
  inassoc=assoc(unit_in,intarr(nx_in,ny_in),header_insize)
if (inbitpix eq 8) then $
  inassoc=assoc(unit_in,bytarr(nx_in,ny_in),header_insize)

; check input image size
im0=inassoc[0]
sizeim0=size(im0)
nx_im0=sizeim0[1]
ny_im0=sizeim0[2]
if (nx_im0 ne nx_in or ny_im0 ne ny_in) then begin
  print, ' ##### nx or ny in header inequal to first image [nx,ny]' 
  return,0
endif

; xrange, yrange, trimbox 
xmin=xrange[0]
xmax=xrange[1]
ymin=yrange[0]
ymax=yrange[1]
if (xmax eq -1) then xmax=nx_in-1
if (ymax eq -1) then ymax=ny_in-1
if (trimbox[0] ne -1) then begin
  if (trimbox[0] gt xmin) then xmin=trimbox[0]
  if (trimbox[2] lt xmax) then xmax=trimbox[2]
  if (trimbox[1] gt ymin) then ymin=trimbox[1]
  if (trimbox[3] lt ymax) then ymax=trimbox[3]
endif
nx_out=xmax-xmin+1  ; needed to define output image size
ny_out=ymax-ymin+1

; trange
itstart=trange[0]
itend=trange[1]
if (itend eq -1) then itend=nt_in-1 

; take the mean
sumcube=fltarr(nx_out,ny_out)
for it=itstart,itend do begin
  inim=inassoc[it]>cutmin
  reformimage,inim,subim,xrange=xrange,yrange=yrange,trimbox=trimbox
  sumcube=sumcube+float(subim)
endfor
sumcube=sumcube/float(itend-itstart+1) ; same name for in-place

; change type to input type
if (inbitpix eq 16) then sumcube=fix(sumcube+0.5)
if (inbitpix eq 8) then sumcube=byte(sumcube+0.5)

; free input file
free_lun,unit_in

return,sumcube
end


; =================== main for testing per IDLWAVE H-c =======================

infile='/home/rutten/data/SDO/2014-06-14-small/target/cubes/aia304.fits'
; nx=120 ny=120 nt=85
; meanim=getmeanfitscube(infile,xrange=[20,100],yrange=[0,80],trange=[30,60])
meanim=getmeanfitscube(infile,trimbox=[20,30,90,100],trange=[20,80])

sv,meanim

end

