; file: findalignfitscubes.pro 
; init: Dec 15 2015  Rob Rutten  Deil from sdo_metcalf2gbo.pro
; last: Jan 12 2016  Rob Rutten  Deil
; note: takes typically 2-3 min/Metcalf on my Toshiba laptop

;+em shiftal

pro findalignfitscubes,file1,file2,px1,px2,angle,nmetcalf,$
  metcalftimes,$
  angfor,shiftfor,scalefor,nxfor,nyfor,$
  angrev,shiftrev,scalerev,nxrev,nyrev,$
  scasmear=scasmear,scametcalf=scametcalf,$
  aligndir=aligndir,alignlabel=alignlabel,$
  plots=plots,blink=blink,verbose=verbose

 ; NAME:
 ;   findalignfitscubes.pro
 ;
 ; PURPOSE:
 ;   find samples of alignment parameters one fitscube to another, both ways
 ;
 ; METHOD:
 ;   use findalignimages.pro on nmetcalf samples
 ;
 ; CALL:
 ;   see above
 ;
 ; REQUIRED INPUTS:
 ;   file1 = string with path/name of larger-field fitscube file 
 ;   file2 = string with path/name of the other fitscube file
 ;   px1 = px size first sequence in some unit (eg arcsec)
 ;   px2 = px size second sequence in the same units
 ;   angle = startoff estimate file2 rotation from file1 in degrees clockwise
 ;   nmetcalf = 0: get shift+rotate+scale for every time step (default, SLOW!)
 ;            = 1: do so for only the first image pair
 ;            = N>1: do so for N pairs spaced evenly in time
 ;            = [intarr]: do so for it-specified pairs (best seeing moments)
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   nxfor,nyfor,nxrev,nyrev: may be specified, then use these without change
 ;     otherwise fixed by first metcalf 
 ;   scasmear = multiplier to pixel ratio for temporary smearing (default 3)
 ;   scametcalf = coarseness parameter in auto_align_images.pro, default 0.5
 ;   aligndir = string with output path plus slash (default 'align/')
 ;   alignlabel = string identifying comparison (no spaces, default '')
 ;   plots = 1/0: yes/no plots (findsplinealign.pro makes same) (default 0)
 ;   blink = flickrr duration in seconds; 0 = no flicker (default 3)
 ;   verbose = 1/0: more/less output (default 0)
 ;
 ; OUTPUTS:
 ;   arrays in ASCII files in dir aligndir/align_arrays and in variables:
 ;      metcalftimes = it values of the Metcalf sample determinations 
 ;      angfor,shiftfor,scalefor,nxfor,nyfor = forward parameter samples
 ;      angrev,shiftrev,scalerev,nxrev,nyrev = reverse parameter samples
 ;   plots in dir aligndir/align_plots if plots ne 0  
 ;
 ; RESTRICTIONS:
 ;   - the timing sequences must be identical and have (nearly) fixed cadence
 ;   - the two fields should show comparable scenes centered fairly well
 ;   - field 1 must at least cover field 2, also after the latter's shift and
 ;       rotation; much larger field 1 is OK
 ;   - field 1 should not have (too) bad side borders or triangles.  Cut them 
 ;       off before and use the resulting parameters for full originals.
 ;   - scale corrections are in (x,y) frame of field 2 as defined by angle 
 ;       so should be applied only to field 2 unless rotation is negligible
 ;
 ; REMARKS:
 ;    - see also findalignimages.pro
 ;    - uses assoc to permit large input files   
 ;    - see also explanation file: rjlib/00-README/align-pipeline.txt
 ;
 ; HISTORY:
 ;   Nov 30 2015 RR: start (for Sunrise-II Sufi)
 ;   Jan  7 2016 RR: revamp after revamp findalignimages.pro
;-

; answer no-parameter query
if (n_params() lt 6) then begin
  print,' pro findalignfitscubes,file1,file2,px1,px2,angle,nmetcalf,$'
  print,' metcalftimes,$'
  print,' angfor,shiftfor,scalefor,nxfor,nyfor,$'
  print,' angrev,shiftrev,scalerev,nxrev,nyrev,$'
  print,' scasmear=scasmear,scametcalf=scametcalf,$'
  print,' aligndir=aligndir,alignlabel=alignlabel,$'
  print,' plots=plots,blink=blink,verbose=verbose'
  return
endif

; set wall-clock timer (seconds)
timestart=systime(1) 

; flag spaces in dirpaths (as Catherine Fischer's Mac disk)
file1=strjoin(strsplit(file1,' ',/extract),'\ ') 
file2=strjoin(strsplit(file2,' ',/extract),'\ ') 

; keyword defaults
if (n_elements(nmetcalf) eq 0) then nmetcalf=0
if (n_elements(nxfor) eq 0) then nxmetfor=0 else nxmetfor=nxfor
if (n_elements(nyfor) eq 0) then nymetfor=0 else nymetfor=nyfor
if (n_elements(nxrev) eq 0) then nxmetrev=0 else nxmterev=nxrev
if (n_elements(nyrev) eq 0) then nymetrev=0 else nymetrev=nyrev
if (n_elements(scasmear) eq 0) then scasmear=3.
if (n_elements(scametcalf) eq 0) then scametcalf=0.5
if (n_elements(aligndir) eq 0) then aligndir='/tmp/align/'
if (n_elements(alignlabel) eq 0) then alignlabel=''
if (n_elements(plots) eq 0) then plots=0
if (n_elements(blink) eq 0) then blink=3
if (n_elements(verbose) eq 0) then verbose=0

; print pacifier
if (verbose ne 0) then print,' ===== findalignfitscubes starts'

; make directories if not yet existing
spawn,'mkdir -p '+aligndir
writedir=aligndir+'align_arrays/'
spawn,'mkdir -p '+writedir
if (plots ne 0) then begin
  plotdir=aligndir+'align_plots/'
  spawn,'mkdir -p '+plotdir
endif

; set endian
bigendian=1

; ======= file1

; get cube1 geometry and file datatype from the fits header
header1=headfits_rr(file1)
header1size=(1+fix(n_elements(header1)/36.))*2880
inbitpix=fxpar(header1,'bitpix')
nx1=fxpar(header1,'naxis1') 
ny1=fxpar(header1,'naxis2') 
nt1=fxpar(header1,'naxis3') 
if (nx1 eq 0 or ny1 eq 0 or nt1 eq 0) then begin
  print,' ###### no proper fitscube since no nx, ny, nt in header '+file1
  return
endif

; open file1 for assoc
get_lun, unit1
if (bigendian) then openr,unit1,file1,/swap_if_little_endian $
else openr,unit1,file1
if (inbitpix eq -32) then $
  assoc1=assoc(unit1,fltarr(nx1,ny1),header1size)
if (inbitpix eq 16) then $
  assoc1=assoc(unit1,intarr(nx1,ny1),header1size)
if (inbitpix eq 8) then $
  assoc1=assoc(unit1,bytarr(nx1,ny1),header1size)

; check file1 image size
im0=assoc1[0]
sizeim0=size(im0)
nx_im0=sizeim0[1]
ny_im0=sizeim0[2]
if (nx_im0 ne nx1 or ny_im0 ne ny1) then begin
  print, ' ##### nx or ny in header inequal to first image [nx,ny]' 
  return
endif

; ============ file2

; get cube2 geometry and file datatype from the fits header
header2=headfits_rr(file2)
header2size=(1+fix(n_elements(header2)/36.))*2880
inbitpix=fxpar(header2,'bitpix')
nx2=fxpar(header2,'naxis1') 
ny2=fxpar(header2,'naxis2') 
nt2=fxpar(header2,'naxis3') 
if (nx2 eq 0 or ny2 eq 0 or nt2 eq 0) then begin
  print,' ###### no proper fitscube since no nx, ny, nt in header '+file2
  return
endif

; check same timing
if (nt2 ne nt1) then begin
  print,' ##### findalignfitscubes requires synched cubes but nt1 # nt2'
  return
endif

; open file2 for assoc
get_lun, unit2
if (bigendian) then openr,unit2,file2,/swap_if_little_endian $
else openr,unit2,file2
if (inbitpix eq -32) then $
  assoc2=assoc(unit2,fltarr(nx2,ny2),header2size)
if (inbitpix eq 16) then $
  assoc2=assoc(unit2,intarr(nx2,ny2),header2size)
if (inbitpix eq 8) then $
  assoc2=assoc(unit2,bytarr(nx2,ny2),header2size)

; check file2 image size
im0=assoc2[0]
sizeim0=size(im0)
nx_im0=sizeim0[1]
ny_im0=sizeim0[2]
if (nx_im0 ne nx2 or ny_im0 ne ny2) then begin
  print, ' ##### nx or ny in header inequal to first image [nx,ny]' 
  return
endif

; ================ get Metcalf(s) ================

; define ntmet = nr of time points in which to metcalf
skipmetcalf=0
nelmet=n_elements(nmetcalf)
if (nelmet eq 1) then begin   ; single value = nr of points spread evenly
  if (nmetcalf eq 0) then nmetcalf=nt1  ; all time steps
  ntmet=nmetcalf
  metskip=fix(nt2/ntmet+0.5)
endif
if (nelmet gt 1) then ntmet=nelmet  ; input array with selected time steps

; initialize angle
angmet=angle

; declare output arrays
metcalftimes=intarr(ntmet)
angfor=fltarr(ntmet)
shiftfor=fltarr(ntmet,2)
scalefor=fltarr(ntmet,2)
nxfor=intarr(ntmet)
nyfor=intarr(ntmet)
angrev=fltarr(ntmet)
shiftrev=fltarr(ntmet,2)
scalerev=fltarr(ntmet,2)
nxrev=intarr(ntmet)
nyrev=intarr(ntmet)

; start loop over Metcalf sampling
for itmet=0,ntmet-1 do begin

; select image pair
  if (nelmet eq 1) then it=itmet*metskip
  if (nelmet gt 1) then it=nmetcalf[itmet]
  if (it gt nt1-1) then break
  im1=assoc1[it]
  im2=assoc2[it]

; print progress (also for verbose=0 since next step is so slow)
  print,' ===== run findalignimages for pair itmet = '+ntostr(itmet)+$
  ' of '+ntostr(ntmet-1)

; set angmet to previous value for better Metcalf initiation
; (helps for time-dependent rotation; result is value + correction)
 if (itmet gt 1) then angmet=angfor[itmet-1]

; get Metcalf results for this image pair
; (nxmet, nymet values remain fixed after first call)
  findalignimages,im1,im2,px1,px2,angmet,$
    angmetfor,shiftmetfor,scalemetfor,nxmetfor,nymetfor,$
    angmetrev,shiftmetrev,scalemetrev,nxmetrev,nymetrev,$
    scasmear=scasmear,$
    skipmetcalf=0,scametcalf=scametcalf,$
    blink=blink,verbose=verbose

; copy into output variables
  metcalftimes[itmet]=it
  angfor[itmet]=angmetfor
  shiftfor[itmet,*]=shiftmetfor
  scalefor[itmet,*]=scalemetfor
  nxfor[itmet]=nxmetfor
  nyfor[itmet]=nymetfor
  angrev[itmet]=angmetrev
  shiftrev[itmet,*]=shiftmetrev
  scalerev[itmet,*]=scalemetrev
  nxrev[itmet]=nxmetrev
  nyrev[itmet]=nymetrev

endfor ; end loop over Metcalf sampling

;; ; check (cut and paste into IDL session)
;; plot,shiftfor[0,*],/ynozero,psym=2
;; STOP

; write forward metcalf output to file (ASCII)
nt1arr=intarr(nt1)+nt1
writecol,writedir+'formet-'+alignlabel+'.dat',$
  metcalftimes,nt1arr,angfor,$
  shiftfor[*,0],shiftfor[*,1],scalefor[*,0],scalefor[*,1],$
  nxfor,nyfor,$
  fmt='(2I6,5F9.4,2I6)' 

; write reverse metcalf output to file (ASCII)
writecol,writedir+'revmet-'+alignlabel+'.dat',$
  metcalftimes,nt1arr,angrev,$
  shiftrev[*,0],shiftrev[*,1],scalerev[*,0],scalerev[*,1],$
  nxrev,nyrev,$
  fmt='(2I6,5F9.4,2I6)' 

; make optional plots for checking and smooth selection
if (plots ne 0 and ntmet gt 2) then begin

; set plot stuff
  xarr=indgen(ntmet)
  xtitle='time step'
  thick=1
  diagname=['xshift','yshift','angle','xscale','yscale']
  senselabel=['forward','reverse']
  ydata=[[[shiftfor[*,0]],$
         [shiftfor[*,1]],[angfor],[scalefor[*,0]],[scalefor[*,1]]],$
        [[shiftrev[*,0]],$
         [shiftrev[*,1]],[angrev],[scalerev[*,0]],[scalerev[*,1]]]]

  for isense=0,1 do begin   ; loop over forward, reverse
    for idiag=0,4 do begin  ; loop over shifts, angle, scales

; make plot (against array index since no timing info) 
      psfilename=plotdir+senselabel[isense]+'-'+$
        diagname[idiag]+'-met-'+alignlabel+'.ps'
      xrange=[-ntmet/10.,(ntmet-1)+ntmet/10.]
      yarr=reform(ydata[*,idiag,isense])
      datarange=[min(yarr),max(yarr)]
      dataspan=datarange[1]-datarange[0]
      yrange=[datarange[0]-0.1*dataspan,datarange[1]+0.1*dataspan]
      openpsplot,psfilename,fontsize=8,thick=thick
      plot,xarr,yarr,$ 
        position=[0.20,0.15,0.95,0.90],/normal,$ 
        xticklen=0.03,yticklen=0.03/1.6,$       
        xtitle=xtitle,ytitle=diagname[idiag],$
        xrange=xrange,xstyle=1,yrange=yrange,ystyle=1,psym=2 

; add alignlabel
      xyouts,0.3,0.92,/norm,senselabel[isense]+' '+alignlabel

; plot done
      closepsplot,psfilename,opengv=verbose

    endfor
  endfor
endif

; print elapsed time (also if non-verbose)
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
print,' ===== findalignfitscubes took '+timelaps+' minutes'

end

; ================= test per hyper-C ==========================

;; ; standard test = SDO > SST alignment 
;; path='/home/rutten/data/SST/2011-05-07-test/'
;; sstfile=path+'sstcubes/fecont.fits'
;; sdofile=path+'sdo/cubes/hmicont.fits'
;; aligndir=path+'trial/'
;; px_sdo=0.6
;; px_sst=0.0592
;; ang_sst=48.
;; nmetcalf=10
;; ; OOPS: need equal timing sequences; I don't have/found timing for SST

;; ; first make synched fitscube pair for alignment
;; ; get timing sequence
;; path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
;; restore,path+'calib_tseries/tseries.8542.08:02:31.calib.sav'
;; nt_sst=n_elements(time)
;; timearr_sst=time
;; for it=0,nt_sst-1 do timearr_sst[it]='2014-06-21 '+time[it]
;; timearr_sst=anytim2tai(timearr_sst)
;; ; get SST cube 8542 first wavelength
;; crisp2fits,path+'crispex/crispex.8542.08:02:31.time_corrected.icube',$
;;   path+'sdo2sst/sst_ca8542_0.fits',0,25
;; ; align AIA1700 to sst_ca8542_0 and apply to all SDO cubes
;; incubesdir=path+'sdo/cubes/'
;; outcubesdir=path+'sdo_sst/'
;; sdofile=path+'sdo/cubes/aia1700.fits' 
;; sstfile=path+'sdo2sst/sst_ca8542_0.fits'
;; reformcubefile,sdofile,outcubesdir+'aia1700_sst.fits',timearr_out=timearr_sst
;; ; check:  showex,outcubesdir+'aia1700_sst.fits'
;; ; STOP

;; ; test with contrail data
;; path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
;; outcubesdir=path+'sdo_sst/'
;; sstfile=path+'sdo2sst/sst_ca8542_0.fits'
;; sdofile=outcubesdir+'aia1700_sst.fits'
;; aligndir=path+'align_sdo_sst/'
;; alignlabel='AIA1700-SST8542_0'
;; px_sdo=0.6
;; px_sst=0.057  ; from Luc
;; angle=62.6380   ; from Luc
;; ; following: best-seeing frames selected with: showex,sstfile
;; nmetcalf=[0,11,20,29,39,48,63,70,79,88,100,$
;;  111,117,121,127,141,150,161,174,184,194,200,$
;;  207,213,221,229,237,242,250,258,268,271,277,282,300,$
;;  311,314,324,330,335,344,361,371,376]
;; ;; nmetcalf=3  ; quickie
;; ;; nmetcalf=[0,174,376]  ; quickie

;; findalignfitscubes,sdofile,sstfile,px_sdo,px_sst,angle,nmetcalf,$
;;   metcalftimes,$
;;   angfor,shiftfor,scalefor,nxfor,nyfor,$
;;   angrev,shiftrev,scalerev,nxrev,nyrev,$
;;   aligndir=aligndir,alignlabel=alignlabel,plots=1,blink=0,verbose=0
;; ; ===== findalignfitscubes took 80.4 minutes
;; ; ===== findalignfitscubes took 154.6 minutes ; ???? why twice slower?

;; ; standard test Tosh-disk data = EB2
;; ; ==================================
;; path='/home/rutten/data/SST/2011-05-07-test/'
;; ; first make equal timing SDO cube
;; sstfile=path+'sstcubes/fecont.fits'
;; sdofullfile=path+'sdo/cubes/hmicont.fits'
;; sdofile=path+'sdo_sst/hmicont_timesst.fits'
;; ;; ; first make equal-timing SDO align
;; ;; reformcubefile,sdofullfile,sdofile,$
;; ;;   cadence_out=27.4,starttime_out='2011-05-07 08:56:00',$
;; ;;   stoptime_out='2011-05-07 09:52:00',$
;; ;;   /verbose
;; ;; ;  check: showex,sdofile    fitsximovie,sstfile   ;; looks good

;; aligndir=path+'sdo_sst/'
;; alignlabel='AIA1700-SST8542_0'
;; px_sdo=0.6
;; px_sst=0.0592    ; from recordidl
;; angle=48         ; from recordidl
;; nmetcalf=30
;; nmetcalf=3  ; quickie
;; findalignfitscubes,sdofile,sstfile,px_sdo,px_sst,angle,nmetcalf,$
;;   metcalftimes,$
;;   angfor,shiftfor,scalefor,nxfor,nyfor,$
;;   angrev,shiftrev,scalerev,nxrev,nyrev,$
;;   aligndir=aligndir,alignlabel=alignlabel,plots=1,blink=5,verbose=1
;; ;;  ===== findalignfitscubes took 3.7 minutes  ;; for nmetcalf=3

;; ; test Sunrise-II Sufi
;; path='/media/rutten/RRDATA/alldata/Sunrise/2013-06-12/'
;; aligndir=path+'sdo_sufi/'
;; sufifile=path+'sufi/sufi397_rev.fits'
;; sdofullfile=path+'sdo/target/cubes/aia1600.fits'
;; sdofile=aligndir+'aia1600_sufitimes.fits'
;; alignlabel='AIA1600-CaIIH'
;; px_sdo=0.6
;; px_sufi=0.02/1.02/0.99      ; from earlier
;; angle_sufi=-10.0+1.8-2.14   ; from earlier
;; nmetcalf=3
;; findalignfitscubes,sdofile,sufifile,px_sdo,px_sufi,angle_sufi,nmetcalf,$
;;   metcalftimes,$
;;   angfor,shiftfor,scalefor,nxfor,nyfor,$
;;   angrev,shiftrev,scalerev,nxrev,nyrev,$
;;   aligndir=aligndir,alignlabel=alignlabel,$
;;   plots=1,blink=10,verbose=1

;; ; test AIA business
;; path='/home/rutten/data/SDO/2014-06-14-benchmark/trial18/center/'
;; file1=path+'cubes/aia193.fits'
;; file2=path+'cubes/aia211.fits'
;; px1=0.6
;; px2=0.6
;; angle=0.0
;; nmetcalf=1
;; aligndir=path+'metcalf/'
;; alignlabel='193-211'
;; findalignfitscubes,file1,file2,px1,px2,angle,nmetcalf,$
;;   metcalftimes,$
;;   angfor,shiftfor,scalefor,nxfor,nyfor,$
;;   angrev,shiftrev,scalerev,nxrev,nyrev,$
;;   aligndir=aligndir,alignlabel=alignlabel,$
;;   plots=1,blink=10,verbose=1


; test AIA business
path='/media/rutten/SSTDATA/alldata/SST/2016-04-29/'
file1=path+'sdo2sst/aia304_algbo.fits'
file2=path+'sst/ha_wc.fits'
px1=0.058
px2=0.058
angle=0.0
nmetcalf=1
aligndir=path+'metcalf/'
alignlabel='304-Ha_core'
findalignfitscubes,file1,file2,px1,px2,angle,nmetcalf,$
  metcalftimes,$
  angfor,shiftfor,scalefor,nxfor,nyfor,$
  angrev,shiftrev,scalerev,nxrev,nyrev,$
  aligndir=aligndir,alignlabel=alignlabel,$
  plots=1,blink=10,verbose=1

end

