; file: concatfitscubes.pro = concatenate fitscube files
; init: May 26 2018  Rob Rutten  Deil
; last: May 26 2018  Rob Rutten  Deil

;+

pro concatfitscubes,indirs,outdir

 ; NAME:
 ;   concatfitscubes
 ;   
 ; PURPOSE:
 ;   concatenate same-named fitscube files in different dirs
 ;   example: SDO cubes of successive time periods (segmented JSOC pullover)
 ;
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   indirs = string array dirnames, must be sequential order
 ;     each dir must contain identical file names, same bitpix, nx, ny
 ;   outdir =  string dirname for the concatenated files, same names
 ;   
 ; HISTORY:
 ;   May 26 2018 RR: start for Alberto Sainz Dalda SDO files
;-

; answer no-parameter query 
if (n_params(0) lt 2) then begin
  print,'concatfitscubes,indirs,outdir'
  sp,concatfitscubes 
  return
endif

; set wall-clock timer (seconds)
timestart=systime(1) 

; get endian of this computer 
;RR l = little, b = big; fits = big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'

; setup arrays
ndirs=n_elements(indirs)
bitpix=intarr(ndirs)
nx=intarr(ndirs)
ny=intarr(ndirs)
nt=intarr(ndirs)

; get fitscube filenames in first dir
cubenames=file_basename(findfile(indirs[0]+'/*.fits'))
if (cubenames[0] eq '') then message,' ##### no fitscubes found in indir = '+$
  indirs[0]
ncubes=n_elements(cubenames)

; get file parameters from first fitscube per dir
for idir=0,ndirs-1 do begin
  header=headfits_rr(indirs[idir]+'/'+cubenames[0])
  if (idir eq 0) then header_out=header
  headersize=(1+fix(n_elements(header)/36.))*2880
  bitpix[idir]=fxpar(header,'bitpix')
  nx[idir]=fxpar(header,'naxis1') 
  ny[idir]=fxpar(header,'naxis2') 
  nt[idir]=fxpar(header,'naxis3') 
endfor

; check bitpix,nx,ny for first cubefile per dir
for idir=1,ndirs-1 do begin
  if (bitpix[idir] ne bitpix[0]) then begin
    print, ' ##### clash bitpix files '+cubenames[0]+$
      ' in dirs 0 and '+trimd(idir)
    return
  endif
  if (nx[idir] ne nx[0]) then begin
    print, ' ##### clash nx files '+cubenames[0]+$
      ' in dirs 0 and '+trimd(idir)
    return
  endif
  if (ny[idir] ne ny[0]) then begin
    print, ' ##### clash nx files '+cubenames[0]+$
      ' in dirs 0 and '+trimd(idir)
    return
  endif
endfor

; get total duration and set endian_out for fits
nt_conc=fix(total(nt))
endian_in='b'
endian_out='b'

; start big loop over cubenames
outfiles=outdir+'/'+cubenames
for icube=0,ncubes-1 do begin
  it_out=-1
  print,' ----- concatfitscubes starts on '+cubenames[icube]

; open output file for assoc, write header
  get_lun,unit_out
  openw,unit_out,outfiles[icube],swap_endian=(endian ne endian_out)
  if (bitpix[0] eq -32) then $
    outassoc=assoc(unit_out,fltarr(nx[0],ny[0]),headersize)
  if (bitpix[0] eq 16) then $
    outassoc=assoc(unit_out,intarr(nx[0],ny[0]),headersize)
  if (bitpix[0] eq 8) then $
    outassoc=assoc(unit_out,bytarr(nx[0],ny[0]),headersize)

; write header_out: same as in first dir but nt adapted
  if (headersize ne 0) then begin
    header=headfits_rr(indirs[0]+'/'+cubenames[icube])
    header_out=header
    sxaddpar,header_out,'naxis3',nt_conc
    rec=assoc(unit_out, bytarr(headersize))
    rec[0]=byte(header_out)
  endif

; now innner loop over idirs
  for idir=0,ndirs-1 do begin

; open assoc infile
    get_lun, unit_in
    infile=indirs[idir]+'/'+cubenames[icube]
    openr,unit_in,infile,swap_endian=(endian ne endian_in)
    if (bitpix[idir] eq -32) then $
      inassoc=assoc(unit_in,fltarr(nx[0],ny[0]),headersize)
    if (bitpix[idir] eq 16) then $
      inassoc=assoc(unit_in,intarr(nx[0],ny[0]),headersize)
    if (bitpix[idir] eq 8) then $
      inassoc=assoc(unit_in,bytarr(nx[0],ny[0]),headersize)

; read and write all images of this fitscube per assoc
    for it=0,nt[idir]-1 do begin
      im=inassoc[it]
      it_out=it_out+1
      outassoc[it_out]=im
    endfor

    free_lun,unit_in

  endfor   ; end loop over indirs

  free_lun,unit_out

endfor     ; end loop over cubenames

; free the input and output files
free_lun,unit_in,unit_out[*]

end


; =============== main for testing per IDLWAVE H-c ======================

; concatenate test cubes
;; cd,'/home/rutten/data/SDO/2014-06-14-small'
;; indirs=['target/cubes','target_scaled/cubes']  ;RR different intscales
;; outdir='concat/cubes'

; concatenate Alberto Sainz Dalda 12 hour runs on contrail pore
cd,'/media/rutten/RRDATA/alldata/SDO/2014-06-21-sainzdalda'
indirs=['sdo_long_a/target/cubes',$
        'sdo_long_b/target/cubes',$
        'sdo_long_c/target/cubes']
outdir='sdo_long/cubes'

concatfitscubes,indirs,outdir

; inspect
showex,/allsdo,sdodir=outdir

end
