; file: coalignfitscubes.pro
; init: Oct 30 2020  Rob Rutten  Deil from selfalignfitscube.pro
; last: Nov 30 2020  Rob Rutten  Deil

;+
pro coalignfitscubes,infile1,infile2,outfile1,outfile2,px1,px2,angle,$
  timearr1=timearr1,timearr2=timearr2,$
  smear1=smear1,smear2=smear2,flatten=flatten,applypx2asym=applypx2asym,$
  alignshift=alignshift,alignmetcalf=alignmetcalf,actsticky=actsticky,$
  maxmetcalf=maxmetcalf,minmetqual=minmetqual,cutmetqual=cutmetqual,$
  finalrinse=finalrinse,$
  trimbox2=trimbox2,splinip2=splinip2,transdatafile=transdatafile,$
  plotpath=plotpath,blink=blink,show=show,verbose=verbose

 ; Align file1 to "reference" file2 per file1 timestep; find and apply
 ;   either by shifts only or by shifts, rotate, and scales using <slow>
 ;   Metcalf iteration per findalignimages.pro.
 ; Outfiles as infile1 (corrected) and infile2 (coaligned).
 ;
 ; INPUTS:
 ;   infile1: string fitscube file (image sequence in classical fits format)
 ;   infile2: string fitscube file (image sequence in classical fits format)
 ;     may differ in size and type; in timing when both timearr's given
 ;   outfile1: string output fitscube path/filename (file1 dims, px, timearr)
 ;   outfile2: string output fitscube path/filename (file2 dims, px, timearr)
 ;   px1: initial estimate px size file1 (eg arcsec)
 ;   px2: fixed px size file2 (same units, eg AIA 0.6)
 ;   angle: initial estimate scene rotation angle 1 => 2 in degrees CCW
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   timearr1, timearr2 = TAI timing arrays, get with anytim2tai
 ;   smear1: smooth images 1 for alignment by boxcar [px] smearing
 ;   smear2: smooth images 2 for alignment by boxcar [px] smearing
 ;   flatten: boxcar size in px to (x,y) flatten = subtract boxcar-smeared
 ;   applypx2asym: as in findalignimages.pro but not to ref images 
 ;   alignshift = 0: no self-align by shifting only 
 ;                1: self-align by shifting with findimshift_rr.pro only
 ;                > 1: by findimshift_tiled.pr with pxtile=alignshift 
 ;   alignmetcalf = 1/0: self-align per Metcalf iteration (SLOW)
 ;   actsticky = 1/0: start findalignimages with params last good one
 ;   maxmetcalf: max number Metcalf iterations (default 4)
 ;   minmetqual: stop Metcalf iteration at this metqual value (1E-5)
 ;   cutmetqual: for higher metqual repeat previous aligned image (1E-2)
 ;   finalrinse: apply shift-together after Metcalf iteration
 ;      0: don't (default)
 ;      1: do using findimshift_rr.pro
 ;      > 1: do using findimshift_tiled.pro with value = pxtile
 ;   trimbox2: subarea images 2 to align to [xmin,ymin,xmax,ymax] 
 ;   splinip2 = 1/0: splinip in final retiming of outfile2 (0)
 ;   transdatafile: filestring writecol results, default '' = none
 ;   plotpath: path/filestart results graphs, default '' = none
 ;   blink: duration in sec for each Metcalf blinker, default 0
 ;   show = 2/1/0: showex results (2 = per Metcalf, 1 = every time step)
 ;   verbose = 2/1/0: printout (2 = Metcalf verbose, 1 = every time step)
 ; 
 ; OUTPUTS:
 ;   - outfile1: same type, size and timing as infile1
 ;   - outfile2: same size and timing as infile2, type as infile1
 ;   - transdatafile with transform values for application to parallel files
 ;   - figures with shifts, for Metcalf also metqual, scales, rotate
 ;
 ; HISTORY:
 ;   Oct 31 2020  RR: start
;-

; answer no-parameter query 
if (n_params(0) lt 6) then begin
  sp,coalignfitscubes
  return  
endif

; set wall-clock timer (seconds)
timestartcoalign=systime(1) 

; defaults for keywords
if (n_elements(naver2) eq 0) then naver2=0
if (n_elements(timearr1) eq 0) then timearr1=-1
if (n_elements(timearr2) eq 0) then timearr2=-1
if (n_elements(flatten) eq 0) then flatten=0
if (n_elements(smear1) eq 0) then smear1=0
if (n_elements(smear2) eq 0) then smear2=0
if (n_elements(naverref) eq 0) then naverref=0
if (n_elements(alignshift) eq 0) then alignshift=0
if (n_elements(alignmetcalf) eq 0) then alignmetcalf=0
if (n_elements(actsticky) eq 0) then actsticky=0
if (n_elements(applypx2asym) eq 0) then applypx2asym=0
if (n_elements(maxmetcalf) eq 0) then maxmetcalf=4     ;NB default
if (n_elements(minmetqual) eq 0) then minmetqual=1E-5  ;NB default
if (n_elements(cutmetqual) eq 0) then cutmetqual=1E-2  ;NB default   
if (n_elements(finalrinse) eq 0) then finalrinse=0
if (n_elements(trimbox2) eq 0) then trimbox2=-1
if (n_elements(splinip2) eq 0) then splinip2=0
if (n_elements(transdatafile) eq 0) then transdatafile=''
if (n_elements(plotpath) eq 0) then plotpath=''
if (n_elements(show) eq 0) then show=0
if (n_elements(verbose) eq 0) then verbose=0

; check inputs
if (alignshift eq 0  and alignmetcalf eq 0) then begin
  print,' ##### coalignfitscubes abort: neither shift nor metcalf align'
  return
endif
if (alignshift gt 0  and alignmetcalf eq 1) then begin
  print,' ##### coalignfitscubes abort: no shift and metcalf both'
  return
endif

; ----- assoc setups 

; set endian
bigendian=1

; get cube1 dimensions and datatype from fits header
head1=headfits_rr(infile1)
head1size=(1+fix(n_elements(head1)/36.))*2880
nx1=fxpar(head1,'naxis1') 
ny1=fxpar(head1,'naxis2') 
nt1=fxpar(head1,'naxis3') 
bitpix1=fxpar(head1,'bitpix')

; open infile1 for assoc = inassoc1
get_lun, unit_in1
if (bigendian) then openr,unit_in1,infile1,/swap_if_little_endian $
else openr,unit_in1,infile1
if (bitpix1 eq -32) then inassoc1=assoc(unit_in1,fltarr(nx1,ny1),head1size)
if (bitpix1 eq 16) then inassoc1=assoc(unit_in1,intarr(nx1,ny1),head1size)
if (bitpix1 eq 8) then inassoc1=assoc(unit_in1,bytarr(nx1,ny1),head1size)

; get cube2 dimensions and file datatype from the fits head
head2=headfits_rr(infile2)
head2size=(1+fix(n_elements(head2)/36.))*2880
nx2=fxpar(head2,'naxis1') 
ny2=fxpar(head2,'naxis2') 
nt2=fxpar(head2,'naxis3') 
bitpix2=fxpar(head2,'bitpix')  

; open infile2 for assoc = inassoc2
get_lun, unit_in2
if (bigendian) then openr,unit_in2,infile2,/swap_if_little_endian $
else openr,unit_in2,infile2
if (bitpix2 eq -32) then inassoc2=assoc(unit_in2,fltarr(nx2,ny2),head2size)
if (bitpix2 eq 16) then inassoc2=assoc(unit_in2,intarr(nx2,ny2),head2size)
if (bitpix2 eq 8) then inassoc2=assoc(unit_in2,bytarr(nx2,ny2),head2size)

; open outfile1 for assoc = outassoc1
nx_out1=nx1
ny_out1=ny1
nt_out1=nt1
bitpix_out1=bitpix1 
headout1=head1 
headout1size=head1size
get_lun, unit_out1
if (bigendian) then openw,unit_out1,outfile1,/swap_if_little_endian $
else openw,unit_out1,outfile1
if (bitpix_out1 eq -32) then $
  outassoc1=assoc(unit_out1,fltarr(nx_out1,ny_out1),headout1size)
if (bitpix_out1 eq 16) then $
  outassoc1=assoc(unit_out1,intarr(nx_out1,ny_out1),headout1size)
if (bitpix_out1 eq 8) then $
  outassoc1=assoc(unit_out1,bytarr(nx_out1,ny_out1),headout1size)
if (headout1size ne 0) then begin
  rec=assoc(unit_out1, bytarr(headout1size))
  rec[0]=byte(headout1)
endif

; open temporary output file with properties file2 for assoc
out2tmpfile='/tmp/coalignfitscubes_out2tmp.fits'
nx_tmp=nx2
ny_tmp=ny2
nt_tmp=nt1         ; still file1
bitpix_tmp=bitpix1 ; still file 1
headtmp=head2
headtmpsize=head2size
sxaddpar,headtmp,'naxis3',nt_tmp,'nr time steps'      ; differs from 2
sxaddpar,headtmp,'bitpix',bitpix_tmp,'bits per pixel' ; may differ from 2
get_lun, unit_tmp
if (bigendian) then openw,unit_tmp,out2tmpfile,/swap_if_little_endian $
else openw,unit_tmp,out2tmpfile
if (bitpix_tmp eq -32) then $
  out2tmpassoc=assoc(unit_tmp,fltarr(nx_tmp,ny_tmp),headtmpsize)
if (bitpix_tmp eq 16) then $
  out2tmpassoc=assoc(unit_tmp,intarr(nx_tmp,ny_tmp),headtmpsize)
if (bitpix_tmp eq 8) then $
  out2tmpassoc=assoc(unit_tmp,bytarr(nx_tmp,ny_tmp),headtmpsize)
if (headtmpsize ne 0) then begin
  rec=assoc(unit_tmp, bytarr(headtmpsize))
  rec[0]=byte(headtmp)
endif

; ----- end assoc setups

; set times
if (timearr1[0] ne -1 and timearr2[0] ne -1) then begin
  retime=1
  times1=timearr1
  times2=timearr2
endif else begin
  retime=0
  times1=indgen(nt1)
  times2=indgen(nt2)
endelse

; check times
if (nt1 ne nt2 and retime eq 0) then begin
  print,' ##### coalignfitscubes abort: inequal nt but no timearr1,timearr2'
  return
endif
if (n_elements(times1) ne nt1 or n_elements(times2) ne nt2) then begin
  print,' ##### coalignfitscubes abort: wrong time array size'
  return
endif

; define output arrays for stickies, plots, writecol
arrit2match=intarr(nt1)
arrpx2=fltarr(nt1)+px2
arrangle=fltarr(nt1)+angle
arrinshift12=fltarr(nt1,2)
arrnmet=intarr(nt1)
arrmetqual=fltarr(nt1)
arrshiftx=fltarr(nt1)
arrshifty=fltarr(nt1)
arrscalex=fltarr(nt1)+1.0
arrscaley=fltarr(nt1)+1.0

; initial value of parameters that get changed in findalignimages.pro
inshift12=[0,0]
px2_in=px2 
angle_in=angle
inshift12_in=[0,0]
lastokay=0

; ----- loop over it1 sampling file1

for it1=0,nt1-1 do begin

  ; im1 = present file1 image to get aligned to closest file2
  im1=inassoc1[it1]
  
  ; im2 
  im1time=times1[it1]
  diftime=min(abs([times2-im1time]),it2match)
  ; if not same time interpolate infile2 straddle pair 
  if (diftime lt 0.1) then im2=inassoc2[it2match] else begin
    it2next=it2match
    if (im1time gt times2[it2match] and it2match+1 lt nt2-1) then $
      it2next=it2match+1
    if (im1time le times2[it2match] and it2match-1 gt 0) then $
      it2next=it2match-1
    if (it2next eq it2match) then im2=inassoc2[it2match] else begin
      im2m=inassoc2[it2match]
      im2n=inassoc2[it2next]
      p=[[[im2m]],[[im2n]]]
      timefrac=(im1time-times2[it2match])/$
        (times2[it2next]-times2[it2match])
      im2=interpolate(p,timefrac)
    endelse
  endelse

; set missing for borders as average central part first im1 
  if (it1 eq 0) then begin
    reformimage,im1,im1trim,cutcentralx=nx1/3,cutcentraly=ny1/3
    missing=avg(im1trim)
  endif
  
; ----- find shift in shift-only align mode  ====== NB: shift-only mode
  
  if (alignshift ne 0) then begin
    if (alignshift eq 1) then $
      shiftit=findimshift_rr(bytscl(im1),bytscl(im2),$
                              /subpix,filter=10,verbose=(verbose eq 2))
     if (alignshift gt 1) then begin
      shiftfour=findimshift_tiled(bytscl(im1),bytscl(im2),$
                                 timediff=0,heightdiff=0,pxtile=alignshift,$
                                   verbose=(verbose eq 2))
      shiftit=shiftfour[0:1]
    endif
    arrshiftx[it1]=shiftit[0]
    arrshifty[it1]=shiftit[1]

    ; shift image for outfile1 and assoc (-shift since other way)
    reformimage,im1,im1cor,shift=-shiftit,missingvalue=missing,$
      nxlarge=nx1,nylarge=ny1,cutcentralx=nx1,cutcentraly=ny1,splinip=1
    outassoc1[it1]=im1cor
    
    ; shift forward image for tmp outfile2 and assoc (idem)
    reformimage,im1,im1for,shift=-shiftit,missingvalue=missing,$
      nxlarge=nx2,nylarge=ny2,cutcentralx=nx2,cutcentraly=ny2,splinip=1
    out2tmpassoc[it1]=im1for

    if (verbose gt 0) then $
      print,' ===== it1 ='+trimd(it1)+'/'+trim(nt1-1)+$
      '  shift ='+trimd(shiftit[0:1])
  endif ; end of shift only mode
  

  ; ----- Metcalf iteration per findalignimages.pro

  if (alignmetcalf eq 1) then begin
    
; make im2 smaller than im1 when it is not nor trimboxed
    if (trimbox2[0] eq -1 and (nx2 gt nx1-20 or ny2 gt ny1-20)) then $
      reformimage,im2,im2,cutcentralx=nx-30,cutcentraly=ny-30 

; run findalign: "forward" meaning im1 on im2 = current im_ref
    findalignimages,bytscl(im1),bytscl(im2),px1,px2,angle,px2asym,$
      shiftfor,shiftrev,nxfor,nyfor,nxrev,nyrev,$    
      nxmuckfor=nxmuckfor,nymuckfor=nymuckfor,$
      nxmuckrev=nxmuckrev,nymuckrev=nymuckrev,$
      skipmetcalf=skipmetcalf,maxmetcalf=maxmetcalf,minmetqual=minmetqual,$
      applypx2asym=applypx2asym,$
      trimboxim2=trimbox2,inshift12=inshift12,$
      smearim1=smear1,smearim2=smear2,$
      histopim1=histopim1,histopim2=histopim2,$
      muckdarkim1=muckdarkim1,muckdarkim2=muckdarkim2,$
      muckbrightim1=muckbrightim1,muckbrightim2=muckbrightim2,$
      flatten=flatten,metqual=metqual,nmet=nmet,finalrinse=finalrinse,$
      blink=blink,show=(show eq 2),verbose=(verbose eq 2)

; copy @@@@@ demo forward mode 
    pxratio=float(px1)/px2 
    pxscalefor=[pxratio,pxratio] 

; store results for sticky, writecol, plots
    arrit2match[it1]=it2match
    arrpx2[it1]=px2
    arrangle[it1]=angle
    arrinshift12[it1,*]=inshift12
    arrnmet[it1]=nmet
    arrmetqual[it1]=metqual
    arrshiftx[it1]=shiftfor[0]
    arrshifty[it1]=shiftfor[1]
    arrscalex[it1]=pxscalefor[0]
    arrscaley[it1]=pxscalefor[1]

; if metqual above cutmetqual use last good images; blanks if first
    if (metqual ge cutmetqual) then begin
      nmet=-nmet
      if (it1 eq 0) then begin
        outassoc1[it1]=im1*0+avg(im1)
        out2tmpassoc[it1]=im2*0+avg(im2)
        px2=px2_in
        angle=angle_in
        inshift12=inshift12_in
      endif else begin
        outassoc1[it1]=outassoc1[lastokay]
        out2tmpassoc[it1]=out2tmpassoc[lastokay]
        if (actsticky eq 1) then begin
          px2=arrpx2[lastokay]
          angle=arrangle[lastokay]
          inshift12=arrinshift12[lastokay,*]
        endif
      endelse
    endif

; metqual is in order: proceed
    if (metqual lt cutmetqual) then begin
      lastokay=it1
      
; correct im1 and put in outfile1
      pxrescale=[px2_in/px2,px2_in/px2] 
      if (applypx2asym eq 1) then pxrescale=pxrescale/px2asym ; apply to im1
      reformimage,im1,im1cor,$
        congridfactor=pxrescale,shift=shiftfor,rotate=angle,$
        missingvalue=missing,$
        nxlarge=nx1,nylarge=ny1,cutcentralx=nx1,cutcentraly=ny1,splinip=1
      outassoc1[it1]=im1cor 

 ; correct the im2-format output version and put into out2tmp file
      if (applypx2asym eq 1) then pxscalefor=pxscalefor/px2asym 
      reformimage,im1,im1for,$
        congridfactor=pxscalefor,shift=shiftfor,rotate=angle,$
        missingvalue=missing,$
        nxlarge=nx2,nylarge=ny2,cutcentralx=nx2,cutcentraly=ny2,splinip=1
      out2tmpassoc[it1]=im1for 
      
    endif ;  end treatment when metqual OK


; optionally inspect latest align pair if not show by findalignimages
    if (show eq 1) then showex,im1for,im2,/blink

; insert for checking variables here
  ;; STOP

; always print results per timestep (lullaby because so slow)
    print,' ===== it1 ='+trimd(it1)+'/'+trim(nt1-1)+$
      ':  nmet ='+trimd(nmet)+'  metqual ='+trimd(metqual,6)+$
      '  scale ='+trimd([arrscalex[it1],arrscaley[it1]],4)+$
      '  angle ='+trimd(angle,2)+$
      '  shiftx ='+trimd(arrshiftx[it1],2)+$
      '  shifty ='+trimd(arrshifty[it1],2)
    
; reset input values for next when nonsticky
    if (actsticky eq 0) then begin
      px2=px2_in
      angle=angle_in
      inshift12=inshift12_in
    endif

  endif ; end Metcalf mode

endfor ; end large loop over it 

; transform out2tmpfile to outfile2 on timearr2 or move if no retime
if (retime eq 1) then $
  reformcubefile,out2tmpfile,outfile2,$
  timearr_in=times1,timearr_out=times2,splinip=splinip2 $
else spawn,' mv '+out2tmpfile+' '+outfile2

; optionally writecol alignment results
if (transdatafile ne '') then begin
  if (alignshift ne 0) then $
    writecol,transdatafile,arrshiftx,arrshifty,fmt='(2F10.3)'
  if (alignmetcalf eq 1) then writecol,transdatafile,$
    arrnmet,arrmetqual,arrshiftx,arrshifty,arrscalex,arrscaley,arrangle,$
    fmt='(1I3,1F10.6,5F10.4)'
endif

; ----- optionally plot alignment results to show time variation

if (plotpath ne '') then begin
  sznt=10/nt1 ; symbol and character size

  if (alignmetcalf eq 1) then begin

; Metcalf quality plot  
    psfilename=plotpath+'metqual.ps'
    axrat=1.62  ; golden ratio
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
    xtitle='time step file 1'
    ytitle='log(metqual) > -5'
    xrange=[0-0.2*nt1,nt1-1+0.2*nt1]
    qual=alog10(arrmetqual>1.E-5) ; avoid log(0)
    mima=minmax(qual)
    range=abs(mima[1]-mima[0])
    offset=range*0.05
    yrange=[mima[0]-0.1*range,mima[1]+0.2*range]
    plot,indgen(nt1),qual,$
      position=[0.2,0.2,0.95,0.95],$        ; margins
      xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
      psym=1,symsize=sznt,xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
    plots,[-10,nt1+10],[alog10(cutmetqual),alog10(cutmetqual)],$
      noclip=0,linestyle=2
    xyouts,indgen(nt1),qual+offset,charsize=0.8*sznt,$
      alignment=0.5,trim(arrit2match)
    closepsplot,psfilename,opengv=0
    
; Metcalf scales plot (only scale1)
    psfilename=plotpath+'scales.ps'
    axrat=1.62 ; golden ratio
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
    xtitle='time step file 1'
    ytitle='scale factors  (x=+  y=0)'
    xrange=[0-0.2*nt1,nt1-1+0.2*nt1]
    mima=minmax([arrscalex,arrscaley])
    range=mima[1]-mima[0]
    offset=range*0.05
    yrange=[mima[0]-0.1*range,mima[1]+0.2*range]
    plot,indgen(nt1),arrscalex,$
      position=[0.2,0.2,0.95,0.95],$        ; margins
      xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
      psym=1,symsize=sznt,xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
    oplot,indgen(nt1),arrscaley,psym=6,symsize=sznt
    xyouts,indgen(nt1),arrscalex+offset,charsize=0.8*sznt,$
      alignment=0.5,trim(arrit2match)
    closepsplot,psfilename,opengv=0

; Metcalf rotate plot  
    psfilename=plotpath+'rotate.ps'
    axrat=1.62 ; golden ratio
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
    xtitle='time step file 1'
    ytitle='rotation angle  [deg]'
    xrange=[0-0.2*nt1,nt1-1+0.2*nt1]
    mima=minmax(arrangle)
    range=mima[1]-mima[0]
    offset=range*0.05
    yrange=[mima[0]-0.1*range,mima[1]+0.1*range]
    plot,indgen(nt1),arrangle,$
      position=[0.2,0.2,0.95,0.95],$        ; margins
      xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
      psym=1,symsize=sznt,xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
    xyouts,indgen(nt1),arrangle+offset,charsize=0.8*sznt,$
      alignment=0.5,trim(arrit2match)
    closepsplot,psfilename,opengv=0
  endif  

; shifts plot  
  psfilename=plotpath+'shifts.ps'
  axrat=1.62 ; golden ratio
  openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8/axrat
  xtitle='time step file 1'
  ytitle='shifts  [px]  (x=+  y=o)'
  xrange=[0-0.2*nt1,nt1-1+0.2*nt1]
  mima=minmax([arrshiftx,arrshifty])
  range=mima[1]-mima[0]
  offset=range*0.05
  yrange=[mima[0]-0.1*range,mima[1]+0.2*range]
  plot,indgen(nt1),arrshiftx,$
    position=[0.2,0.2,0.95,0.95],$        ; margins
    xticklen=0.03,yticklen=0.03/axrat,$   ; same-length ticks
    psym=1,symsize=sznt,xtitle=xtitle,ytitle=ytitle,$
    xrange=xrange,xstyle=1,yrange=yrange,ystyle=1
  oplot,indgen(nt1),arrshifty,psym=6,symsize=sznt
  xyouts,indgen(nt1),arrshiftx+offset,charsize=0.8*sznt,$
    alignment=0.5,trim(arrit2match)
  closepsplot,psfilename,opengv=0

endif ; end plot production

; print elapsed time (always since so slow)
timedone=systime(1)
timelaps=ntostr((timedone-timestartcoalign)/60.,format='(F11.1)')
print,' ===== coalignfitscubes took '+timelaps+' minutes'

end ; end of program



; =============== main for testing per IDLWAVE H-c ======================

; similar heavy-deform test as under findalignimages.pro, test that there

; sharp SST scene from web demo 
cd,'/home/rutten/rr/web/sdo-demo/2014-06-24/sst'
sstfile='kr_width_6563_cut.fits'   ; Kevinesq detail ; int [932, 920, 53]
nt=10
infile2='/tmp/infile2.fits'  ; short reference file to align to
reformcubefile,sstfile,infile2,trange=[0,nt-1]

; set trimbox, avoid SST rotation triangles, smaller means faster
trimbox2=[200,500,400,700] ; nice 200x200 area upper left

; define missing for blank borders from deforming from first trimboxed im2
firstim2=readfitscube(infile2,trange=[0,0],xrange=[200,400],yrange=[500,700])
missing=avg(firstim2)

; randomly deform full shortened sstfile > infile1
; (NB: program aborts on crop problem when deformed too much)
seed=1 ; same series for repeated tests
r1=randomn(seed,nt)
r2=randomn(seed,nt) ; continues sampling the same series via same seed
r3=randomn(seed,nt)
r4=randomn(seed,nt)
r5=randomn(seed,nt)
shiftsall=30.*[[r1],[r2]]
angleall=0.5*r3
scaleall=[[1.0+0.005*r4],[1.0+0.005*r5]] ; random, with px2asym
scaleall=[[1.0+0.005*r4],[1.0+0.005*r4]] ; random, no px2asym
smearall=2.*abs((r5+1)>1)-1 ; add not-too-bad seeing moments above 1 px
smearall=5.*abs((r5+1)>1)-4 ; add worse seeing moments above 1 px 

; add very bad seeing for it=5 to be cut by cutmetqual
smearall[5]=50.

; choose mode
alignmetcalf=0
alignshift=1

; deform with only shifts for alignshift mode
if (alignshift ne 0) then begin
  angleall=0
  scaleall=0*scaleall+1.
endif

; deform infile2 to produce infile1
infile1='/tmp/infile1.fits'
reformcubefile,infile2,infile1,shift=shiftsall,rotate=angleall,$
  scale=scaleall,smear=smearall,missingvalue=missing,splinip=1

; fake timing arrays
startsst=anytim2tai('2014-06-24_09:08:53.290') ; actually true
cadencesst='11.4'                              ; about true
timearr1=startsst+indgen(nt)*cadencesst
timearr2=timearr1+6.1  ; about in between
timearr2=timearr1+0.5  ; nearly the same for check while using straddling

; iteration parameters
px1=1
px2=1
angle=0
smear1=0 
smear2=0
flatten=0 
maxmetcalf=5
applypx2asym=0
actsticky=0
finalrinse=50

; info on the run
blink=0
show=0
verbose=0

; output files
outfile1='/tmp/coalign_out1.fits'
outfile2='/tmp/coalign_out2.fits'
transdatafile='/tmp/coalignfitscubes_transform.dat'
plotpath='/tmp/coalignfitscubes_'

; run program
coalignfitscubes,infile1,infile2,outfile1,outfile2,px1,px2,angle,$
  timearr1=timearr1,timearr2=timearr2,$
  smear1=smear1,smear2=smear2,flatten=flatten,applypx2asym=applypx2asym,$
  alignshift=alignshift,alignmetcalf=alignmetcalf,actsticky=actsticky,$
  maxmetcalf=maxmetcalf,minmetqual=minmetqual,cutmetqual=cutmetqual,$
  finalrinse=finalrinse,$
  trimbox2=trimbox2,splinip2=splinip2,transdatafile=transdatafile,$
  plotpath=plotpath,blink=blink,show=show,verbose=verbose

; inspect result
showex,outfile2,infile2,infile1,/blink  ; this case file1 and 2 same dims
showex,outfile1,infile2,infile1,/blink

end

