; file: checksplinealign.pro 
; init: Jan  8 2016  Rob Rutten  Deil
; last: Jan 16 2016  Rob Rutten  Deil
; note: below is also demo how to apply alignment results 

;+

pro checksplinealign,file1,file2,px1,px2,$
  aligndir=aligndir,alignlabel=alignlabel,$
  forfile=forfile,revfile=revfile,$
  rottime=rottime,scaletime=scaletime,$
  blinkmovies=blinkmovies,skip=skip,verbose=verbose

 ; NAME:
 ;   checksplinealign 
 ;  
 ; PURPOSE:
 ;   check alignment results; speed up by setting skip > 1   
 ;
 ; DESCRIPTION:
 ;   assumes that file1 and file2 have the same timing
 ;
 ; CALL:
 ;   see above
 ;    
 ; INPUTS:
 ;   file1 = string 'path/filename'
 ;   file2 = string 'path/filename'
 ;   px1 = pixel size images file1 in whatever units (eg arcsec)
 ;   px2 = pixel size images file2 in the same units
 ;   NB: do not yet rescale px2 with findalign results, it is done here
 ; 
 ; OPTIONAL KEYWORD INPUTS:
 ;   aligndir: string with output path plus slash (default 'align/')
 ;   alignlabel: string identifying comparison (no spaces, default '')
 ;   forfile: string 'path+filename' forward spline data (see default)
 ;   revfile: string 'path+filename' reverse spline data (see default)
 ;   rottime=1/0: use time-dependent rotate or mean (default 0)
 ;   scaletime=1/0: use time-dependent scales or means (x and y; default 0)
 ;   blinkmovies=1/0: start two ximoie blinker sessions (default 0)
 ;   skip: skip time steps to speed up this test 
 ;   verbose=1/0: print more output in terminal
 ; 
 ; OUTPUTS:
 ;   temporary files in /tmp for checking, see tests below program   
 ;
 ; HISTORY:
 ;   Jan 16 2016 RR: start    
;-

; answer no-parameter query
if (n_params() lt 4) then begin
  print,' checksplinealign,file1,file2,px1,px2,$'
  print,' aligndir=aligndir,alignlabel=alignlabel,$'
  print,' forfile=forfile,revfile=revfile,$'
  print,' rottime=rottime,scaletime=scaletime,$'
  print,' skip=skip,verbose=verbose'
  return
endif

; flag spaces in dirpaths (as Catherine Fischer's Mac disk)
file1=strjoin(strsplit(file1,' ',/extract),'\ ') 
file2=strjoin(strsplit(file2,' ',/extract),'\ ') 

; set endian
bigendian=1

; keyword defaults
if (n_elements(aligndir) eq 0) then aligndir='/tmp/align/'
if (n_elements(alignlabel) eq 0) then alignlabel=''
if (n_elements(forfile) eq 0) then $
  forfile=aligndir+'align_arrays/forspline-'+alignlabel+'.dat'
if (n_elements(revfile) eq 0) then $
  revfile=aligndir+'align_arrays/revspline-'+alignlabel+'.dat'
if (n_elements(rottime) eq 0) then rottime=0
if (n_elements(scaletime) eq 0) then scaletime=0
if (n_elements(blinkmovies) eq 0) then blinkmovies=0
if (n_elements(skip) eq 0) then skip=0
if (n_elements(verbose) eq 0) then verbose=0

; read spline result files
readcol,forfile,xshiftfor,yshiftfor,angfor,xscalefor,yscalefor,nxfor,nyfor
readcol,revfile,xshiftrev,yshiftrev,angrev,xscalerev,yscalerev,nxrev,nyrev

; make shift and scale arrays
shiftfor=[[xshiftfor],[yshiftfor]]
shiftrev=[[xshiftrev],[yshiftrev]]
scalefor=[[xscalefor],[yscalefor]]
scalerev=[[xscalerev],[yscalerev]]

; get mean scales if scaletime not set
if (scaletime eq 0) then begin
  scalefor=[mean([xscalefor]),mean([yscalefor])]
  scalerev=[mean([xscalerev]),mean([yscalerev])]
endif

; get mean angles if rottime not set
if (rottime eq 0) then begin
  angfor=avg(angfor)
  angrev=avg(angrev)
endif

; set sizes (should be time-independent already in the files) 
nxfor=nxfor[0]
nyfor=nyfor[0]
nxrev=nxrev[0]
nyrev=nyrev[0]

; set congridfactorss (no scaling yet)
pxratio=float(px1)/px2   ; pxratio > 1 means im2 finer sampled
congridfac1=[1.0,1.0]    ; when 1 is the finest (eg SDO)
congridfac2=[1.0,1.0]    ; when 2 is the finest (eg SST)
if (pxratio gt 1.001) then congridfac1=[pxratio,pxratio]
if (pxratio lt 0.999) then congridfac2=[1./pxratio,1./pxratio]

; rescale file2
if (verbose ne 0) then print,' ====== reverse: rescale file2 > file2s'
file2s='/tmp/'+alignlabel+'-file2s.fits'
reformcubefile,file2,file2s,scale=scalerev,skip=skip,/missingvalue

; ===== do reverse transformations (file2s > file1 + file1 shift)
;       transform both to center rotated field2 in larger field1
; reform file2s with the reverse parameters except shift
if (verbose ne 0) then print,' ====== reverse: reform file2s > file21r'
file21r='/tmp/'+alignlabel+'-file21r.fits'
reformcubefile,file2s,file21r,congridfactor=congridfac2,$
  nxlarge=nxrev,nylarge=nyrev,rotate=angrev,$ ;;;; scale=1./scalefor,$
  cutcentralx=nxrev,cutcentraly=nyrev,/missingmean ;;;;; ,skip=skip
; reform file1 with reverse shift
if (verbose ne 0) then print,' ====== reverse: shift file1 > file12r'
file12r='/tmp/'+alignlabel+'-file12r.fits'
reformcubefile,file1,file12r,$
  congridfactor=congridfac1,shift=-shiftrev,$
  cutcentralx=nxrev,cutcentraly=nyrev,skip=skip

; ===== do forward transformation (file1 > file12f)
if (verbose ne 0) then print,' ====== forward: reform file1 > file12f'
file12f='/tmp/'+alignlabel+'-file12f.fits'
reformcubefile,file1,file12f,$
  congridfactor=congridfac1,$
  shift=shiftfor,rotate=angfor,$ ;;;; scale=scalefor,$
  cutcentralx=nxfor,cutcentraly=nyfor,skip=skip

; ===== optionally show results 
if (blinkmovies ne 0) then begin
  if (verbose ne 0) then print,' ====== press blink; close for reverse blink'
  showex,file21r,file12r
  showex,file12f,file2s
endif

end

; ============= test per IDLWAVE Hyper-C =====================

;; ; test with 1700-8542_0 alignment
;; ; -------------------------------
;; path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
;; sstfile=path+'sdo2sst/sst_ca8542_0.fits'
;; outcubesdir=path+'sdo_sst/'
;; sdofile=outcubesdir+'aia1700_sst.fits'
;; aligndir=path+'align_sdo_sst/'
;; alignlabel='AIA1700-SST8542_0'
;; px_sdo=0.6
;; px_sst=0.057  ; from Luc, must be same as in findalignfitscubes.pro
;; checksplinealign,sdofile,sstfile,px_sdo,px_sst,$
;;   alignlabel=alignlabel,aligndir=aligndir,$
;;   blinkmovies=1,$
;;   skip=30,$
;;   verbose=1

; standard test
; -------------
path='/home/rutten/data/SST/2011-05-07-test/'
sstfile=path+'sstcubes/fecont.fits'
sdofile=path+'sdo_sst/hmicont_timesst.fits'
aligndir=path+'sdo_sst/'
alignlabel='HMIcont-FeIcont'
px_sdo=0.6
px_sst=0.0592    ; from recordidl

checksplinealign,sdofile,sstfile,px_sdo,px_sst,$
  aligndir=aligndir,alignlabel=alignlabel,$
;;  rottime=1,scaletime=1,$   ############### ???? kan niet met skip
  blinkmovies=1,skip=30,verbose=1

end
