; file: boxcarfitscube.pro 
; init: Oct  6 2015  Rob Rutten  Deil
; last: Jun  9 2020  Rob Rutten  Deil

;+

pro boxcarfitscube,infile,outfile,boxcar=boxcar,$
  clipmin=clipmin,clipmax=clipmax,verbose=verbose

 ; NAME:
 ;   boxcarfitscube.pro   
 ; PURPOSE:
 ;   boxcar-average an image sequence fitscube in x,y and or t 
 ; EXAMPLE CALL:
 ;   see end of program file
 ; INPUTS:
 ;   infile = string with path/filename of the 3D input cube in a fits file
 ;   outfile = string with path/filename 
 ;   boxcar = 3-element array with boxcar sizes dx,dy,dt (px,px,timestep) 
 ; OPTIONAL KEYWORD INPUT
 ;   clipmin=value: first clip each image to this minimum
 ;   clipmax=value: first clip each image to ths maximum
 ;   verbose = 1/0 
 ; RESTRICTIONS
 ;   does not modify the fits header, so leaves no signal of the smoothing
 ;   even boxcar values are extended by 1 (here for t, for x,y by smooth)
 ; METHOD:
 ;   uses assoc to accommodate large files, so uses smooth only for (x,y)
 ; HISTORY:
 ;   Oct  6 2015 RR: start
 ;   Jun  7 2020 RR: dt was double boxcar, made suited to doallfiles
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  sp,boxcarfitscube
  return
endif

; keyword defaults
if (n_elements(boxcar) eq 0) then boxcar=[0,0,0]
if (n_elements(clipmin) eq 0) then clipmin=0
if (n_elements(clipmax) eq 0) then clipmax=0
if (n_elements(verbose) eq 0) then verbose=0

; check boxcar
sz=size(boxcar)
if (not(sz[0] eq 1 and sz[1] eq 3 and sz[2] eq 2) or $
    boxcar[0] lt 0 or boxcar[1] lt 0 or boxcar[2] lt 0) then begin
  print,' ##### boxcar not 3-element positive int array; abort'
  STOP
endif

; add 1 to make temporal boxcar uneven (for x and y smooth does so)
if ((boxcar[2]+1)/2 ne (boxcar[2]+1)/2.) then boxcar[2]=boxcar[2]+1

; set endian
bigendian=1

; get fitscube geometry and file datatype from the fits header
header=headfits_rr(infile)
headersize=(1+fix(n_elements(header)/36.))*2880
bitpix=fxpar(header,'bitpix')
nx=fxpar(header,'naxis1') 
ny=fxpar(header,'naxis2') 
nt=fxpar(header,'naxis3') 

; check dimensions
if (nx eq 0 or ny eq 0 or nt eq 0) then begin
  print,' ###### no 3D fitscube since no nx, ny, nt in header '+infile
  return
endif

; open input file for assoc
get_lun, unit_in
if (bigendian) then openr,unit_in,infile,/swap_if_little_endian $
else openr,unit_in,infile
if (bitpix eq -32) then $
  inassoc=assoc(unit_in,fltarr(nx,ny),headersize)
if (bitpix eq 16) then $
  inassoc=assoc(unit_in,intarr(nx,ny),headersize)
if (bitpix eq 8) then $
  inassoc=assoc(unit_in,bytarr(nx,ny),headersize)

; check input image size
im0=inassoc[0]
sizeim0=size(im0)
nx_im0=sizeim0[1]
ny_im0=sizeim0[2]
if (nx_im0 ne nx or ny_im0 ne ny) then begin
  print, ' ##### nx or ny in header inequal to first image [nx,ny]' 
  return
endif

; open output file for assoc, write header
get_lun, unit_out
if (bigendian) then openw,unit_out,outfile,/swap_if_little_endian $
else openw,unit_out,outfile
if (bitpix eq -32) then $
  outassoc=assoc(unit_out,fltarr(nx,ny),headersize)
if (bitpix eq 16) then $
  outassoc=assoc(unit_out,intarr(nx,ny),headersize)
if (bitpix eq 8) then $
  outassoc=assoc(unit_out,bytarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_out, bytarr(headersize))
  rec[0]=byte(header)
endif

; apply 3D boxcar averaging 
for it=0,nt-1 do begin

; temporal boxcar, can't use smooth since using assoc (no cube)
; OOPS this is double boxcar ;?? what do you mean?
  itmin=it-boxcar[2]/2
  if (itmin lt 0) then itmin=0
  itmax=it+boxcar[2]/2
  if (itmax gt nt-1) then itmax=nt-1
  ntsteps=itmax-itmin+1  ; 11 trees along 100m road at 10m intervals
  sumim=float(inassoc[it])
  if (clipmin ne 0) then sumim=sumim>clipmin
  if (clipmax ne 0) then sumim=sumim<clipmax
  for ait=itmin,it-1 do begin
    addim=inassoc[ait]
    if (clipmin ne 0) then addim=addim>clipmin
    if (clipmax ne 0) then addim=addim<clipmax
    sumim=sumim+addim
  endfor
  for ait=it+1,itmax do begin
    addim=inassoc[ait]
    if (clipmin ne 0) then addim=addim>clipmin
    if (clipmax ne 0) then addim=addim<clipmax
    sumim=sumim+addim
  endfor
  if (bitpix eq -32) then averim=sumim/ntsteps
  if (bitpix eq 16) then averim=fix(sumim/float(ntsteps))
  if (bitpix eq 8) then averim=byte(fix(sumim/float(ntsteps)))

; now the spatial smearing
  outassoc[it]=smooth(averim,[boxcar[0],boxcar[1]],/edge_truncate)

endfor ; end loop over it

; print pacifier
if (verbose eq 1) then print,' ===== boxcarfitscube wrote '+outfile

; free the files 
free_lun,unit_in,unit_out

end


;================= main for testing per IDLWAVE H-c =======================

infile='/home/rutten/data/SDO/2014-06-14-small/target/cubes/aia304.fits'
outfile='/tmp/boxcarout.fits'
boxcar=[10,10,10]

boxcarfitscube,infile,outfile,boxcar=boxcar,/verbose
showex,infile,outfile

end

