; file: alignonefitscube.pro 
; init: Jan 12 2016  Rob Rutten  Deil
; last: Jan 16 2016  Rob Rutten  Deil

;+

pro alignonefitscube,file1,file2,$
  px1=px1,px2=px2,$
  sense=sense,type=type,$
  aligndir=aligndir,alignlabel=alignlabel,$
  forfile=forfile,revfile=revfile,$
  rottime=rottime,scaletime=scaletime,$ 
  verbose=verbose,_extra=reformkeywords

 ; NAME:
 ;   alignonefitscube 
 ;  
 ; PURPOSE:
 ;   transform one fitscube with findsplinealign output parameters
 ;
 ; DESCRIPTION:
 ;   follows checksplinealign.pro
 ;   good for multi-file operation using doallfiles.pro
 ;
 ; CALL:
 ;   see above
 ;    
 ; INPUTS:
 ;   file1 = string 'path/filename' = inputfile
 ;   file2 = string 'path/filename' = outputfile
 ; 
 ; REQUIRED KEYWORD INPUTS:
 ;   px1 = pixel size images file1 in whatever units (eg arcsec)
 ;   px2 = pixel size images file2 in the same units
 ;     NB: do not yet rescale with the findalign output, it is done here
 ;   sense: 1 = forward, 2 = reverse
 ;   type: 1 = type1 (larger FOV, eg SDO), 2 = type2 (eg SST)
 ;   aligndir: string with output path plus slash (default 'align/')
 ;   alignlabel: string identifying comparison (no spaces, default '')
 ;   forfile: string 'path+filename' forward spline data (see default)
 ;   revfile: string 'path+filename' reverse spline data (see default)
 ;
 ; OPTIONAL KEYWORD INPUTS
 ;   rottime=1/0: use time-dependent rotate or mean (default 0)
 ;   scaletime=1/0: use time-dependent scale or means (x,y; default 0)
 ;   skip: skip time steps to speed up this test 
 ;   verbose=1/0: print more output in terminal
 ;   all keywords of reformcubefile.pro (there are many)
 ; 
 ; OUTPUTS:
 ;   file2
 ;
 ; HISTORY:
 ;   Jan 14 2016 RR: start    
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  print,' alignonefitscube,file1,file2,$'
  print,' px1=px1,px2=px2,sense=sense,type=type,$'
  print,' aligndir=aligndir,alignlabel=alignlabel,$'
  print,' forfile=forfile,revfile=revfile,$'
  print,' rottime=rottime,scaletime=scaletime,$'
  print,' verbose=verbose,_extra=reformkeywords'
  return
endif

; flag spaces in dirpaths (as Catherine Fischer's Mac disk)
file1=strjoin(strsplit(file1,' ',/extract),'\ ') 
file2=strjoin(strsplit(file2,' ',/extract),'\ ') 

; set endian
bigendian=1

; keyword defaults
if (n_elements(aligndir) eq 0) then aligndir='/tmp/align/'
if (n_elements(alignlabel) eq 0) then alignlabel=''
if (n_elements(forfile) eq 0) then $
  forfile=aligndir+'align_arrays/forspline-'+alignlabel+'.dat'
if (n_elements(revfile) eq 0) then $
  revfile=aligndir+'align_arrays/revspline-'+alignlabel+'.dat'
if (n_elements(rottime) eq 0) then rottime=0
if (n_elements(scaletime) eq 0) then scaletime=0
if (n_elements(skip) eq 0) then skip=0
if (n_elements(verbose) eq 0) then verbose=0

; check
if (sense ne 1 and sense ne 2) then begin
  print,' ##### alignonefitscube abort: sense not 1 or 2'
  return
endif
if (type ne 1 and type ne 2) then begin
  print,' ##### alignonefitscube abort: type not 1 or 2'
  return
endif

; print pacifier
if (verbose ne 0) then print,' ===== alignonefitscube starts on file ',file1
; read spline result files
readcol,forfile,xshiftfor,yshiftfor,angfor,xscalefor,yscalefor,nxfor,nyfor
readcol,revfile,xshiftrev,yshiftrev,angrev,xscalerev,yscalerev,nxrev,nyrev

; make shift and scale arrays
shiftfor=[[xshiftfor],[yshiftfor]]
shiftrev=[[xshiftrev],[yshiftrev]]
scalefor=[[xscalefor],[yscalefor]]
scalerev=[[xscalerev],[yscalerev]]

; get mean scales if scaletime not set
if (scaletime eq 0) then begin
  scalefor=[mean([xscalefor]),mean([yscalefor])]
  scalerev=[mean([xscalerev]),mean([yscalerev])]
endif

; get mean angles if rottime not set
if (rottime eq 0) then begin
  angfor=avg(angfor)
  angrev=avg(angrev)
endif

; set sizes (should be time-independent already in the files) 
nxfor=nxfor[0]
nyfor=nyfor[0]
nxrev=nxrev[0]
nyrev=nyrev[0]

; set congridfactors (no scaling yet)
pxratio=float(px1)/px2   ; pxratio > 1 means im2 finer sampled
congridfac1=[1.0,1.0]    ; when 1 is the finest (eg SDO
congridfac2=[1.0,1.0]    ; when 2 is the finest (eg SST)
if (pxratio gt 1.001) then congridfac1=[pxratio,pxratio]
if (pxratio lt 0.999) then congridfac2=[1./pxratio,1./pxratio]

; ===== do reverse transformation (needs to be done on both fitscubes)
; reform file2 with the reverse parameters except shift
if (sense eq 2) then begin
  if (type eq 2) then begin
    reformcubefile,file1,file2,congridfactor=congridfac2,$
      nxlarge=nxrev,nylarge=nyrev,rotate=angrev,scale=1./scalefor,$
      cutcentralx=nxrev,cutcentraly=nyrev,/missingmean,skip=skip,$
      _extra=reformkeywords
  endif
  if (type eq 1) then begin
    reformcubefile,file1,file2,$
      congridfactor=congridfac1,shift=-shiftrev,$
      cutcentralx=nxrev,cutcentraly=nyrev,$
      _extra=reformkeywords
  endif
endif

; ===== do forward transformation (type1 > type2)
if (sense eq 1) then begin
  if (type eq 1) then begin
    reformcubefile,file1,file2,$
      congridfactor=congridfac1,$
      shift=shiftfor,rotate=angfor,scale=1./scalerev,$
      cutcentralx=nxfor,cutcentraly=nyfor,$
      _extra=reformkeywords
  endif
endif

end


; ============= test per IDLWAVE Hyper-C =====================

;; ; test forward without affecting timing
;; path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
;; sstfile=path+'sdo2sst/sst_ca8542_0.fits'
;; sdofile=path+'sdo_sst/aia1700_sst.fits'
;; aligndir=path+'align_sdo_sst/'
;; alignlabel='AIA1700-SST8542_0'
;; forfile=aligndir+'align_arrays/forspline-AIA1700-SST8542_0.dat'
;; revfile=aligndir+'align_arrays/revspline-AIA1700-SST8542_0.dat'
;; px_sdo=0.6
;; px_sst=0.057  ; from Luc
;; newfile='/tmp/trial.fits'
;; skip=50
;; alignonefitscube,sdofile,newfile,px1=px_sdo,px2=px_sst,$
;;   sense=1,type=1,$
;;   alignlabel=alignlabel,aligndir=aligndir,$
;;   forfile=forfile,revfile=revfile,$
;;   rottime=1,scaletime=1,skip=skip,$
;;   verbose=1
;; ; check 
;; sstskipfile='/tmp/sstskip.fits'
;; reformcubefile,sstfile,sstskipfile,skip=skip
;; ;; showex,newfile,sstskipfile

;; ; test backward without affecting timing
;; path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
;; sstfile=path+'sdo2sst/sst_ca8542_0.fits'
;; sdofile=path+'sdo_sst/aia1700_sst.fits'
;; aligndir=path+'align_sdo_sst/'
;; alignlabel='AIA1700-SST8542_0'
;; forfile=aligndir+'align_arrays/forspline-AIA1700-SST8542_0.dat'
;; revfile=aligndir+'align_arrays/revspline-AIA1700-SST8542_0.dat'
;; px_sdo=0.6
;; px_sst=0.057  ; from Luc
;; skip=50
;; newfile='/tmp/trial21r.fits'
;; alignonefitscube,sstfile,newfile,px1=px_sdo,px2=px_sst,$
;;   sense=2,type=2,$
;;   alignlabel=alignlabel,aligndir=aligndir,$
;;   forfile=forfile,revfile=revfile,$
;;   rottime=0,scaletime=0,skip=skip,$
;;   verbose=1
;; newfile='/tmp/trial12r.fits'
;; alignonefitscube,sdofile,newfile,px1=px_sdo,px2=px_sst,$
;;   sense=2,type=1,$
;;   alignlabel=alignlabel,aligndir=aligndir,$
;;   forfile=forfile,revfile=revfile,$
;;   rottime=0,scaletime=0,skip=skip,$
;;   verbose=1
;; ; check 
;; ;; showex,'/tmp/trial21r.fits','/tmp/trial12r.fits'

; test forward including timing 
path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
sstfile=path+'sdo2sst/sst_ca8542_0.fits'
sdofile=path+'sdo/cubes/aia1700.fits'  ; original SDO fitscube

; ====== get SST timing 
path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
restore,path+'calib_tseries/tseries.8542.08:02:31.calib.sav'
nt_sst=n_elements(time)
timearr_sst=time
for it=0,nt_sst-1 do timearr_sst[it]='2014-06-21 '+time[it]
timearr_sst=anytim2tai(timearr_sst)
; ====== set up aligment
aligndir=path+'align_sdo_sst/'
alignlabel='AIA1700-SST8542_0'
forfile=aligndir+'align_arrays/forspline-AIA1700-SST8542_0.dat'
revfile=aligndir+'align_arrays/revspline-AIA1700-SST8542_0.dat'
px_sdo=0.6
px_sst=0.057  ; from Luc
skip=0 ; you can't have skip with timearrr_out 
; ===== do alignment
newfile='/tmp/trial12f.fits'
alignonefitscube,sdofile,newfile,px1=px_sdo,px2=px_sst,$
  sense=1,type=1,$
  alignlabel=alignlabel,aligndir=aligndir,$
  forfile=forfile,revfile=revfile,$
  rottime=1,scaletime=1,$
  timearr_out=timearr_sst,$  
  verbose=1
; check 
;; showex,newfile,sstfile

end
