; file: ssb2_instruct.pro = IDL pro SSB2 "Chandrasekhar" as instruction
; last: Oct 19 2018  Rob Rutten  WeiHai
; site: https://robrutten.nl/rrweb/rjr-edu/exercises/ssb
; note: should be same as in instruction 
;       there is a STOP after every item; type .c to continue to next  

; ===========================================  START OF MAIN PROGRAM


; ========================= solar spectral distributions

; read file solspect.dat = AQ table solar spectrum
close,3
openr,3,'solspect.dat'
nwav=38
solspec=fltarr(5,nwav)
readf,3,solspec
  wav=solspec[0,*]           ; wavs in micron = 1E-4 cm
  Flines=solspec[1,*]*1E10   ; 1E10 = ergs/ cm2 s ster [1 micron]
  Fcont=solspec[2,*]*1E10    ; flux = F, not {\cal F} = !pi F
  Ilines=solspec[3,*]*1E10   ; otherwise should equal !pi <I>
  Icont=solspec[4,*]*1E10

; plot both intensity and both flux curves
plot,wav,Icont,$
   xrange=[0,2],$
   xtitle='wavelength [micron]',ytitle='solar flux, intensity',$
   title='       solar intensity and flux'

oplot,wav,Ilines,linestyle=2
oplot,wav,Fcont
oplot,wav,Flines,linestyle=2
print,' max(Icont) =',max(Icont),' at wav = ',wav(where(Icont eq max(Icont)))

STOP

; same per frequency bandwidth
c=2.99792D14      ; light speed in micron/sec in double precision
conv=wav*wav/c    ; from per wavelength to per frequency
plot,wav<2,Icont*conv,$
  xrange=[0,2],$
  xtitle='wavelength [micron]',ytitle='solar flux, intensity',$
   title='       per frequency bandwidth'
oplot,wav,Ilines*conv,linestyle=2
oplot,wav,Fcont*conv
oplot,wav,Flines*conv,linestyle=2
print,' max(Icont*conv) =',max(Icont*conv),' at wav = ',$
  wav(where(Icont*conv eq max(Icont*conv)))

; ====================== Planck overlays and brightness temperature

STOP

; overplot Planck function
temp=6300
plot,wav,Icont,$
  xrange=[0,2],$
  xtitle='wavelength [micron]',ytitle='solar intensity, Planck',$
  title='      comparison with Planck law'
oplot,wav,planck(temp,wav),linestyle=2
xyouts,wav[25],planck(temp,wav[25]),temp

STOP

; plot brightness temperature
plot,wav,brighttemp(wav,Icont),$
  xrange=[0,2],/ynozero,$
  xtitle='wavelength [micron]',ytitle='brightness temperature',$
  title='       brightness temperature'

; ========================== H-minus extinction

STOP

; plot H- extinction against wavelength at FALC h=0
ext=fltarr(nwav)
for i=0,nwav-1 do ext[i]=exthmin(wav[i]*1E4,6520,7.7E13)  ; FALC table
plot,wav,ext,$
  xrange=[0,3],/ynozero,$
  xtitle='wavelength [micron]',ytitle='H-min extinction [cm2/H-atom]',$
  title='       H-min extinction'

STOP

; look-alike plot to brightness temperature
plot,wav,1./ext,$
  xrange=[0,2],/ynozero,$
  xtitle='wavelength [micron]',ytitle='inverse H-min extinction',$
  title='       H-min transmittance'
    ; doesn't fit below 500 nm since no lines and bf edges

STOP

; ========================= apply to FALC

; read FALC
nh=80
readcol,'falc.dat',$
      h,tau5,colm,temp,vturb,nhyd,nprot,nel,ptot,pgasplot,dens,$
      skipline=4

; plot H-min(h) at 500 nm
wl=5000.
ext=exthmin(wl,temp,nel)*(nhyd-nprot)
plot,h,ext,/ylog,$
  xtitle='height [km]',ytitle='extinction [cm-1]',$
  title='       H-min + Thomson'

STOP

; add Thomson scattering
thomson=0.6648E-24*nel
oplot,h,thomson,linestyle=2
ext=ext+thomson
oplot,h,ext
xyouts,-250,1E-13,'Thomson'
xyouts,200,1E-7,'total'
xyouts,1200,1E-16,'H-min'

STOP

; ===========================  integrate to tau scale

; compute and plot tau at 500 nm, compare with FALC tau5
tau=fltarr(nh)
for ih=1,nh-1 do tau[ih]=tau[ih-1]+$
   0.5*(ext[ih]+ext[ih-1])*(h[ih-1]-h[ih])*1E5
plot,h,tau,/ylog,$
  xtitle='height [km]',ytitle='tau at 500 nm',$
  title='       tau scales'
oplot,h,tau5,linestyle=2 
xyouts,500,1E-7,'computed'
xyouts,1000,1E-4,'FALC'

STOP

; ======================== emergent intensity at 0.5 micron
wl=0.5
@emergint.pro
print,' computed intensity =',int
print,' observed intensity =',icont[where(wl eq wav)]
print,' mean formation h =',hmean
plot,h,contfunc/max(contfunc),$
   xrange=[-100,300],yrange=[0,1.1],$
   xtitle='height [km]',ytitle='contribution function',$
   title='       contribution functions'
print,' max contfunc at h = ',h[where(contfunc eq max(contfunc))]

STOP

; ======================== overplot at 1, 1.6, 5 micron
wl=1.0
@emergint.pro
oplot,h,contfunc/max(contfunc),linestyle=2
wl=1.6
@emergint.pro
oplot,h,contfunc/max(contfunc),linestyle=2
wl=5.0
@emergint.pro
oplot,h,contfunc/max(contfunc),linestyle=2
xyouts,-80,0.8,'1.6'
xyouts,-50,1.05,'0.5'
xyouts,0,1.05,'1'
xyouts,50,1.05,'5 micron'

STOP

; ======================== emergent intensity spectrum
comspec=fltarr(nwav)
for iw=0,nwav-1 do begin
  wl=wav[iw]
  @emergint.pro
  comspec[iw]=int
endfor
plot,wav,comspec,$
   xrange=[0,2],yrange=[0,5E10],$
   xtitle='wavelength [micron]',ytitle='solar intensity',$
   title='       intensity comparison'
oplot,wav,icont,linestyle=2
xyouts,0.6,4.3E10,'computed'
xyouts,0.35,0.8E10,'observed'

STOP

plot,wav,comspec,/ylog,$
   xtitle='wavelength [micron]',ytitle='solar intensity',$
   title='       full spectrum'
oplot,wav,icont,linestyle=2
xyouts,0.8,3E10,'computed'
xyouts,0.3,0.5E9,'observed'

STOP

plot,wav,comspec/icont,$
  yrange=[0.5,1.5],$
   xtitle='wavelength [micron]',ytitle='computed/observed intensity',$
   title='       ratio'

STOP


; ========================= emergent intensity as function of mu

plot,wav,comspec,$
   xrange=[0,2],yrange=[0,5E10],$
   xtitle='wavelength [micron]',ytitle='solar intensity',$
   title='       mu-dependent intensity'
intmu=fltarr(10,nwav)
for imu=0,9 do begin
  mu=0.1*imu+0.1               ; mu=0.1, 0.2, ..., 1.0
  comspec=fltarr(nwav)
  for iw=0,nwav-1 do begin
    wl=wav[iw]
    @emergintmu.pro
    intmu[imu,iw]=int
  endfor
  oplot,wav,intmu[imu,*]
  xyouts,0.8,4E10,'!4l!3 = 0.1, 0.2, ...1.0'   ; Greek mu
endfor

STOP

; =============================== limb darkening

;limb darkening against mu
mu=indgen(10)/10.+0.1       ; mu=0.1, 0.2, ..., 1.0
plot,mu,intmu(*,where(wav eq 0.5))/max(intmu(*,where(wav eq 0.5))),$
  xrange=[1,0.1],xstyle=1,yrange=[0,1.1],ystyle=1,$
  xtitle='viewing angle mu',ytitle='limb darkening',$
  title='       limb darkening mu'
oplot,mu,intmu[*,where(wav eq 1.0)]/max(intmu[*,where(wav eq 1.0)])
oplot,mu,intmu[*,where(wav eq 2.0)]/max(intmu[*,where(wav eq 2.0)])
oplot,mu,intmu[*,where(wav eq 5.0)]/max(intmu[*,where(wav eq 5.0)])
xyouts,0.9,0.2,'!4k!3 = 0.5, 1.0, 2.0, 5.0 micron'   ; Greek lambda

STOP

; limb darkening against r/R
sintheta=sqrt(1.-mu*mu)
plot,sintheta,intmu[*,where(wav eq 0.5)]/max(intmu[*,where(wav eq 0.5)]),$
  xrange=[0,0.99],xstyle=1,yrange=[0,1.1],ystyle=1,$
  xtitle='r/R',ytitle='limb darkening',$
   title='       limb darkening r/R'
oplot,sintheta,intmu[*,where(wav eq 1.0)]/max(intmu[*,where(wav eq 1.0)])
oplot,sintheta,intmu[*,where(wav eq 2.0)]/max(intmu[*,where(wav eq 2.0)])
oplot,sintheta,intmu[*,where(wav eq 5.0)]/max(intmu[*,where(wav eq 5.0)])
xyouts,0.1,0.2,'!4k!3 = 0.5, 1.0, 2.0, 5.0 micron'    ; Greek lambda

STOP

; ======================== emergent flux integration

; trapezoidal integration neglecting mu=0
fluxspec=fltarr(nwav)
for imu=1,9 do begin
  mu=0.1*imu+0.1             ; mu=0.1, 0.2, ..., 1.0
  for iw=0,nwav-1 do begin
    fluxspec[iw]=fluxspec[iw]+0.5*mu*(intmu[imu,iw]+intmu[imu-1,iw])*0.1
  endfor
endfor
fluxspec=2*fluxspec   ; no !pi, AQ has flux F, not {\cal F}
plot,wav,fluxspec,$
  xrange=[0,2],yrange=[0,5E10],$
  xtitle='wavelength [micron]',ytitle='solar flux',$
   title='       bad flux integration'
oplot,wav,Fcont,linestyle=2
xyouts,0.5,4E10,'computed'
xyouts,0.35,1E10,'observed'
xyouts,1.2,3E10,'TRAPEZOIDAL'

STOP

; ===== three-point Gaussian integration intensity -> flux
; abscissae + weights n=3 Abramowitz & Stegun page 916
xgauss=[-0.7745966692,0.0000000000,0.7745966692]
wgauss=[ 0.5555555555,0.8888888888,0.5555555555]
fluxspec=fltarr(nwav)
intmu=fltarr(3,nwav)
for imu=0,2 do begin
  mu=0.5+xgauss[imu]/2.   ; rescale xrange [-1,+1] to [0,1]
  wg=wgauss[imu]/2.       ; weights add up to 2 on [-1,+1]
  for iw=0,nwav-1 do begin
    wl=wav[iw]
    @emergintmu.pro       ; old trapezoidal integration I(0,mu)
    intmu[imu,iw]=int
    fluxspec[iw]=fluxspec[iw]+wg*intmu[imu,iw]*mu
  endfor
endfor
fluxspec=2*fluxspec    ; no !pi, AQ has flux F, not {\cal F}
plot,wav,fluxspec,$
  xrange=[0,2],yrange=[0,5E10],$
  xtitle='wavelength [micron]',ytitle='solar flux',$
  title='       solar flux comparison'
oplot,wav,Fcont,linestyle=2
xyouts,0.5,4E10,'computed'
xyouts,0.35,1E10,'observed'
xyouts,1.2,3E10,'GAUSSIAN'

end

