; file: ssa3_instruct.pro = IDL pro SSA3  "Marcel Minnaert" as instruction
; last: Oct 19 2018  Rob Rutten  WeiHai
; site: https://robrutten.nl/rrweb/rjr-edu/exercises/ssa
; note: should be same as in instruction; (..) => [..] done 
;       there is a STOP after every item; type .c to continue to next  


; Planck function
;----------------
function planck,temp,wav
  ; compute Planck function in cgs: erg / cm2 sec [delta lambda=1cm] ster
  ; input: temp = temperature (K)
  ; wav = wavelength in cm (1cm = 10^8 Angstrom)

; physics constants
k=1.38065D-16         ; Boltzmann constant (erg K; double precision)
h=6.62607D-27         ; Planck constant (erg s)
c=2.99792D10          ; velocity of light (cm/s)

blambda = 2*h*c^2/(wav^5*(exp(h*c/(wav*k*temp))-1))  ; cgs units
return,blambda
end


; plot Planck functions double log
;---------------------------------
pro planckdouble
wav=10^(indgen(101)/10.-8.)  ; 10^-8 cm to 1 m
temp=10^(indgen(61)/10.)     ; 1 K to 10^6 K
wav=1.d0*wav
b=wav
for i=0,100 do b[i]=planck(1E6,wav[i])
plot,wav,b,xtitle='wavelength [cm]',ytitle='Planck function',$
  xmargin=[15,5],$
  charsize=1.2,$
  /xlog,/ylog
for t=0,60,5 do begin
  for i=0,100 do b[i]=planck(temp[t],wav[i])
  oplot,wav,b
endfor
xyouts,4E-2,1E-20,'1 K'
xyouts,2E-8,1E27,'1 000 000 K'
end


; SS line profile
;----------------
function profile,a,tau0,u
  ; return a Schuster-Schwarzschild profile
  ; input: a = damping parameter
  ;       tau0 = SS layer thickness at line center
  ;       u = wavelength array in Doppler units
  ; output: int = intensity array 
Ts=5700
Tl=4200
wav=5000.D-8
int=u
usize=SIZE(u)
for i=0,usize[1]-1 do begin
   tau=tau0 * VOIGT(a,abs(u[i])) 
   int[i]=PLANCK(Ts,wav) * exp(-tau) + PLANCK(Tl,wav)*(1.-exp(-tau))
endfor
return,int
end


; 
; ==================================== START IDL MAIN SESSION 
;

  ; plot Planck functions
wav=indgen(100)*200.+1000.    ; produces wav(0,...99) = 1000 - 20800 K
print,wav                     ; check that
b=wav                         ; declare float array of the same size
for i=0,99 do b[i]=planck(8000,wav[i]*1E-8)
plot,wav,b,$
  xtitle='wavelength [Angstrom]',ytitle='Planck function',$
  title='           Planck'


for T=8000,5000,-200 do  begin    ; step from 8000 down to 5000
  for i=0,99 do b[i]=planck(T,wav[i]*1E-8)
  oplot,wav,b        ; overplots extra curves in existing graph
endfor               ; begin...end sequences can't go on command line

STOP          ; use .con to continue


  ; emergent intensity homogeneous layer against tau
B=2.
tau=indgen(101)/10.+0.01     ; tau = 0.1-10
int=tau
for I0=4,0,-1 do begin
  for i=0,100 do int[i]=I0 * exp(-tau[i]) + B*(1-exp(-tau[i]))
  if i0 eq 4 then $
    plot,tau,int,$
      xtitle='tau',ytitle='intensity',$
      title='           emergent intensity'
  if i0 ne 4 then oplot,tau,int
endfor

STOP

  ; Voigt profile
u=indgen(201)/10.-10.   ; u = -10 to 10 in 0.1 steps
vau=u                   ; declare same-size array
a=0.1                   ; damping parameter
for i=0,200 do vau[i]=VOIGT(a,abs(u[i]))
plot,u,vau,yrange=[0,1],$
  xtitle='u',ytitle='Voigt function',$
  title='           Voigt'

STOP

  ; solar line profile
Ts=5700                ; solar surface temperature
Tl=4200                ; solar T-min temperature = `reversing layer'
a=0.1                  ; damping parameter
wav=5000.D-8           ; wavelength in cm
tau0=1                 ; reversing layer thickness at line center
u=indgen(201)/10.-10.  ; u = -10 to 10 in 0.1 steps
int=u                  ; declare array 
for i=0,200 do begin
   tau=tau0 * VOIGT(a,abs(u[i])) 
   int[i]=PLANCK(Ts,wav) * exp(-tau) + PLANCK(Tl,wav)*(1.-exp(-tau))
endfor
plot,u,int,$
  xtitle='wavelength',ytitle='intensity',$
  title='           emergent line profile'

STOP

  ; multiple profiles with variation in tau0
tau0=[0.01, 0.05, 0.1, 0.5, 1, 5, 10, 50, 100]
for itau=0,8 do begin
  for i=0,200 do begin
    tau=tau0[itau] * VOIGT(a,abs(u[i])) 
    int[i]=PLANCK(Ts,wav) * exp(-tau) + PLANCK(Tl,wav)*(1.-exp(-tau))
  endfor
  oplot,u,int
endfor

STOP

  ; relative profiles with variation in tau0 at three wavelengths
for iwav=1,3 do begin
  wav=(iwav^2+1)*1.D-5             ; wav = 2000, 5000, 10000 Angstrom
  for itau=0,8 do begin
    for i=0,200 do begin
      tau=tau0[itau] * VOIGT(a,abs(u[i])) 
      int[i]=PLANCK(Ts,wav) * exp(-tau) + PLANCK(Tl,wav)*(1.-exp(-tau))
    endfor
  int=int/int(0)               ; convert into relative intensity
  if iwav eq 1 and itau eq 0 then $
      plot,u,int,$
        xtitle='wavelength',ytitle='intensity',$
        title='           relative profiles' 
  if iwav eq 1 and itau gt 0 then oplot,u,int
  if iwav eq 2 then oplot,u,int,linestyle=1       ; dotted
  if iwav eq 3 then oplot,u,int,linestyle=4       ; dash dot dot dot 
  endfor
endfor

STOP

  ; test function profile
u=indgen(1001)/2.5-200.     ; u = -200 to +200 with delta u = 0.4
a=0.01
tau0=1e4
int=profile(a,tau0,u)
plot,u,int,$
  xtitle='wavelength',ytitle='intensity',$
  title='           test profile'

STOP

  ; equivalent width
reldepth=(int(0)-int)/int(0)
plot,u,reldepth,$
  xtitle='wavelength',ytitle='relative depth',$
  title='           linedepth profile'
eqw=total(reldepth)*0.4
print,'equivalent width = ',eqw

STOP

  ; curve of growth
tau0=10^(indgen(61)/10.-2.)    ; tau0 = 10^-2 to 10^4
eqw=tau0
for i=0,60 do begin
  int=profile(a,tau0[i],u)
  reldepth=(int[0]-int)/int[0]
  eqw[i]=total(reldepth)*0.4
endfor
plot,alog10(tau0),alog10(eqw),$
   xtitle='log tau0',ytitle='log equivalent width',$
   title='           curve of growth'

STOP

  ; emission line curve of growth (set Tl=7000 in function profile)
tau0=10^(indgen(61)/10.-2.)    ; 10^-2 to 10^4, 0.1 steps in the log
eqw=tau0
for i=0,60 do begin
  int=profile(a,tau0[i],u)
  reldepth=(int[0]-int)/int[0]
  eqw[i]=total(reldepth)*0.4
endfor
plot,alog10(tau0),alog10(abs(eqw)),$
  xtitle='log tau0',ytitle='log equivalent width',$
  title='           emission line curve of growth'

end



