; file: ssa2_instruct.pro = IDL pro SSA2 "Cecilia Payne" as instruction 
; last: Oct 19 2018  Rob Rutten  WeiHai
; note: this should be the same as in the instruction
;       there is a STOP after every item; type .c to continue to next  
; site: https://robrutten.nl/rrweb/rjr-edu/exercises/ssa
; beware: I converted old (arr index) notation into proper [array index]
;         but there may be errors left (also in the instruction)

;================================================================ PARTFUNC_E
function partfunc_E, temp
  ; compute partition functions Schadee element
  ; input: temp (K)
  ; output: fltarr(4) = partition functions U1,..,U4

u=fltarr(4)                
chiion=[7,16,31,51]   ; Schadee element ionization energies     
k=8.61734D-5          ; Boltzmann constant in ev/deg               
for r = 0,3 do $           
  for s=0, chiion[r]-1 do u[r] = u[r] + exp(-s/(k*temp))
return,u     

end


;=================================================================== BOLTZ_E
function boltz_E,temp,r,s
 ; compute Boltzmann population for level r,s of Schadee element E 
 ; input: temp (temperature, K)
 ;        r (ionization stage nr, 1 - 4 where 1 = neutral E) 
 ;        s (level nr, starting at s=1)
 ; output: relative level population n_(r,s)/N_r
u=partfunc_E(temp)
keV=8.61734D-5         ; Boltzmann constant in ev/deg               
relnrs = 1./u[r-1]*exp(-(s-1)/(keV*temp))
return, relnrs 
end


;==================================================================== SAHA_E
function saha_E,temp,elpress,ionstage
  ; compute Saha population fraction N_r/N for Schadee element E
  ; input: temperature, electron pressure, ion stage

; physics constants
kerg=1.380658D-16        ; Boltzmann constant (erg K; double precision)
kev=8.61734D-5           ; Boltzmann constant (eV/deg)
h=6.62607D-27            ; Planck constant (erg s)
elmass=9.10939D-28       ; electron mass (g)

; kT and electron density
kevT=kev*temp
kergT=kerg*temp
dens=elpress/kergT

chiion=[7,16,31,51]      ; ionization energies for element E
u=partfunc_E(temp)       ; get partition functions u[0]...u[3)
u=[u,2]                  ; add fake fifth value to get N_4 too
sahaconst=(2*!pi*elmass*kergT/(h*h))^1.5 * 2./dens
nstage=dblarr(5)         ; double-precision float array
nstage[0]=1.             ; relative fractions only (no abundance)
for r=0,3 do $    
  nstage[r+1] = nstage[r]*sahaconst*u[r+1]/u[r]*exp(-chiion[r]/kevT)
ntotal=total(nstage)     ; sum all stages = element density
nstagerel=nstage/ntotal  ; fractions of element density

return,nstagerel[ionstage-1]
end


;================================================================ SAHABOLT_E
function sahabolt_E,temp,elpress,ion,level
  ; compute Saha-Boltzmann populaton n_(r,s)/N for level of E
  ; input: temperature, electron pressure, ionization stage, level nr
return, saha_E(temp,elpress,ion) * boltz_E(temp,ion,level) 
end


;================================================================ SAHABOLT_H
function sahabolt_H,temp,elpress,level
  ; compute Saha-Boltzmann population n_(1,s)/N_H for hydrogen level
  ; input: temperature, electron pressure, level number

; physics constants
kerg=1.380658D-16        ; Boltzmann constant (erg K; double precision)
kev=8.61734D-5           ; Boltzmann constant (eV/deg)
h=6.62607D-27            ; Planck constant (erg s)
elmass=9.10939D-28       ; electron mass (g)

; kT and electron density
kevT=kev*temp
kergT=kerg*temp
dens=elpress/kergT

; energy levels and weights for hydrogen
nrlevels=100                  ; reasonable partition function cut-off value
g=intarr(2,nrlevels)          ; declaration weights (too many for proton)
chiexc=fltarr(2,nrlevels)     ; declaration excitation energies (idem)  
for s=0,nrlevels-1 do begin    
  g[0,s]=2*(s+1)^2
  chiexc[0,s]=13.598*(1-1./(s+1)^2)
endfor
g[1,0]=1                      ; statistical weight free proton
chiexc[1,0]=0.                ; excitation energy proton ground state

; partition function
u=fltarr(2)                   
u[0]=0
for s=0,nrlevels-1 do u[0]=u[0]+ g[0,s]*exp(-chiexc[0,s]/kevT)  
u[1]=g[1,0]

; Saha
sahaconst=(2*!pi*elmass*kergT/(h*h))^1.5 * 2./dens
nstage=dblarr(2)              ; double-precision float array
nstage[0]=1.                  ; relative fractions only
nstage[1] = nstage[0] * sahaconst * u[1]/u[0] * exp(-13.598/kevT)
ntotal=total(nstage)          ; sum both stages = total hydrogen density

; Boltzmann 
nlevel = nstage[0]*g[0,level-1]/u[0]*exp(-chiexc[0,level-1]/kevT)
nlevelrel=nlevel/ntotal       ; fraction of total hydrogen density

;stop                         ; in for parameter inspection
return,nlevelrel
end 


;=============================================================== PARTFUNC_Ca
function partfunc_Ca, temp
  ; compute partition functions Schadee element = quick fake for Ca
  ; input: temp (K)
  ; output: fltarr(4) = partition functions U1,..,U4

u=fltarr(4)                
chiion=[6.11,11.87,50.91,67.15]   ; Schadee element ionization energies     
k=8.61734D-5                      ; Boltzmann constant in ev/deg
for r = 0,3 do $           
  for s = 0, chiion[r]-1 do u[r]=u[r] + exp(-s/(k*temp))
return,u     

end


;================================================================== BOLTZ_Ca
function boltz_Ca,temp,r,s
 ; compute Boltzmann population for level r,s of Schadee element E => Ca 
 ; input: temp (temperature, K)
 ;        r (ionization stage nr, 1 - 4 where 1 = neutral E) 
 ;        s (level nr, starting at s=1)
 ; output: relative level population n_(r,s)/N_r
u=partfunc_Ca(temp)
keV=8.61734D-5                    ; Boltzmann constant in ev/deg
relnrs = 1./u[r-1]*exp(-(s-1)/(keV*temp))
return, relnrs 
end


;=================================================================== SAHA_Ca
function saha_Ca,temp,elpress,ionstage
  ; compute Saha population fraction N_r/N for Schadee element E => Ca
  ; input: temperature, electron pressure, ion stage

; physics constants
kerg=1.380658D-16        ; Boltzmann constant (erg K; double precision)
kev=8.61734D-5           ; Boltzmann constant (eV/deg)
h=6.62607D-27            ; Planck constant (erg s)
elmass=9.10939D-28       ; electron mass (g)

; kT and electron density
kevT=kev*temp
kergT=kerg*temp
dens=elpress/kergT

chiion=[6.11,11.87,50.91,67.15]
u=partfunc_Ca(temp)
u=[u,2]                  ; add fake fifth value to get N_5 too
sahaconst=(2*!pi*elmass*kergT/(h*h))^1.5 * 2./dens
nstage=dblarr(5)         ; double-precision float array
nstage[0]=1.             ; relative fractions only (no abundance)
for r=0,3 do $    
  nstage[r+1] = nstage[r]*sahaconst*u[r+1]/u[r]*exp(-chiion[r]/kevT)
ntotal=total(nstage)     ; sum all stages = element density
nstagerel=nstage/ntotal  ; fractions of element density

return,nstagerel[ionstage-1]
end

;========================================================= SAHABOLT_Ca
function sahabolt_Ca,temp,elpress,ion,level
  ; compute Saha-Boltzmann populaton n_(r,s)/N for level of E => Ca
  ; input: temperature, electron pressure, ionization stage, level nr
return, saha_Ca(temp,elpress,ion) * boltz_Ca(temp,ion,level) 
end



;========================================================== MAIN PART


  ; Schadee partition functions for element E
u=fltarr(4)                ; declare 4-element float array; set values 0
chiion=[7,16,31,51]        ; Schadee ionization energies into integer array
k=8.61734D-5               ; Boltzmann constant in eV/deg (double precision)
temp=5000.                 ; the decimal point makes it a float
for r = 0,3 do $           ; a $ sign extends a command to the next line
  for s = 0, chiion[r]-1 do u[r]=u[r] + exp(-s/(k*temp))
print,u                    ; print resulting values for U[0]...U[3]


STOP    ; comment out when done with experimenting


  ; Payne curves for element E
temp=1000*indgen(31)       ; make array 0,...,30000 in steps of 1000 K
print,temp                 ; check
pop=fltarr(5,31)           ; declare float array for n(r,T)
for T=1,30 do $            ; $ continues statement to next line
  for r=1,4 do pop[r,T]=sahabolt_E(temp[T],131.,r,1)
plot,temp,pop[1,*],/ylog,yrange=[1E-3,1.1],$
  xtitle='temperature',ytitle='level population',$
  title='         S-B curves for E'
oplot,temp,pop[2,*]        ; first ion stage in the same graph
oplot,temp,pop[3,*]        ; second ion stage            
oplot,temp,pop[4,*]        ; third ion stage            
xyouts,1.5E3,1.2,'E I'
xyouts,6.0E3,1.2,'E II'
xyouts,1.3E4,1.2,'E III'
xyouts,2.3E4,1.2,'E IV'
xyouts,2.3E4,0.5,'0 eV'
xyouts,2.3E4,0.16,'1 eV'
xyouts,2.3E4,0.05,'3 eV'

STOP    ; comment out when done with experimenting

for T=1,30 do $            ; repeat for s=2 (excitation energy = 1 eV)
  for r=1,4 do pop[r,T]=sahabolt_E(temp[T],131.,r,2)
oplot,temp,pop[1,*],linestyle=1
oplot,temp,pop[2,*],linestyle=1
oplot,temp,pop[3,*],linestyle=1
oplot,temp,pop[4,*],linestyle=1

STOP    ; comment out when done with experimenting

for T=1,30 do $             ; repeat for s=4 (excitation energy = 3 eV)
  for r=1,4 do pop[r,T]=sahabolt_E(temp[T],131.,r,4)
oplot,temp,pop[1,*],linestyle=4
oplot,temp,pop[2,*],linestyle=4
oplot,temp,pop[3,*],linestyle=4
oplot,temp,pop[4,*],linestyle=4

STOP    ; comment out when done with experimenting



; ================= Saha Boltz Ca/H

temp=indgen(191)*100.+1000.           ; T = 1000-20000 in delta T = 100
CaH = temp                            ; declare ratio array
Caabund=2.E-6                         ; A_Ca = N_Ca / N_H
for i=0,190 do begin
  NCa = sahabolt_Ca(temp[i],1e2,2,1)
  NH = sahabolt_H(temp[i],1e2,2)
  CaH[i]=NCa*Caabund/NH
endfor           
plot,temp,CaH,/ylog,$
  xtitle='temperature',ytitle='Ca II K / H alpha',$
  title='        K/Halpha population ratio'

print,'Ca/H ratio at 5000~K = ',CaH(where(temp eq 5000))

STOP    ; comment out when done with experiments

  ; temperature sensitivity CaIIK and Halpha
temp=indgen(101)*100.+2000.            ; T = 2000-12000, delta T = 100
dNCadT = temp                          ; declare array
dNHdt = temp                           ; declare array
dT=1.
for i=0,100 do begin                  
    NCa = sahabolt_ca(temp[i],1e2,2,1)        ; Ca ion ground state
    Nca2 = sahabolt_ca(temp[i]-dT,1e2,2,1)    ; idem dT cooler
    dNCadT[i]= (NCa - NCa2)/NCa/dT            ; fractional diff quotient
    NH = sahabolt_H(temp[i],1e2,2)            ; H atom 2nd level
    NH2 = sahabolt_H(temp[i]-dT,1e2,2)        ; idem dT cooler
    dNHdT[i] = (NH-NH2)/NH/dT                 ; fractional diff quotient
endfor           
plot,temp,abs(dNHdT),/ylog,yrange=[1E-5,1],$
    xtitle='temperature',ytitle='abs d n[r,s] / n[r,s]',$
    title='         temperature sensitivity'
oplot,temp,abs(dNCadT),linestyle=2                 ; Ca curve dashed

STOP    ; comment out when done with experiments

  ; recompute as arrays and overplot relative populations
NCa=temp
NH=temp
for i=0,100 do begin                  
    NCa[i] = sahabolt_ca(temp[i],1e2,2,1)        ; Ca ion ground state
    NH[i] = sahabolt_H(temp[i],1e2,2)            ; H atom 2nd level
endfor           
oplot,temp,NH/max(NH)
oplot,temp,NCa/max(NCa),linestyle=2              ; Ca curve again dashed


STOP    ; comment out when done with experimenting

  ; hydrogen ionization as function of temperature
temp=indgen(191)*100.+1000. 
nH=temp
for i=0,190 do nH[i]=sahabolt_H(temp[i],1e2,1)
plot,temp,nH, $
   xtitle='temperature',ytitle='neutral hydrogen fraction',$
    title='         hydrogen ionization'


end

