; file: makedepthfile.pro
; init: Feb 27 2018  Rob Rutten  Deil
; last: Mar 11 2018  Rob Rutten  Deil

;+

pro makedepthfile,filein,fileout,corerange,contrange,$
  trimbox=trimbox,trange=trange,verbose=verbose

 ; PURPOSE:
 ;   special: produce line depth file for 10830
 ;   
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   filein = string for 10830 mwav file
 ;   fileout = string for output file 
 ;   corerange = wavelength band in line core to average over, in iwav indices
 ;   contrange = wavelength band in continuum to average over, in iwav indices
 ;   
 ; OPTIONAL INPUTS:
 ;   trange: use for shorter-duration tests
 ;   trimbox: only used for minmax in integer range check at end
 ; 
 ; OUTPUTS:
 ;   fileout   
 ;   
 ; HISTORY:
 ;   Feb 27 2018 RR: start
;-

; answer no-parameter query 
if (n_params(0) lt 4) then begin
  sp,makedepthfile
  return
endif

; defaults for keywords
if (n_elements(trimbox) eq 0) then trimbox=-1
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(verbose) eq 0) then verbose=0

; set wall-clock timer (seconds)
timestart=systime(1) 

; print pacifier
if (verbose ne 0) then print,' ===== makedepthfile started at UT = '+$
  anytim2utc(timestart,/vms)

; set swap_endian for this computer 
;RR l = little, b = big; fits = big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'
if (endian eq 'l') then swap_endian=1 else swap_endian=0

; get nt from header info
inhead=headfits(filein)
nz=fxpar(inhead,'naxis3')
nwavs=fxpar(inhead,'nrwavs')
nt=nz/nwavs

; open assoc filein 
fits_openassoc,filein,1,nx,ny,nz,bitpix1,headeroff1,swap1

; default trimbox = full image
if (trimbox[0] eq -1) then trimbox=[0,0,nx-1,ny-1]

; set partial time sequence (e.g, for quickie tests) 
if (trange[1] ne -1) then begin
  tstart=trange[0]
  tend=trange[1]
endif else begin
  tstart=0
  tend=nt-1
endelse

; define inassoc = per it full spectral image range
inassoc=assoc(1,intarr(nx,ny,nwavs),headeroff1)

; make minimal fits output header
outtype=2
ntout=tend-tstart+1
mkhdr,header_out,outtype,[nx,ny,ntout]
header_outsize=(1+fix(n_elements(header_out)/36.))*2880  

; open output fxile for assoc, write header_out
get_lun, unit_out
endian_out='b'
openw,unit_out,fileout,swap_endian=(endian ne endian_out)
outassoc=assoc(unit_out,intarr(nx,ny),header_outsize)
rec=assoc(unit_out, bytarr(header_outsize))
rec[0]=byte(header_out)

; big loop over time steps
for it=tstart,tend do begin
  print,' ----- makedepthfile: it ='+trimd(it)

; sum core and cont wavelengths
  coresum=fltarr(nx,ny)
  contsum=fltarr(nx,ny)
  speccube=inassoc[it]
  for iw=corerange[0],corerange[1] do coresum=coresum+speccube[*,*,iw]
  for iw=contrange[0],contrange[1] do contsum=contsum+speccube[*,*,iw]
  coresum=coresum/(corerange[1]-corerange[0]+1)
  contsum=contsum/(contrange[1]-contrange[0]+1)

; make depth image
  amp=(contsum-coresum)/contsum
;;;  depth=1.-amp  ; true line depth
  depth=amp  ; I prefer bright-bright in 10830-304 correlation
  depth=fix(depth*2.E4)  ; only appears as absorption line 

; write output image
  outassoc[it]=depth

endfor ; end loop over time steps

; get minmax output file for check on integer excess
minmaxout=minmaxcubefile(fileout,trimbox=trimbox)
if (abs(minmaxout[0]) ge 32000 or abs(minmaxout[1]) ge 32000) then begin
  print,' ##### makedepthfile warning: output exceeds integer range'
  print,' ----- trimbox minmaxout ='+trimd(minmaxout)
endif
if (verbose ne 0) then print,' ----- trimbox minmaxout ='+trimd(minmaxout) 

; free the input and output files
free_lun,1,2,unit_out

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
if (verbose ne 0) then print,' ===== makedepthfile took '+timelaps+' minutes'

end



; =============== main for testing per IDLWAVE H-c ======================

; DST Schad data
cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/mxis'
filein='mxis_mwseq_10830_20170614_104106.fits'
fileout='mxis_depth_10830.fits'
trimbox=[200,200,700,700]
corerange=[52,54]
contrange=[32,45]
;;; trange=[0,20]
makedepthfile,filein,fileout,corerange,contrange,$
  trimbox=trimbox,trange=trange,verbose=1

; check
showex,fileout,trimbox=trimbox

;; ; check = full monty
;; showex,fileout,$
;;   'widwid_ha_ca.fits',$
;;   'ibis_width_kevin_6563_0.9.fits',$
;;   'ibis_width_kevin_8542_0.6.fits',$
;;   'ibis_mwseq_6563_20170614_164052_alrev_ip_ibis.fits',$
;;   'ibis_mwseq_8542_20170614_164052_alrev_ip_ibis.fits',$
;;    /allsdo,sdodir='./',/plotscatter,$
;;     mwfwavsfiles=['ibis_wavs_6563_20170614_164052.txt',$
;;                   'ibis_wavs_8542_20170614_164052.txt',$
;;                   'mxis_wavs_10830_20170614_104106.txt'],$
;;     trimbox=[200,200,700,700]

end
