; file: ibisfile2wav.pr0
; init: Jan 16 2018  Rob Rutten  Deil
; last: Jan 22 2018  Rob Rutten  Deil

;+

pro ibisfile2wav,ibisfile,wavfile,iwav,trange=trange


 ; NAME:
 ;   ibisfile2wav   
 ;   
 ; PURPOSE:
 ;   distill single-wav fitscube file out of a multi-wav IBIS fits file
 ;     written with dstlib/ibis2cubefiles.pro
 ;
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   ibisfile = string with path+filename input IBIS multi-wav file
 ;   iwav = wavelength index, negative = nwavs - abs(iwav)
 ;
 ; OPTIONAL KEYWORD INPUTS
 ;   trange = [tstart,tend] to limit time sequence   
 ;   
 ; OUTPUT
 ;   wavfile = string with path+filename output single-wav fitscube 
 ;   
 ; HISTORY:
 ;   Jan 15 2018 RR: start
;-

; answer no-parameter query 
if (n_params(0) lt 3) then begin
  sp,ibisfile2wav
  return
endif

; defaults for keywords
if (n_elements(trange) eq 0) then trange=[0,-1]

; pacifier
print,' ----- start producing wavfile = '+wavfile

; read header for dimensions including nwavs (in ibisscans2files.pro files)
header_in=headfits_rr(ibisfile)
nx_in=fxpar(header_in,'naxis1')
ny_in=fxpar(header_in,'naxis2')
nz_in=fxpar(header_in,'naxis3')
nwavs=fxpar(header_in,'nrwavs') ;RR better than CRISP headers
nt_in=nz_in/nwavs
bitpix_in=fxpar(header_in,'bitpix')

; negative iwav = 
if (iwav lt 0) then iwav=nwavs-abs(iwav)

; set swap_endian for this computer 
;RR l = little, b = big; fits = big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'
if (abs(bitpix_in) gt 8) and (endian eq 'l') then $
  swap_endian=1 else swap_endian=0 ; fits files are bigendian

; check
if (bitpix_in ne 16) then begin
  print,' ##### ibisfile2wav abort: bitpix_in not 16 as expected'
  return
endif

; trange >  output nt
if (trange[1] eq -1) then trange[1]=nt_in-1
nt_out=trange[1]-trange[0]+1

; define input assoc
inarr=intarr(nx_in,ny_in)

; inheader size (fits header = Nx36 "card images" = Nx2880 bytes)
sizeinhead=size(inheader)
inheadsize=(1+fix(sizeinhead[1]/36.))*2880

; open infile for assoc
; (cut immediately to reduce array size for too large data cubes)
get_lun, unit_in
openr,unit_in,ibisfile,swap_endian=swap_endian
inassoc=assoc(unit_in,inarr,inheadsize)

; output header
mkhdr,outheader,abs(bitpix_in)/8,[nx_in,ny_in,nt_out]
sizeoutheader=size(outheader)  
; fits header = Nx36 "card images" = Nx2880 bytes
outheadersize=(1+fix(sizeoutheader[1]/36.))*2880

; open outfile for assoc
get_lun, unit_out
openw,unit_out,wavfile,/swap_if_little_endian
outassoc=assoc(unit_out,intarr(nx_in,ny_in),outheadersize)
rec=assoc(unit_out, bytarr(outheadersize))
rec[0]=byte(outheader)

; loop over images
for it=trange[0],trange[1] do begin
  image=inassoc[it*nwavs+iwav]
  outassoc[it-trange[0]]=image
endfor

; free files
free_lun,unit_in,unit_out

; done
print,' ----- wrote wavfile = '+wavfile

end


; =============== main for testing per IDLWAVE H-c ======================

;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/ibisA'
;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/ibisB'
cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/ibisB2sdo'
ibisfile='ibis_mwseq_6563_20170614_164052_alrev_ip.fits'
wavfile='ibis_ha_lc_alrev_ip.fits'
iwav=14

ibisfile2wav,ibisfile,wavfile,iwav,trange=trange

showex,wavfile,trimbox=[200,200,700,700]

end
