; file: ibis2meanprof
; init: Feb 17 2018  Rob Rutten  Deil from sstlib/crips2meanprof.pro 
; last: Feb 17 2018  Rob Rutten  Deil
; site: rridl/dstlib

;+

function ibis2meanprof,ibisfile,wavsfile,ibisdir=ibisdir,itstep=itstep

 ; PURPOSE:
 ;   return mean profile of an iBIS file using assoc
 ; CALL:
 ;   see above
 ; INPUTS:
 ;   ibisfile = string with IBIS filename
 ;   wavsfile = corresponding IDL save file with timing sequence
 ; OPTIONAL INPUTS:
 ;   ibisdir = subdir with ibisfile (default './ibis/'
 ;   itstep = step time steps, > 1 speeds up this program (default 10)
 ; OUTPUTS:
 ;   meanprof = double array delwav, intensity
 ; HISTORY:
 ;   Feb 17 2018 RR: start from sstlib/crips2meanprof.pro
;-

; default keywords
if (n_elements(ibisdir) eq 0) then ibisdir='/ibis/'
if (n_elements(itstep) eq 0) then itstep=10

; set endian for this computer 
;RR l = little, b = big; fits files are big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'

; get line (= wavelength) from file name
line=-1
if (strmatch(ibisdir+ibisfile,'*ibis_mwseq_6563*')) then line=6563
if (strmatch(ibisdir+ibisfile,'*ibis_mwseq_8542*')) then line=8542
if (line eq -1) then begin
  print, ' ##### ibis2meanprof abort: no known line in ibisfile'
  return,0
endif

; get profile sampling wavelength offsets
readcol,ibisdir+wavsfile,mwfwavs 
; line center wavs from MMH table (IBIS gets set to Neckel atlas)
if (line eq 6563) then wavlc=6562.8
if (line eq 8542) then wavlc=8542.1
delwav=mwfwavs-wavlc

; get cube geometry and file datatype from the fits header
  header=headfits_rr(ibisdir+ibisfile)
  headersize=(1+fix(n_elements(header)/36.))*2880
  bitpix=fxpar(header,'bitpix')
  nx=fxpar(header,'naxis1') 
  ny=fxpar(header,'naxis2') 
  nz_file=fxpar(header,'naxis3') 
  nt=nz_file
  nw=fxpar(header,'nrwavs')  ; becomes 0 if not present
  if (nw eq 0) then nw=1 else nt=nz_file/nw
  endian_in='b'  ; fits files are always big endian
; check
  if (nx eq 0 or ny eq 0 or nz_file eq 0) then begin
    print,' ##### reformcubefile abort: no nx, ny, nz in fits header '+infile
    return,-1
  endif

; define assoc input size and type
if (bitpix eq 8) then imscan=bytarr(nx,ny,nw)
if (bitpix eq 16) then imscan=intarr(nx,ny,nw)
if (bitpix eq -32) then imscan=fltarr(nx,ny,nw)

; open assoc to the input images 
get_lun, unit
openr,unit,ibisdir+ibisfile,swap_endian=(endian ne endian_in)
p=assoc(unit,imscan,headersize) 

; do the work
sumprof=fltarr(nw)
for it=0,nt-1,itstep do begin
  scan=p[it]
  prof=total(total(scan,1),1)/(float(nx)*ny)
  sumprof=sumprof+prof
endfor
meanprof=sumprof/fix(nt/itstep+1)

; ready with reading
free_lun,unit

; done
return,[[delwav],[meanprof]]

end

; =============== test per IDLWAVE S-c ==============================

cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/ibis_mxis_2sdo'
ibisdir='./'
outdir='./'
ibisfile='ibis_mwseq_6563_20170614_164052_alrev_ip_ibis.fits'
wavsfile='ibis_wavs_6563_20170614_164052.txt'

mp=ibis2meanprof(ibisfile,wavsfile,ibisdir=ibisdir)
plot,mp[*,0],mp[*,1]

end
