; file: ibis2cubefiles.pro
; init: Dec 27 2017  Rob Rutten  Deil
; last: Jan 22 2018  Rob Rutten  Deil

;+

pro ibis2cubefiles,indir,wav,starttime,outdir=outdir,$
  intscale=intscale,trimbox=trimbox,verbose=verbose,$
  cropbox=cropbox,specfitscube=specfitscube,wavsfile=wavsfile,$
  timesfile=timesfile,wbfitscube=wbfitscube

 ; NAME:
 ;  ibis2cubefiles 
 ;   
 ; PURPOSE:
 ;  rewrite a list of IBIS scan files into specfitscube and other files
 ;   
 ; DESCRIPTION:
 ;  see comments in the pro and usage example below 
 ;   
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   indir = string with directory of input files
 ;   wav = line identifier, 6563 or 8542 
 ;   starttime = number as 164052, second in file name = sequence start
 ;  
 ; OPTIONAL KEYWORD INPUTS:
 ;   outdir = string with directory for output files, default 'ibis/'
 ;   intscale=[multmin,multmax]: multipliers to min and max intensities
 ;     of the first image to use as thresholds in setting the integer scale
 ;     default:[10.0,10.0] (for negative min)
 ;   trimbox = [xmin,ymin,xmax,yma] to select a subfield for the 
 ;     min-max determination (default -1: instead auto-detect cropbox
 ;     within circular field-of-view perimeter)
 ;  verbose  = 1/0: more/less output
 ;
 ; OUTPUTS:
 ;   cropbox = trimbox resulting from auto-detect circular perimeter
 ;   files in outdir = names in specfitscube, wavsfile, timesfile, wbfitscube
 ;  
 ; RESTRICTIONS:
 ;   for cropbox auto-detect the circular perimeter must be maximum black 
 ;     or white where it crosses the image diagonals
 ;   
 ; HISTORY:
 ;   Dec 30 2017 RR: start
;-

; answer no-parameter query 
if (n_params(0) lt 3) then begin
  print,' ibis2cubefiles,indir,wav,starttime,outdir,'
  print,'    + keywords'
  sp,ibis2cubefiles
  return
endif

; defaults for keywords
if (n_elements(trimbox) eq 0) then trimbox=-1
if (n_elements(intscale) eq 0) then intscale=[10.,10.]
if (n_elements(outdir) eq 0) then outdir='ibis/'
if (n_elements(verbose) eq 0) then verbose=0

; set wall-clock timer (seconds)
timestart=systime(1) 

; print pacifier
if (verbose ne 0) then print,' ===== ibis2cubefiles started at UT = '+$
  anytim2utc(timestart,/vms)

; find endian of this computer 
;RR l = little, b = big; fits files are big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'

; make outdir if not yet existing 
spawn,'mkdir -p '+outdir

; get file list defined by starttime and wav  as selectors
files=findfile(indir+'ibis*'+trim(starttime)+'*'+trim(wav)+'.fits*')
nfiles=n_elements(files)

; read first and last headers to define assoc parameters
dummy=fxposit(files[0],1,/headeronly) ;RR position file pointer to ext 1
firsthead=headfits_rr(files[0],exten=1)
nx=fxpar(firsthead,'znaxis1') 
ny=fxpar(firsthead,'znaxis2')
nw=fxpar(firsthead,'znaxis3')
nt=nfiles
nz=long(nt)*nw
xscale=fxpar(firsthead,'xscale')
yscale=fxpar(firsthead,'yscale')
seqstart=fxpar(firsthead,'tstart')
lasthead=headfits_rr(files[nfiles-1],exten=1)
seqend=fxpar(lasthead,'tstart') ; only for the header

; find cadence
taiseqstart=anytim2tai(seqstart)
taiseqend=anytim2tai(seqend)
cadence=(taiseqend-taiseqstart)/(nt-1)
if (verbose ne 0) then print,' ----- mean cadence ='+trimd(cadence,3)+' sec'

; set output file names
date=strmid(seqstart,0,10)
; remove hyphens
date=strsplit(date,'-',/extract)
date=date[0]+date[1]+date[2] 
tail='_'+trim(wav)+'_'+trim(date)+'_'+trim(starttime)
specfitscube='ibis_mwseq'+tail+'.fits'
wavsfile='ibis_wavs'+tail+'.txt'
wbfitscube='ibis_wb'+tail+'.fits'
timesfile='ibis_times'+tail+'.txt'

; ===== write wavelengths file
wavs0=mrdfits(files[0],2,/silent)
writecol,outdir+wavsfile,wavs0,fmt='(3f15.3)'
if (verbose) then print,' ----- wrote wavsfile = '+wavsfile

; ===== write timing file
datetime=strarr(nt)
for it=0,nt-1 do begin
  head=headfits_rr(files[it],exten=1)
  tstart=fxpar(head,'tstart')
  tend=fxpar(head,'tend')
  taistart=anytim2tai(tstart)
  taiend=anytim2tai(tend)
  taimid=(taistart+taiend)/2.
  datetime[it]=anytim2utc(taimid,/ccsds)
endfor
writecol,outdir+timesfile,datetime,fmt='(A)'
if (verbose) then print,' ----- wrote timesfile = '+timesfile

; ===== rewrite all scans into a single fitscube = specfitscube

; auto-find trimbox within the circular field of view perimeter
; (in simple fashion; otherwise install Kevin Reardon's programs)
if (trimbox[0] eq -1) then begin
  scan0=mrdfits(files[0],1,/silent)
  image=scan0[*,*,0]
  diag1=diag_matrix(image)
  diag1=diag1[nx*0.1:nx*0.9]
  diag2=diag_matrix(reverse(image))
  diag2=diag2[nx*0.1:nx*0.9]
  maxcut=[where(diag1 eq max(diag1)),where(diag2 eq max(diag2))]
  mincut=[where(diag1 eq min(diag1)),where(diag2 eq min(diag2))]
; NB!! assumes lower/upper half of circle are pairwise extreme black or white
  if (maxcut[0] lt nx*0.3 and maxcut[1] lt nx*0.3) then begin
    xymin=max(maxcut)+nx*0.1 
    xymax=min(mincut)+nx*0.1
  endif
  if (maxcut[0] gt nx*0.6 and maxcut[1] gt nx*0.6) then begin
    xymin=max(mincut)+nx*0.1  
    xymax=min(maxcut)+nx*0.1 
  endif
  margin=10
  cropbox=[fix(xymin+margin),fix(xymin+margin),$
           fix(xymax-margin),fix(xymax-margin)]
; check
  if (verbose) then begin
    plot,diag1
    oplot,diag2
    sv,image
    cgplots,[cropbox[0],cropbox[2]],[cropbox[1],cropbox[1]],color='red'
    cgplots,[cropbox[0],cropbox[2]],[cropbox[3],cropbox[3]],color='red'
    cgplots,[cropbox[0],cropbox[0]],[cropbox[1],cropbox[3]],color='red'
    cgplots,[cropbox[2],cropbox[2]],[cropbox[1],cropbox[3]],color='red'
  endif
endif else cropbox=trimbox

; set some assoc out parameters
outbitpix=16l
mkhdr,outhead,abs(outbitpix)/8,[nx,ny,nz]
sxaddpar,outhead,'seqstart',seqstart,' UTC start of IBIS sequence'
sxaddpar,outhead,'seqend',seqend,' UTC end of IBIS sequence'
sxaddpar,outhead,'nrscans',nt,' number of IBIS scans'
sxaddpar,outhead,'nrwavs',nw,' number of wavelength samples per scan' 
sxaddpar,outhead,'xscale',xscale,' arcsec per pixel'
sxaddpar,outhead,'yscale',yscale,' arcsec per pixel'

; get size outhead (fits header = Nx36 "card images" = Nx2880 bytes)
sizeouthead=size(outhead) 
outheadsize=(1+fix(sizeouthead[1]/36.))*2880

; open output specfitscube for assoc, write header
endian_out='b' ;RR fits files are always bigendian
get_lun, unit_out
openw,unit_out,outdir+specfitscube,swap_endian=(endian ne endian_out)
outassoc=assoc(unit_out,intarr(nx,ny),outheadsize)
if (verbose) then print,' ----- writing specfitscube = '+specfitscube 

; write outhead
rec=assoc(unit_out, bytarr(outheadsize))
rec[0]=byte(outhead)

; read scans one by one and assoc-write into the combination file
for it=0,nt-1 do begin
  if (verbose and it/10. eq it/10) then print,' ----- progress: it ='$
    +trimd(it)
  scan=mrdfits(files[it],1,/silent)
  for iw=0,nw-1 do begin
    image=scan[*,*,iw]
    cutim=image[cropbox[0]:cropbox[2],cropbox[1]:cropbox[3]]
    mima=minmax(cutim)
    if (it eq 0 and iw eq 0) then begin
      if (mima[0] lt 0) then minint=mima[0]*intscale[0] else minint=0
      maxint=mima[1]*intscale[1]
    endif else if (mima[0] lt minint or mima[1] gt maxint) then $
      print,' ===== warning: image it='+trim(it)+' iw='+trim(iw)+$
      ' has minmax ='+trimd(mima)+' exceeding intscale thresholds ='+$
      trimd([minint,maxint])   
    imageout=image>(minint)<maxint
    imageout=fix(3E4*(imageout-minint)/(maxint-minint))  ;RR 12bit max 32767
    iwit=it*nw+iw
    outassoc[iwit]=imageout
  endfor
endfor

; free the image output file
free_lun,unit_out

; ======= write wide-band file in similar fashion

; set some assoc out parameters
outbitpix=16l
mkhdr,outhead,abs(outbitpix)/8,[nx,ny,nt]

; get size outhead (fits header = Nx36 "card images" = Nx2880 bytes)
sizeouthead=size(outhead) 
outheadsize=(1+fix(sizeouthead[1]/36.))*2880

; open output wbfitscube for assoc, write header
endian_out='b' ;RR fits files are always bigendian
get_lun, unit_out
openw,unit_out,outdir+wbfitscube,swap_endian=(endian ne endian_out)
outassoc=assoc(unit_out,intarr(nx,ny),outheadsize)
if (verbose) then print,' ----- writing wbfitscube = '+wbfitscube 

; write outhead
rec=assoc(unit_out, bytarr(outheadsize))
rec[0]=byte(outhead)
; read scans one by one and assoc-write into the combination file
for it=0,nt-1 do begin
  if (verbose and it/10. eq it/10) then print,' ----- progress: it ='$
    +trimd(it)
  image=mrdfits(files[it],3,/silent)
  cutim=image[cropbox[0]:cropbox[2],cropbox[1]:cropbox[3]]
  mima=minmax(cutim)
  if (it eq 0) then begin
    if (mima[0] lt 0) then minint=mima[0]*intscale[0] else minint=0
    maxint=mima[1]*intscale[1]
  endif else if (mima[0] lt minint or mima[1] gt maxint) then $
    print,' ===== warning: image it='+trim(it)+' iw='+trim(iw)+$
    ' has minmax ='+trimd(mima)+' exceeding intscale thresholds ='+$
    trimd([minint,maxint])   
  imageout=image>(minint)<maxint
  imageout=fix(3E4*(imageout-minint)/(maxint-minint))  ;RR 12bit max 32767
  outassoc[it]=imageout
endfor

; free the wide-band output file
free_lun,unit_out

; print cropbox for future use
print,' ----- cropbox = '+trimd(cropbox)

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
if (verbose ne 0) then print,' ===== ibis2cubefiles took '+$
  timelaps+' minutes to write '+outdir+specfitscube

end


; =============== main for testing per IDLWAVE H-c ======================

; flip pairs below for A,B and wav selection
indir='/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/schad/IBIS_CA8542/'
indir='/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/schad/IBIS_H6563/'
wav=8542
wav=6563
starttime=151541  ; ibisA
starttime=164052  ; ibisB
outdir='/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/ibisA/'
outdir='/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/ibisB/'

ibis2cubefiles,indir,wav,starttime,outdir=outdir,$
  intscale=intscale,trimbox=trimbox,cropbox=cropbox,/verbose,$
  specfitscube=specfitscube,wavsfile=wavsfile,timesfile=timesfile,$
  wbfitscube=wbfitscube

;; ; check wavs
;; readcol,outdir+wavsfile,wavs
;; print,wavs

; inspect
showex,outdir+wbfitscube,outdir+specfitscube,$
       specwavsfiles=outdir+wavsfile,$
       trimbox=cropbox,/addlabels

end
