; file: sdoimages2fitscubes.pro = loose SDO image files to fits cubes
; init: Dec  2 2013  Rob Rutten  Sac Peak
; last: Dec 29 2013  Rob Rutten  Deil


pro sdoimages2fitscubes,refwav,indir,outdir,$
  aiascale=aiascale,selfalign=selfalign,itend=itend

;+
 ; NAME:
 ;   sdoimages2fitscubes.pro
 ; PURPOSE:
 ;   convert all SDO image files in indir into cubes per wavelength
 ; CALL:
 ;   sdoimages2fitscubes,refwav,indir,cubedir,$
 ;     aiascale=aiascale,selfalign=selfalign,itend=itend
 ; INPUTS:
 ;   refwav: string with SDO channel definer, e.g. '304', 'continuum'
 ;   indir: string with path/dirname containing image fits files ('level2')
 ;   outdir: string with path/dirname to write cube fits files
 ; KEYWORDS:
 ;   aiascale: use C.J. Schrijver aia_intscale to rescale AIA intensities
 ;   selfalign: self-align the reference and apply shifts to the others
 ;   itend=itend: limit duration for faster tests 
 ; OUTPUTS:
 ;   the cube fits files in outdir
 ; METHOD
 ;   uses assoc for cube fits file writing so can treat large file lists
 ; HISTORY:
 ;   Dec  2 2013: start, from fitsimages2fitscube.pro
;-

; answer a no-parameter query
if (n_params() lt 3) then begin
  print,' sdoimages2fitscubes,refwav,indir,outdir,$'
  print,'  [aiascale=aiascale,selfalign=selfalign,itend=itend]'
  return
endif

; keyword defaults
if (n_elements(itend) eq 0) then itend=0

; define permitted SDO (AIA + HMI) file names
wavs=['94','131','171','193','211','304','335','1600','1700','4500',$  ; AIA
      'magnetogram','continuum','Dopplergram']                         ; HMI
nwavs=n_elements(wavs)

; check refwav specification 
iref=-1
for iwav=0,nwavs-1 do if (wavs[iwav] eq refwav) then iref=iwav
if (iref eq -1 ) then begin
  print,' === ERROR: refwav nonvalid name = ',refwav
  exit
endif

; #########################################

; get filelist
files=findfile(inpath+'*'+filestring+'*')
ntfull=n_elements(files)
if (itend eq 0) then itend=ntfull-1
nt=itend-itstart+1

; get file size 
inheader=headfits(files[0])
nxfull=fxpar(inheader,'naxis1') 
nyfull=fxpar(inheader,'naxis2') 
if (xmax eq 0) then xmax=nxfull-1
if (ymax eq 0) then ymax=nyfull-1
nx=xmax-xmin+1
ny=ymax-ymin+1

; get data type, define outheader
bigendian=1
bitpix=fxpar(inheader,'bitpix')
mkhdr,outheader,abs(bitpix)/8,[nx,ny,itend-itstart+1]
sizeoutheader=size(outheader)  
; fits header = Nx36 "card images" = Nx2880 bytes
outheadersize=(1+fix(sizeoutheader[1]/36.))*2880

; open output file for assoc, write header
get_lun, unit_out
if (bigendian) then openw,unit_out,cubefile,/swap_if_little_endian $
  else openw,unit_out,cubefile
if (bitpix eq -32) then outassoc=assoc(unit_out,fltarr(nx,ny),outheadersize)
if (bitpix eq 16) then outassoc=assoc(unit_out,intarr(nx,ny),outheadersize)
if (bitpix eq 8) then outassoc=assoc(unit_out,bytarr(nx,ny),outheadersize)
if (outheadersize ne 0) then begin
  rec=assoc(unit_out, bytarr(outheadersize))
  rec[0]=byte(outheader)
endif

; loop over timestep it 
for it=itstart,itend do begin
  image=readfits(files[it])
  outassoc[it-itstart]=image[xmin:xmax,ymin:ymax]
  print,'==== time step ',ntostr(it)
endfor

; free the input and output files
free_lun,unit_out

print,' === wrote ',cubefile



end
