; file: alignfac1fac2.pro = former sdo_alignfitscubes2gbo.pro v2
; init: Aug 18 2014  Rob Rutten  Deil
; last: Jul  8 2016  Rob Rutten  Eindhoven   ## needs many adapts

pro alignfac1fac2,incubesdir,outcubesdir,sdofile,gbofile,$
  px_gbo=px_gbo,angle_gbo=angle_gbo,rescale=rescale,$
  nmetcalf=nmetcalf,$
  edgefrac=edgefrac,splinesmooth=splinesmooth,$
  starttime_gbo=starttime_gbo,stoptime_gbo=stoptime_gbo,$
  cadence_gbo=cadence_gbo,timearr_gbo=timearr_gbo,$
  plots=plots,$
  dirplots=dirplots,plotlabel=plotlabel,periodograms=periodograms,$
  crispexcube=crispexcube,$
  writerotshift=writerotshift,blink=blink,verbose=verbose

;+
 ; NAME:
 ;   alignfac1fac2   ## adapt 
 ; PURPOSE:
 ;   align a full set of SDO cubes to GBO image sequence (e.g. SST)
 ; DESCRIPTION:
 ;   this is a wrapper calling multiple programs
 ; CALL:
 ;   see above
 ; INPUTS:
 ;   incubesdir = string with SDO cubes dir
 ;   outcubesdir = string with dir for all output cubes
 ;   sdofile = string with path/name of SDO fitscube file to use for alignment
 ;   gdofile = string with path/name of GBO fitscube file to use for alignment
 ;   px_gbo = pixel size GBO data in arcsec
 ;   angle_gbo = GBO data rotation from solar (X,Y) in degrees clockwise
 ;   for GBO timing specification use either:
 ;     starttime_gbo, stoptime_gbo, cadence_gbo (in seconds)
 ;       time format example: starttime_gbo='2014-06-21 08:02:32'
 ;   or use:
 ;     timearr_gbo = array individual image times in TAI (from anytim2tai)
 ; OPTIONAL INPUTS:
 ;   rescale = 1/0: apply mean of Metcalf scale results (default not)
 ;   nmetcalf = number of Metcalf colalignment calls (slow; default 1 = start)
 ;   edgecut = 4-elem integer vector [xmin,ymin,xmax,ymax] to ignore borders
 ;   edgecross = idem for crossalign
 ;   splinesmooth = smooth value (0 > default=10, 1 linear, 1000 thru points)
 ;   plots =  1/0: make plots
 ;   plotlabel = string for plot identifier (no spaces)
 ;   periodograms: 1/0 plot these too
 ;   crispexcube = 1/0: add single multi-channel CRISPEX-format cubefile
 ;   writerotshift = 1/0: write rotation angles and (x,y) shifts into files
 ;   blink = flickrr duration in sec; -1 = no flicker;  default = 10
 ;   verbose = 1/0: more output
 ; OUTPUTS:
 ;   aligned cubes and alignment plots and writerotshift files in cubesdir
 ; RESTRICTIONS:
 ;   uses assoc to permit large input files   
 ; HISTORY:
 ;   Aug 19 2014 RR: start
 ;   Oct 21 2014 RR: crispexcube (suggested by Luc Rouppe vd Voort)
 ;   Feb  8 2015 RR: option edgefrac
 ;   Sep 26 2015 RR: variable spline smooth; admit dirpath spaces
 ;   Sep 27 2015 RR: improved drift correction
 ;   Nov 26 2015 RR: added multiple Metcalf calls for FOV rotation
;-

; answer no-parameter query
if (n_params() lt 4) then begin
  print,' alignfac1fac2,incubesdir,outcubesdir,sdofile,gbofile,'
  print,'   px_gbo=px_gbo,angle_gbo=angle_gbo,rescale=rescale,'
  print,'   nmetcalf=nmetcalf,'
  print,'   edgefrac=edgefrac,splinesmooth=splinesmooth,'
  print,'   starttime_gbo=starttime_gbo,stoptime_gbo=stoptime_gbo,'
  print,'   cadence_gbo=cadence_gbo,timearr_gbo=timearr_gbo,'
  print,'   plots=plots,'
  print,'   plotlabel=plotlabel,periodograms=periodograms,'
  print,'   blink=blink,verbose=verbose'
  return
endif

; set wall-clock timer (seconds)
timestart=systime(1) 

; print pacifier
print,' ===== alignfac1fac2 starts'

; flag spaces in dirpaths (as Catherine Fischer Mac disk)
incubesdir=strjoin(strsplit(incubesdir,' ',/extract),'\ ') 
outcubesdir=strjoin(strsplit(outcubesdir,' ',/extract),'\ ') 

; keyword defaults ;RR not(keyword_set)=1 if keyword=0; wrong for default#0
cd,current=thisdir
if (n_elements(rescale) eq 0) then rescale=0
if (n_elements(nmetcalf) eq 0) then nmetcalf=1
if (n_elements(edgefrac) eq 0) then edgefrac=0.1
if (n_elements(splinesmooth) eq 0) then splinesmooth=0
if (n_elements(starttime_gbo) eq 0) then starttime_gbo=0
if (n_elements(stoptime_gbo) eq 0) then stoptime_gbo=0
if (n_elements(cadence_gbo) eq 0) then cadence_gbo=0
if (n_elements(timearr_gbo) eq 0) then timearr_gbo=0
if (n_elements(plots) eq 0) then plots=0
if (n_elements(plotlabel) eq 0) then plotlabel=''
if (n_elements(periodograms) eq 0) then periodograms=0
if (n_elements(crispexcube) eq 0) then crispexcube=0
if (n_elements(writerotshift) eq 0) then writerotshift=0 
if (n_elements(blink) eq 0) then blink=10 
if (n_elements(verbose) eq 0) then verbose=0

; pixel size and ratio
px_sdo=0.6             ; AIA pixel size in arcsec; also co-aligned HMI cubes 
pxratio=px_sdo/px_gbo  ; normally > 1

; make outcubesdir and dirplots if not yet existing
spawn,'mkdir -p '+outcubesdir
if (plots ne 0) then begin
  dirplots=outcubesdir+'/plots_align/'
  spawn,'mkdir -p '+dirplots
endif
if (writerotshift ne 0) then begin
  dirrotshift=outcubesdir+'/rotshift/'
  spawn,'mkdir -p '+dirrotshift
endif

; check GBO timing specification
if  ((cadence_gbo ne 0 and timearr_gbo[0] ne 0) or $
     (cadence_gbo eq 0 and timearr_gbo[0] eq 0)) then begin
  print,' ##### alignfac1fac2: either cadence_gbo or timearr_gbo'
  return
endif
if (cadence_gbo gt 0 and (starttime_gbo eq 0 or stoptime_gbo eq 0)) $
then begin
  print,' ##### alignfac1fac2: cadence needs starttime + stoptime'
  return
endif

; set endian
bigendian=1

; GBO timearr_gbo specified
if (timearr_gbo[0] ne 0) then nt_gbo_seq=n_elements(timearr_gbo)

; GBO cadence_gbo, starttime_gbo stoptime_gbo specified: get timearr_gbo 
if (cadence_gbo ne 0) then begin  
  start_gbo=anytim2tai(starttime_gbo)
  stop_gbo=anytim2tai(stoptime_gbo)
  nt_gbo_seq=fix((stop_gbo-start_gbo)/cadence_gbo)+1
  timearr_gbo=indgen(nt_gbo_seq)*cadence_gbo+start_gbo
endif 

; get SDO cube geometry and file datatype
header_sdo=headfits_rr(sdofile)
header_sdosize=(1+fix(n_elements(header_sdo)/36.))*2880
bitpix_sdo=fxpar(header_sdo,'bitpix')
nx_sdo=fxpar(header_sdo,'naxis1') 
ny_sdo=fxpar(header_sdo,'naxis2') 
nt_sdo=fxpar(header_sdo,'naxis3') 

;  get SDO time array from keywords (present in RJR-made SDO cubes)
starttim_sdo=fxpar(header_sdo,'starttim')
cadence_sdo=fxpar(header_sdo,'cadence')
if (starttim_sdo eq 0 or cadence_sdo eq 0) then begin
  print, ' ##### SDO input file has no STARTTIM or CADENCE in header'
  return
endif
timearr_sdo=indgen(nt_sdo)*cadence_sdo+anytim2tai(starttim_sdo)

; open sdofile for input assoc
get_lun, unit_sdo
if (bigendian) then openr,unit_sdo,sdofile,/swap_if_little_endian $
else openr,unit_sdo,sdofile
if (bitpix_sdo eq -32) then $
  inassoc_sdo=assoc(unit_sdo,fltarr(nx_sdo,ny_sdo),header_sdosize)
if (bitpix_sdo eq 16) then $
  inassoc_sdo=assoc(unit_sdo,intarr(nx_sdo,ny_sdo),header_sdosize)
if (bitpix_sdo eq 8) then $
  inassoc_sdo=assoc(unit_sdo,bytarr(nx_sdo,ny_sdo),header_sdosize)

; check SDO input image size against header info
im0_sdo=inassoc_sdo[0]
sizeim0_sdo=size(im0_sdo)
nx_im0_sdo=sizeim0_sdo[1]
ny_im0_sdo=sizeim0_sdo[2]
if (nx_im0_sdo ne nx_sdo or ny_im0_sdo ne ny_sdo) then begin
  print, ' ##### nx or ny in header sdofile inequal to first image [nx,ny]' 
  return
endif

; get GBO cube geometry and file datatype
header_gbo=headfits_rr(gbofile)
header_gbosize=(1+fix(n_elements(header_gbo)/36.))*2880
bitpix_gbo=fxpar(header_gbo,'bitpix')
nx_gbo=fxpar(header_gbo,'naxis1') 
ny_gbo=fxpar(header_gbo,'naxis2') 
nt_gbo=fxpar(header_gbo,'naxis3') 

; open gbofile for input assoc
get_lun, unit_gbo
if (bigendian) then openr,unit_gbo,gbofile,/swap_if_little_endian $
else openr,unit_gbo,gbofile
if (bitpix_gbo eq -32) then $
  inassoc_gbo=assoc(unit_gbo,fltarr(nx_gbo,ny_gbo),header_gbosize)
if (bitpix_gbo eq 16) then $
  inassoc_gbo=assoc(unit_gbo,intarr(nx_gbo,ny_gbo),header_gbosize)
if (bitpix_gbo eq 8) then $
  inassoc_gbo=assoc(unit_gbo,bytarr(nx_gbo,ny_gbo),header_gbosize)

; check actual gbo image size
im0_gbo=inassoc_gbo[0]
sizeim0_gbo=size(im0_gbo)
nx_im0_gbo=sizeim0_gbo[1]
ny_im0_gbo=sizeim0_gbo[2]
if (nx_im0_gbo ne nx_gbo or ny_im0_gbo ne ny_gbo) then begin
  print, ' ##### nx or ny in header gbofile inequal to first image [nx,ny]' 
  return
endif

; check gbo nt value against nt from timing specification
if (nt_gbo_seq ne nt_gbo) then begin
  print,' ##### gbofile has nt = '+ntostr(nt_gbo)+$
    ' but timing specification has nt = '+ntostr(nt_gbo_seq)
  print,' ##### I will use smallest value'
  nt_gbo=min([nt_gbo,nt_gbo_seq])
  timearr_gbo=timearr_gbo[0:nt_gbo-1]
endif

; ================ Metcalf alignment(s) ================

; declare Metcalf output arrays 
angmet=fltarr(nmetcalf)
xshiftmet=fltarr(nmetcalf)
yshiftmet=fltarr(nmetcalf)
scalemet=fltarr(nmetcalf)
nx2met=intarr(nmetcalf)
ny2met=intarr(nmetcalf)
timearr_met=dblarr(nmetcalf) 

; loop over Metcalf samplings
metskip=fix(nt_gbo/nmetcalf+0.5)
for itmet=0,nmetcalf-1 do begin

  itgbo=itmet*metskip
  if (itgbo gt nt_gbo-1) then itgbo=nt_gbo-1
  absdiff=abs(timearr_sdo-timearr_gbo[itgbo])
  itsdo=where(absdiff eq min(absdiff))
  imgbo=inassoc_gbo[itgbo]
  imsdo=inassoc_sdo[itsdo]

; use previous angle in the next one
  if (itmet gt 0) then angle_gbo=angmet[itmet-1]

; get Metcalf results for this pair
  print,' ==== run findalignimages for pair itmet = '+ntostr(itmet)
  findalignimages,imsdo,imgbo,px_sdo,px_gbo,angle_gbo,$
    outang,outshift,outscale,nx2=nx2,ny2=ny2,$
    edgefrac=edgefrac,smearref=2.*pxratio,$
    blink=blink,verbose=verbose

; put results of this Metcalf into arrays
  timearr_met[itmet]=timearr_gbo[itgbo]
  angmet[itmet]=outang
  xshiftmet[itmet]=outshift[0]
  yshiftmet[itmet]=outshift[1]
  scalemet[itmet]=avg(outscale)
  nx2met[itmet]=nx2
  ny2met[itmet]=ny2

endfor  ; end loop over Metcalf samplings

; get spline approximation on timearr_gbo
if (nmetcalf gt 1) then begin
  var=moment(angmet)
  rms=sqrt(var[1])
  if (splinesmooth eq 0) then $
    splinang=imsl_cssmooth(timearr_met,angmet,smpar=rms/10.) else $
      splinang=imsl_cssmooth(timearr_met,angmet,smpar=rms/splinesmooth)
  angmet_gbo=imsl_spvalue(timearr_gbo,splinang)

; write angles to file for trials
  fmt='(3D18.3)'
  if (writerotshift ne 0) then begin
    writecol,dirrotshift+'/alrotatemet.dat',timearr_met,angmet,fmt='(2D18.3)'
    writecol,dirrotshift+'/alrotategbo.dat',timearr_gbo,angmet_gbo,$
      fmt='(2D18.3)'
  endif

; make check plot of the angles and their spline fit
  if (plots ne 0) then begin
    psfilename=dirplots+'angmet_gbo'+plotlabel+'.eps'
    sdt=anytim2utc(timearr_gbo,/external)
    jd_gbo=julday(sdt.month,sdt.day,sdt.year,sdt.hour,sdt.minute,sdt.second)
    sdt=anytim2utc(timearr_met,/external)
    jd_met=julday(sdt.month,sdt.day,sdt.year,sdt.hour,sdt.minute,sdt.second)
    dummy=label_date(date_format='%H:%I')
    xtitle='time'
    ytitle='rotation angle'
    thick=1
    xrange=[min(jd_gbo)-0.005,max(jd_gbo)+0.005]
    datarange=[min(angmet),max(angmet)]
    dataspan=datarange[1]-datarange[0]
    yrange=[datarange[0]-0.1*dataspan,datarange[1]+0.1*dataspan]
    openpsplot,psfilename,fontsize=8,thick=thick
    plot,jd_gbo,angmet_gbo,$ ;; psym=6,symsize=0.1,$
      position=[0.15,0.15,0.9,0.9],/normal,$ 
      xticklen=0.03,yticklen=0.03/1.6,$       
      xtickformat='label_date',xtickunits='Time',$
      xtitle=xtitle,ytitle=ytitle,$
      xrange=xrange,xstyle=1,yrange=yrange,ystyle=1 
    oplot,jd_met,angmet,psym=2,symsize=0.7

; angle plot done
    closepsplot,psfilename,opengv=1
  endif  ; endif for if (plots)
endif    ; endif fir (if (nmetcalf gt 1)

; reform sdo cube in temporary cubefile to gbo cube params 
; (size, timing; scale, angle, shift from above first pair)
if (rescale eq 1) then begin
  if (nmetcalf eq 1) then scalefac=avg(outoutscale) else $
    scalefac=avg(scalemet)
end else scalefac=1
if (nmetcalf eq 1) then rotate=outang else rotate=angmet_gbo
reformcubefile,sdofile,outcubesdir+'/temp_sdo_al.fits',$
  rotate=-outang,congridfactor=scalefac*pxratio,$
  shift=outshift,cutcentralx=nx2,cutcentraly=ny2,$
  timearr_out=timearr_gbo,verbose=verbose

; cross-align this temporary cubefile to gbo to find time-dependent drifts
crossalignfitscubes,gbofile, outcubesdir+'/temp_sdo_al.fits',$
  outcubesdir+'/temp_gbo_xal.fits',outcubesdir+'/temp_sdo_xal.fits',$
  edgefrac=edgefrac,smearref=2.*pxratio,$
;;  plotshifts=dirplots+'/xalshifts.eps',verbose=verbose,$
  shifts=xalshifts

; write the shifts to file for trials
if (writerotshift ne 0) then writecol,$
  dirrotshift+'/alshifts.dat',timearr_gbo,xalshifts[0,*],xalshifts[1,*],$
  fmt='(3D18.3)'

; get and show spline approximations
if (splinesmooth ne 0) then sdo_splineshifts,xalshifts,splxalshifts,$
  timearr_gbo,plots=plots,dirplots=dirplots,plotlabel=plotlabel,$
  periodograms=periodograms,smooth=splinesmooth

; setup total shift array on SDO scale with or without spline fit
sdoshifts=fltarr(2,nt_gbo)
for it=0,nt_gbo-1 do begin
  if (splinesmooth ne 0) then begin
    sdoshifts[0,it]=splxalshifts[0,it]/pxratio+outshift[0]
    sdoshifts[1,it]=splxalshifts[1,it]/pxratio+outshift[1]
  endif else begin
    sdoshifts[0,it]=xalshifts[0,it]/pxratio+outshift[0]
    sdoshifts[1,it]=xalshifts[1,it]/pxratio+outshift[1]
  endelse
endfor

;; ; old trial  to improve by Metcalfing means, no good
;; ; -------------------------------------------------- 
;; ; apply shifts above
;; reformcubefile,outcubesdir+'/temp_sdo_xal.fits',$
;;   outcubesdir+'/temp_sdo_splal.fits',$
;;   shift=sdoshifts,cutcentralx=nx2,cutcentraly=ny2
;; ; get cube means
;; meanimsdo=getmeanfitscube(outcubesdir+'/temp_sdo_splal.fits')
;; meanimgbo=getmeanfitscube(outcubesdir+'/temp_gbo_xal.fits')
;; ;; ; insert manual check
;; ;; STOP
;; ;; flickrr,meanimsdo,meanimgbo
;; ; Metcalf mean images
;; findalignimages,meanimsdo,meanimgbo,px_gbo*scalefac,px_gbo,0.0,$
;;   edgefrac=edgefrac,smearref=2.*0.6/px_gbo,$
;;   outang2,outshift2,outscale2,nx2=nx2,ny2=ny2,blink=20.,verbose=verbose
;; STOP

; delete the temp files
spawn,'rm -f '+outcubesdir+'/temp*.fits'

;; ; now do all cubes
;; doallfiles,incubesdir,outcubesdir,'.fits','_algbo.fits','reformcubefile',$
;;   rotate=outang,congridfactor=scalefac*pxratio,$
;;   shift=sdoshifts,cutcentralx=nx2,cutcentraly=ny2,$
;;   timearr_out=timearr_gbo

;; ; produce multi-channel crispex cubefile (SST format, not fits)
;; if (crispexcube ne 0) then $
;;   sdo_fitscubes2crispex,outcubesdir,'_algbo.fits',outcubesdir+'/sdo_crispex',$
;;   /diagname,bytscale=2,/verbose

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
print,' ===== alignfac1fac2 took '+timelaps+' minutes'

end

; ================= main for testing IDLWAVE hyper C ==================

; Jul  8 2016 old test is gone (maybe USB?)
;; path='/home/rutten/data/SST/2011-05-07-test/'
;; sdofile=path+'sdo/cubes/hmicont.fits'
;; gbofile=path+'sstcubes/fecont.fits'
;; alignfac1fac2,path+'sdo/cubes',path+'sdo2sst',$
;;   sdofile,gbofile,$
;;   nmetcalf=10,$
;;   px_gbo=0.0592,angle_gbo=48,$
;;   cadence_gbo=27.4,starttime_gbo='2011-05-07 08:56:00',$
;;   stoptime_gbo='2011-05-07 09:52:00',$ 
;; ;;  /crispexcube,$
;;   /plots,plotlabel='Fecont-HMIcont',$
;;   edgefrac=0.1,/writerotshift,blink=0,/verbose

;; ; check
;; movex,path+'sdo2sst/hmicont_algbo.fits',file2=gbofile

;; ; Jul  8 2016 try IRIS contrail data  ## too early?
;; path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
;; incubesdir=path+'test'
;; outcubesdir=path+'test'
;; irisfile=path+'iristmp/iris_init.fits'
;; sstfile=path+'sstcubes/ca_0.fits'
;; ; SST time line (IRIS file interpolated to this)
;; restore,path+'calib_tseries/tseries.8542.08:02:31.calib.sav'
;; nt=n_elements(time)
;; timearr_sst=time
;; for it=0,nt-1 do timearr_sst[it]='2014-06-21 '+time[it]
;; timearr_sst=anytim2tai(timearr_sst)

;; alignfac1fac2,path+'sdo/cubes',path+'sdo2sst',$
;;   irisfile,sstfile,$
;;   nmetcalf=5,$
;;   px_gbo=0.0566 ,angle_gbo=62.4,$
;;   timearr_gbo=timearr_sst,$
;; ;;  /crispexcube,$
;;   plotlabel='IRIS-Ca8542_0',$
;;   edgecut=[100,100,800,800],/writerotshift,blink=0,/verbose

; Feb 26 2017 
path='/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
incubesdir=path+'test'
outcubesdir=path+'test'
irisfile=path+'iristmp/iris_init.fits'
sstfile=path+'sstcubes/ca_0.fits'
; SST time line (IRIS file interpolated to this)
restore,path+'calib_tseries/tseries.8542.08:02:31.calib.sav'
nt=n_elements(time)
timearr_sst=time
for it=0,nt-1 do timearr_sst[it]='2014-06-21 '+time[it]
timearr_sst=anytim2tai(timearr_sst)

alignfac1fac2,path+'sdo/cubes',path+'sdo2sst',$
  irisfile,sstfile,$
  nmetcalf=5,$
  px_gbo=0.0566 ,angle_gbo=62.4,$
  timearr_gbo=timearr_sst,$
;;  /crispexcube,$
  plotlabel='IRIS-Ca8542_0',$
  edgecut=[100,100,800,800],/writerotshift,blink=0,/verbose

end

