; file: write_mpeg
; last: Nov 18 2013  Rob Rutten  Sac Peak

PRO Write_MPEG, mpegFileName, image_array, $
                BSEARCH_ALG=bsalg, FRAMERATE=framerate, GOP_SIZE=gop_size, $
                HIGH_QUALITY=hq, KEEP_TMP=keep_tmp, PATTERN=pattern, $
                PIXEL=pxl, PSEARCH_ALG=psalg, RANGE=range, $
                SLICES_PER_FRAME=spf, TMPDIR=tmpdir, VERBOSE=verbose
;+
; NAME:
;       Write_MPEG
; PURPOSE:
;       write a 3D (byte) array as MPEG movie
; CALLING SEQUENCE:
;       Write_MPEG, File, Array
; INPUTS:
;       File : (string) name for the mpeg file
;       Array: (byte) variable with images, size (sx, sy, num)
; OPTIONAL PARAMETERS:
;       
; KEYWORDS:
;       KEEP_TMP:   (flag) keep the temporary files
;       FRAMERATE:  (float) framerate for encoding.  Legal values are:
;                   23.976, 24, 25, 29.97, 30, 50 ,59.94, 60
;       VERBOSE:    (int) Verbosity. 1=decent info, 2=full info
;       HIGH_QUAL:  (flag) switch on high quality settings (a
;                   combination of flags)
;       
;       The following keywords are explained in the manpage of
;       mpeg_encode.  Here, only allowed/default values are given
;
;       BSEARCH_ALG      A: [SIMPLE | CROSS2 | EXHAUSTIVE] D: SIMPLE
;       GOP_SIZE         D: 12
;       PATTERN          D: IBBPBBPBBPBB
;       PIXEL            A: [FULL | HALF]  D: HALF
;       PSEARCH_ALG      A: [EXHAUSTIVE | TWOLEVEL | SUBSAMPLE | LOGARITHMIC]
;                        D: LOGARITHMIC
;       RANGE            D: 8
;       SLICES_PER_FRAME D: 8
;       
; OUTPUTS:
;       
; COMMON BLOCKS:
;       
; SIDE EFFECTS:
;       creates temporary files in /tmp (deleted after conversion)
; RESTRICTIONS:
;       needs working mpeg_encode (Tested with version 1.5)
; PROCEDURE:
;       Write single frames as PPM files in /tmp/idl2mpeg.frames
;       Create appropriate control file
;       Call mpeg_encode for the conversion
;       Remove temporary files
; MODIFICATION HISTORY:
;                    Original Version by A. Scott Denning,
;                    scott@abyss.Atmos.ColoState.edu
;       23-Feb-1996  P.Suetterlin, KIS
;                    Use WRITE_PPM instead of GIF/giftoppm
;                    Different testing for existence of TMPDIR (not
;                    dependend on users SHELL
;       19-Nov-1999  P.Suetterlin, SIU
;                    Change Orientation of Images: PPM is upside-down
;       25-Jan-2001  Keyword FRAMERATE
;       23-Oct-2001  More keywords to support features of mpeg_encode
;       25-Feb-2002  switch '-p' to mkdir for recursive dir creation
;       25-Oct-2002  Different default pattern, PIXEL and RANGE values
;                    HIGH_QUALITY keyword
;       Oct 27 2002  Rob Rutten, SIU: added -f to rm commands
;       Nov 18 2013  RR: mpeg_encode > ppmtompeg
;-

IF keyword_set(framerate) THEN BEGIN
    frame_allow = [23.976, 24, 25, 29.97, 30, 50, 59.94, 60]
    IF (where(frame_allow EQ framerate))(0) EQ -1 THEN BEGIN
        message, 'Wrong frame rate.  Allowed values are:', /cont
        message, '23.976, 24, 25, 29.97, 30, 50, 59.94, 60', /cont
        fbest = abs(frame_allow-framerate)
        framerate = (frame_allow(where(fbest EQ min(fbest))))(0)
        message, 'Using closest value: '+strtrim(framerate, 2), /cont
    ENDIF
ENDIF

IF keyword_set(hq) THEN BEGIN
    pattern = 'I'
    gop_size = 1
    spf = 1
    range = 12
    psalg = 'TWOLEVEL'
    bsalg = 'CROSS2'
ENDIF 
IF NOT keyword_set(pattern) THEN pattern = 'IBBPBBPBBPBB'
IF NOT keyword_set(gop_size) THEN gop_size = strlen(pattern)
IF NOT keyword_set(spf) THEN spf = 8
IF NOT keyword_set(pxl) THEN pxl = 'HALF'
IF NOT keyword_set(range) THEN range = 8
IF NOT keyword_set(psalg) THEN psalg = 'LOGARITHMIC'
IF NOT keyword_set(bsalg) THEN bsalg = 'SIMPLE'

;; ##### RR Aug 17 2007    mpeg_command='mpeg_encode -realquiet '
;; ##### RR Nov 18 2013    mpeg_command='mpeg_encode '
IF keyword_set(verbose) THEN BEGIN
    IF verbose EQ 1 THEN  $
      mpeg_command='ppmtompeg -quiet 2 ' $
    ELSE  $
      mpeg_command='ppmtompeg '
ENDIF ELSE  $
  mpeg_command='ppmtompeg -realquiet '

movieSize = SIZE(image_array)
xSize = movieSize(1)
ySize = movieSize(2)
nFrames = movieSize(3)

nDigits = 1+FIX(ALOG10(nFrames))
formatString = STRCOMPRESS('(i'+STRING(nDigits)+'.'+STRING(nDigits) + $
                           ')', /REMOVE_ALL)

  ;;; Load current color table into byte arrays
TVLCT, red, green, blue, /GET
red = BYTE(red)
green = BYTE(green)
blue = BYTE(blue)

ON_IOERROR, badWrite

  ;;; Make a temporary directory if necessary or clear it otherwise'

IF NOT keyword_set(TMPDIR) THEN TMPDIR = '/tmp/idl2mpeg.frames'

dirExists = (strlen(findfile(tmpdir+'/..')))(0) NE 0
IF dirExists THEN $
  command='rm -f ' + TMPDIR + '/*' $
ELSE $
  command='mkdir -p ' + TMPDIR 
SPAWN, command 

  ;;; Write each frame into TMPDIR as an .ppm image file
FOR frameNum = 0, nFrames-1 DO BEGIN
  fileName = TMPDIR + '/frame.' + STRING(frameNum,FORMAT=formatString)$
           + '.ppm'
  tmp = bytarr(3, xsize, ysize)
  r = rotate(image_array(*, *, frameNum), 7)
  tmp(0, *, *) = red(r)
  tmp(1, *, *) = green(r)
  tmp(2, *, *) = blue(r)
  write_ppm, fileName, tmp
ENDFOR

; Build the mpeg parameter file

paramFile = TMPDIR + '/idl2mpeg.params'
;; #### Aug 17 2007 
paramFile = TMPDIR + '/idl2mpeg.par'
OPENW, unit, paramFile, /GET_LUN
PRINTF, unit, 'PATTERN	        '+strupcase(pattern)
PRINTF, unit, 'OUTPUT           ' + mpegFileName
PRINTF, unit, 'GOP_SIZE         '+strtrim(gop_size, 2)
PRINTF, unit, 'SLICES_PER_FRAME '+strtrim(spf, 2)
PRINTF, unit, 'BASE_FILE_FORMAT	PPM'
PRINTF, unit, 'INPUT_CONVERT *'
PRINTF, unit, 'INPUT_DIR        '+tmpdir
PRINTF, unit, 'INPUT'
printf, unit, 'frame.*.ppm ['+STRING(0,FORMAT=formatString)+ $
  '-'+ STRING(nFrames-1, FORMAT=formatString)+']'
PRINTF, unit, 'END_INPUT'
PRINTF, unit, 'PIXEL            '+strupcase(pxl)
PRINTF, unit, 'RANGE            '+strtrim(range, 2)
PRINTF, unit, 'PSEARCH_ALG      '+strupcase(psalg)
PRINTF, unit, 'BSEARCH_ALG      '+strupcase(bsalg)
PRINTF, unit, 'IQSCALE          8'
PRINTF, unit, 'PQSCALE          8'
PRINTF, unit, 'BQSCALE          8'
PRINTF, unit, 'REFERENCE_FRAME  DECODED'
PRINTF, unit, 'FORCE_ENCODE_LAST_FRAME'
IF keyword_set(framerate) THEN $
  PRINTF, unit, 'FRAME_RATE       '+strtrim(framerate, 2)
FREE_LUN, unit

; spawn a shell to process the mpeg_encode command        
SPAWN, mpeg_command+paramFile 

IF NOT keyword_set(keep_tmp) THEN spawn, 'rm -fr '+tmpdir

RETURN

badWrite:
message, 'Unable to write MPEG file!'

END

