FUNCTION Smoother, data, p1, p2, smoothdim=sd, double=doub, au=a, al=al, $
                   indx=indx, init=init
;+
; NAME:
;       SMOOTHER
; PURPOSE:
;       Returns a smoothed copy of the input data. 
; CATEGORY:
;       Curve fitting, noise removal
; CALLING SEQUENCE:
;       RESULT = SMOOTHER( DATA [,P1] [,P2] )
; INPUTS:
;       DATA  : (1 or 2-dim) array to smooth. Should be
;               ***  EQUIDISTANT  ***
; OPTIONAL INPUT PARAMETER:
;       P1,P2 : smooth-parameters as described below. Default to 10
; KEYWORDS:
;       SMOOTHDIM  : For 2-dim arrays, the dimension for wich the
;                    smoothing should be performed. Possible values are
;                    1 for (horizontal) x-direction and 2 for (vertical)
;                    y-direction. Default is 1 (horizontal)
;
;       DOUBLE     : (Flag) For 2-dim arrays: Perform a double pass
;                    smoothing, i.e. first in vertical, then in
;                    horizontal direction. Actually, it makes no
;                    difference whether smoothing is applied first in
;                    x and then in y or vice versa. Nevertheless, you
;                    can specify both the DOUBLE and SMOOTHDIM
;                    keyword. Smoothing will then be done first in the
;                    direction orthogonal to the specified one. This
;                    is, if you call with SMOOTHDIM=2 and DOUBLE,
;                    first pass smoothing will be in horizontal, and
;                    second pass in vertical direction.
;
;       AU, AL,    : Passes the upper and lower triangular matrices and
;       INDX,INIT    the permutation vector indx of the LU decomposition
;                    of the bandmatrix a to the user (if keyword INIT is
;                    set) or to smoother resp.  This is particularly
;                    usefull if the user wishes to smooth different
;                    datasets of ** SAME LENGTH ** and with the same set
;                    of parameters P1,P2. The main computational work of
;                    this routine lies in the LU-decomposition of the
;                    matrix that describes the linear equation system.
;                    This matrix depends only on the parameters P1, P2
;                    and the length of the dataset. Therefore one can
;                    call smoother once with keywords AU,AL and INIT
;                    set. For later use, passing AL and AU without INIT
;                    set will skip the computation of the LU
;                    decomposition and save a lot of time.
; OUTPUTS:
;       RESULT : array of same dimension and type as DATA with the
;                smoothed data.
; SIDE EFFECTS:
;       none
; RESTRICTIONS:
;       
; PROCEDURE:
;       This is an IDL-implementation of a FORTRAN code written by U.
;       Grabowski (Kiepenheuer-Institut fuer Sonnenphysik, FRG). It
;       minimizes the following functional:  
;
;       SUM { (Yi-Fi)^2 + P1*[(F"i-F"i-1)^2 + (F"i-F"i+1)] + P2*(F"i)^2 }
;
;       where i (i=1...n) denotes the index of the point, Y the data
;       values F the points of the smoothed data set and F" the second
;       derivative. P1 and P2 are parameters the user is free to choose
;       according to his personal requirements. It is recommended to
;       begin with both values equal and of order 1 to 10. Larger values
;       force the routine to generate a smoother approximation.  Both
;       values zero will mainly reproduce the original data set. Setting
;       p1 larger than p2 reduces the variation of the curvature from
;       point to point, p2 greater than p1 forces the routine to find a
;       smoothed set with a small curvature.  
;       Remark: For fitting of spectral data it is recomended NOT to use
;       large values for P2.  This is because a small curvature produces
;       a bad fit of the line center of spectral lines or other features
;       with localy high (true) curvature.
;
; MODIFICATION HISTORY:
;       Dec 1992    Basic idea and FORTRAN-subroutine from U.Grabowski
;       18-02-1993  First IDL-Implementation: Calling an external
;                   Program. FORTRAN main routine by U.Grabowski,
;                   IDL-code P.Suetterlin
;       28-02-1993  Solution of the band-diagonal equation system in
;                   IDL (routines BANDEC and BANBKS from Numerical
;                   Recipes)                                  PS
;-

on_error, 2 

;;;  
;;;  Default values for P1, P2 and SD if not given
;;;  sd=1 means: smoothing in x-direction
;;;
IF n_params() LT 3 THEN p2 = 10 
IF n_params() LT 2 THEN p1 = 10
IF NOT keyword_set(sd) THEN sd = 1

s = size(data)
IF (s(0) EQ 2) THEN BEGIN
    IF min(s(1:2)) NE 1 THEN BEGIN             ;;;  'real' 2-dim
                                               ;;;  case
        IF sd EQ 2 THEN BEGIN                    ;;;  Smooth in Y:
            sm_dat = transpose(data)                 ;;;  Transpose
                                                     ;;;  data, use
            tra = 1                                  ;;;  same
                                                     ;;;  code
        ENDIF ELSE BEGIN
            sm_dat = data
            tra = 0
        ENDELSE
        s = size(sm_dat)
        dim = s(2)
    ENDIF ELSE IF s(1) EQ 1 THEN BEGIN         ;;;  Coloumn
                                               ;;;  vector:
        sm_dat = transpose(data)                   ;;;  Transpose
        s = size(sm_dat)
        tra = 1
        dim = 1
    ENDIF
ENDIF ELSE BEGIN                             ;;;  data is 1-dim
                                             ;;;  row-vector
    sm_dat = data
    dim = 1
    tra = 0
ENDELSE

;;;
;;;  If a double pass smoothing is requested, now call smoother with
;;;  smoothdim set to 2. SM_DAT holds the (eventually transposed) data,
;;;  so smoothing will in all cases be performed in 2 different
;;;  directions, no matter if smoother was called allready with
;;;  SMOOTHDIM=2. But perform this option ONLY with real 2-dim data
;;;  
IF keyword_set(doub) AND dim NE 1 THEN $
  sm_dat = smoother(sm_dat, p1, p2, smoothdim = 2)
     
;;; length of array in 1. dimension
n = s(1)

;;;
;;;  Test if a, al and indx are allready computed
;;;
IF (keyword_set(a)) AND (keyword_set(al)) AND (keyword_set(indx)) AND $
  (NOT keyword_set(init)) THEN GOTO, go_on

;;;
;;;  Now set up the linear equation system: Init Matrix a
;;;

a = fltarr(n, 7)
al = fltarr(n, 3)
indx = intarr(n)

dia0 = 4.*p1+ 5.*p2+1.           ;;;  These are the non-zero elements
dia1 = 38.*p1+30.*p2+1.           ;;;  of the matrix a
dia2 = 56.*p1+22.*p2+1.
dia3 = 42.*p1+ 7.*p2+1.
diag = 40.*p1+ 6.*p2+1.
sd11 = -12.*(p1+p2)
sd12 = -6.*(7.*p1+4.*p2)
sd13 = -4.*(9.*p1+2.*p2)
sdg1 = -30.*p1-4.*p2
sd22 = 3.*(4.*p1+3.*p2)
sd23 = 6.*(3.*p1+p2)
sdg2 = 12.*p1+p2
sd33 = -2.*(2.*p1+p2)
sdg3 = -2.*p1

;;;
;;;  Now fill the bandmatrix a in the way described in
;;;     [1] Press et al.  Numerical recipes.
;;;
a(0,*) = [0   ,0   ,0   ,dia0,sd11,sd22,sd33]
a(1,*) = [0   ,0   ,sd11,dia1,sd12,sd23,sdg3]
a(2,*) = [0   ,sd22,sd12,dia2,sd13,sdg2,sdg3]
a(3,*) = [sd33,sd23,sd13,dia3,sdg1,sdg2,sdg3]
a(n-1,*)=[sd33,sd22,sd11,dia0,0   ,0   ,0   ]
a(n-2,*)=[sdg3,sd23,sd12,dia1,sd11,0   ,0   ]
a(n-3,*)=[sdg3,sdg2,sd13,dia2,sd12,sd22,0   ]
a(n-4,*)=[sdg3,sdg2,sdg1,dia3,sd13,sd23,sd33]
FOR i = 4, n-5 DO $
  a(i, *) = [sdg3, sdg2, sdg1, diag, sdg1, sdg2, sdg3]

;;;
;;;  Now perform LU-decomposition for bandmatrix a 
;;;

bandec, a, al, indx, 3, 3, d

Go_on:

nm1 = n-1
FOR row = 0, dim-1 DO BEGIN                ;;;  loop through all rows
    b = float(sm_dat(*, row))
    l = 2
    FOR k = 0, nm1 DO BEGIN                ;;;  This is an involved
        i = indx(k)                        ;;;  form of BANBKS to
        IF (i NE k) THEN BEGIN             ;;;  avoid too much
            dum = b(k)                     ;;;  parameter passing.
            b(k) = b(i)
            b(i) = dum
        ENDIF
        IF (l LT nm1) THEN l = l+1
        FOR i = k+1, l DO b(i) = b(i)-al(k, i-k-1)*b(k)
    ENDFOR
    l = 0
    FOR i = nm1, 0, -1 DO BEGIN
        dum = b(i)
        FOR k = 1, l DO dum = dum-a(i, k)*b(k+i)
        b(i) = dum/a(i, 0)
        IF (l LT 6) THEN l = l+1
    ENDFOR
    sm_dat(*, row) = b
ENDFOR


;;;
;;;  If the data had been transposed (either due to SMOOTHDIM setting 
;;;  or because data was a coloumn vector) now reverse the operation to
;;;  keep the data format on return.
;;;
IF tra EQ 1 THEN sm_dat = transpose(sm_dat)

return, sm_dat

END


