FUNCTION Rescale, image, xsize, ysize, scale=scale, SAMPLE=sample
;+
; NAME:
;       RESCALE
; PURPOSE:
;       Expand or shrink a given (1- or 2-dimensional) Array to new size
; CATEGORY:
;       
; CALLING SEQUENCE:
;       RESULT = RESCALE( IMAGE, [[ XSIZE [, YSIZE]] || SCALE=Scale] )
; INPUTS:
;       IMAGE : Original Image
;       XSIZE : New dimension for the X-axis
;       YSIZE : 1-d case: If ysize is given, a pseudo-2d Array of
;               identical rows is returned, else a 1-dim Array of size
;               xsize .
;               2-d case: If ysize is given it is taken as the new
;               dimension for the Y-axis. If omitted, the Y-axis is
;               scaled with the same factor as the X-axis to keep the
;               aspect ratio
; KEYWORDS:
;       SCALE : (input) 1- or 2-element vector with the scales for x
;               and y axis. Overrides YSIZE and YSIZE.
;       SAMPLE: (Flag) Use nearest neighbour sampling for the 2D-
;               interpolation (default is linear interpolation)
; OUTPUTS:
;       RESULT: Rescaled array of same type as IMAGE, size is
;               (XSIZE,YSIZE)                 (1/2-d case, YSIZE given)
;               (XSIZE,YOrig*XOrig/XSIZE)     (2-d case, YSIZE omitted)
;               (XSIZE)                       (1-d case, YSIZE omitted)
; PROCEDURE:
;       Scale-Factors are computed from the Size of the original data
;       and the parameters XSIZE and YSIZE. For 1-dim case SPLINE is
;       used for interpolation, in 2-d case POLY_2D is used.
; MODIFICATION HISTORY:
;       19-Feb-1993  P.Suetterlin, KIS
;       24-Feb-2009  Don't change submitted variable in keyword scale
;-

on_error, 2                                  ; Return on error

IF n_params() LT 2 AND NOT keyword_set(scale) THEN BEGIN
    message, 'Usage:  result = rescale( Image, XSize [, YSize] )'
ENDIF

IF keyword_set(sample) THEN interp = 0 ELSE interp = 1

s=size(image)
IF s(s(0)+1) EQ 8 THEN BEGIN    ; Image is ST4-Structure?
    pic = image.pic             ; Only for my personal
                                ; use
    s = size(pic)
ENDIF ELSE pic = image

IF keyword_set(scale) THEN BEGIN
      ;;; scale overrides xsize/ysize
    xsize = s(1)*scale(0)
    IF n_elements(scale) EQ 2 THEN $
      ysize = s(2)*scale(1) $
    ELSE $
      ysize = s(2)*scale(0)
ENDIF

IF s(0) EQ 1 THEN BEGIN                      ; Image is 1-d Array
    line = spline(indgen(s(1)), pic, $
                  findgen(xsize)/xsize*s(1))
    IF n_params() EQ 3 THEN BEGIN            ; Expand to 2-dim
        scalepic = fltarr(xsize, ysize)
        FOR i = 0, ysize-1 DO scalepic(*, i) = line
        return, scalepic                     ; Return 2-dim
    ENDIF ELSE return, line                  ; Return 1-dim
ENDIF ELSE BEGIN
    IF n_params() LT 3 AND NOT keyword_set(scale) THEN  $  ; Rescale Y by same
       ysize = fix(s(2)*(float(xsize)/s(0))+.5)            ; amount
    P = [[0, 0], [float(s(1))/xsize, 0]]     ; Matrices for
    Q = [[0, float(s(2))/ysize], [0, 0]]     ; Poly_2d
    return, poly_2d(pic, P, Q, interp, xsize, ysize)
ENDELSE
END


