PRO Mkmov, mov, files, img, ANGLES=ang0, CLIP16=clip16, FILE=file, $
           NSUM=nsum, SMOOTH=sm, CORIMG=cimg, ROI=roi, SCALE=scl, $
           POLY=poly, SHIFTS=sh, BIN=bin, PLOTOUT=plotout
;+
; NAME:
;       MKMOV
; PURPOSE:
;       Create a 3-D array of aligned images of a speckle time series
; CALLING SEQUENCE:
;       Mkmov, Mov, Files [, Img]
; INPUTS:
;       Mov:    Name of a variable that will hold the array
;       Files:  Regexp for the filenames (FITS files) to read
;       Img:    0-2, index of image to use.
;                    0: Reconstruction
;                    1: Best frame
;                    2: Average
; OPTIONAL PARAMETERS:
;       
; KEYWORDS:
;       ANGLES:  (float) array of same number of elements as there are
;                images to align.  Gives rotation angles in case of
;                telescopes with field rotation.
;       BIN:     (integer) reduce image size by binning
;       CLIP16:  (flag) when set, the resulting image size will be
;                clipped to a multiple of 16 to keep MPEG encoders happy.
;       CORIMG:  (integer) index of image to use for correlation (see
;                input Img). E.g., use the average for computing
;                shifts (corimg=2) when doing a movie of the
;                reconstructed frames (img=0)
;       FILE:    (string) use assoc and a disc file to hold the result.
;                FILE will be the base of the filename: FILE_XSIZExYSIZE
;       NSUM:    (integer) number of images to sum for reference.  The
;                last NSUM frames will be averaged and used for
;                computing the image shift.  Default: 1
;       POLY:    (integer) passed to shc: subtract polynomial fit of
;                degree POLY
;       ROI:     (integer) 4-element array with box coordinate of the
;                image part to be used for the correlation computation.
;       SHIFTS:  (integer) pass back computed shifts
;       SMOOTH:  (integer) smooth images with boxcar of width SMOOTH
;                before computing the correlation
;       PLOTOUT: put images on screen (1) or not (0)
; OUTPUTS:
;       
; RESTRICTIONS:
;       
; PROCEDURE:
;       
; MODIFICATION HISTORY:
;       20-Oct-1999  P.Suetterlin, SIU
;       16-Mar-2001  Take out explicit dimensions in cross-correlation
;       02-Jun-2001  Use assoc for large movies
;       04-Mar-2002  Add smoothing of images
;       16-Jan-2003  index error when img not 0; take out /swap from
;                    rdfits call (handeled automately)
;       21-Jan-2003  add CORIMG keyword.
;       12-Sep-2003  add scaling for images (via ANGLES)
;                    be smarter with filenames
;       05-Dec-2003  ROI definition for correlation computation
;                    fixed error: ref image was integer, but read-in
;                    fits files float with avg(1) -> nonsense!
;       20-Apr-2004  keyword POLY
;       14-Jul-2005  keywords SHIFT and BIN
;       Dec 10 2010  keyword plotout (RR)
;-

  ;;; Default is to use first image (=reko, also works for 2D-FITS files)
IF n_params() LT 3 THEN img = 0
IF NOT keyword_set(nsum) THEN nsum = 1
IF n_elements(cimg) EQ 0 THEN cimg = img
IF NOT keyword_set(scl) THEN scl = 5000.
IF NOT keyword_set(bin) THEN bin = 1
IF n_elements(plotout) EQ 0 THEN plotout = 0

  ;;; ang also allows to scale, if 2 parameters are submitted.  if
  ;;; not, set scaling to 1.
IF keyword_set(ang0) THEN BEGIN
    s = size(reform(ang0))
    IF s(0) EQ 1 THEN BEGIN
        ang = fltarr(2, s(1))
        ang(0, *) = ang0
        ang(1, *) = 1.
    ENDIF ELSE BEGIN
        ang = ang0
    ENDELSE
ENDIF
    
  ;;; determine needed sizes
IF n_elements(files) GT 1 THEN $
  f=files $
ELSE $
  f = findfile(files)
n = n_elements(f)
  ;;; arrays to hold shifts and image sizes
sh = intarr(2, n)
ss = intarr(2, n)

  ;;; read first image
rdfits, p1, f(0)
  ;;; derotate?
IF keyword_set(ang) THEN BEGIN
    p1 = rot(reform(p1(*, *, cimg)), ang(0, 0), ang(1, 0), cubic=-0.5)
    p1 = bestarea(p1, ang(0))
ENDIF ELSE BEGIN
    p1 = reform(p1(*, *, cimg))
ENDELSE
IF bin GT 1 THEN BEGIN
    ts = (size(p1))(1:2)/bin
    p1 = rebin(p1(0:bin*ts(0)-1, 0:bin*ts(1)-1), ts(0), ts(1))
ENDIF
IF keyword_set(sm) THEN p1 = smooth(p1, sm, /edge)

ss(*, 0) = (size(p1))(1:2)

IF keyword_set(roi) THEN BEGIN
    p = fltarr(nsum, roi(2)-roi(0)+1, roi(3)-roi(1)+1)
    p(0, *, *) = p1(roi(0):roi(2), roi(1):roi(3))
ENDIF ELSE BEGIN
    p = fltarr(nsum, ss(0, 0), ss(1, 0))
    p(0, *, *) = p1
ENDELSE

  ;;; read all other images.  compute shift.
FOR i=1, n-1 DO BEGIN
    rdfits, p1, f(i)
    IF keyword_set(ang) THEN BEGIN
        p1 = rot(reform(p1(*, *, cimg)), ang(0, i), ang(1, i), cubic=-0.5)
        p1 = bestarea(p1, ang(i))
    ENDIF ELSE BEGIN
        p1 = reform(p1(*, *, cimg))
    ENDELSE
    
    IF bin GT 1 THEN BEGIN
        ts = (size(p1))(1:2)/bin
        p1 = rebin(p1(0:bin*ts(0)-1, 0:bin*ts(1)-1), ts(0), ts(1))
    ENDIF

    IF keyword_set(sm) THEN p1 = smooth(p1, sm, /edge)
    ss(*, i) = (size(p1))(1:2)
    IF keyword_set(roi) THEN BEGIN
        sh(*, i) = shc(total(p, 1), p1(roi(0):roi(2), $
                                       roi(1):roi(3)), /filt, /N2, POLY=poly)
        p(i MOD nsum, *, *) = $
          (shift(p1, sh(0, i), sh(1, i)))(roi(0):roi(2), roi(1):roi(3))
    ENDIF ELSE BEGIN
        sh(*, i) = shc(total(p, 1), p1, /filt, /N2, POLY=poly)
        sx = (ss(0, i) < ss(0, 0))
        sy = (ss(1, i) < ss(1, 0))
        p1 = p1(0:sx-1, 0:sy-1)
        p(i MOD nsum, 0:sx-1, 0:sy-1) = shift(p1, sh(0, i), sh(1, i))
    ENDELSE
    if (plotout eq 1) then tvscl, p1(*, *)
    print, i, sh(*, i)
ENDFOR

sx = max(sh(0, *))-sh(0, *)
sy = max(sh(1, *))-sh(1, *)

resx = min(ss(0, *)-sx)
resy = min(ss(1, *)-sy)

IF resx/2 NE resx/2. THEN resx = resx-1
IF resy/2 NE resy/2. THEN resy = resy-1

IF keyword_set(clip16) THEN BEGIN
    resx = resx/16*16
    resy = resy/16*16
ENDIF

print, 'Size is ', string(resx, resy, f="(i4,'x',i3)")

if (plotout eq 1) then erase ;RR 

IF keyword_set(file) THEN BEGIN
    openw, unit, file+'_'+strtrim(resx, 2)+'x'+strtrim(resy, 2), /get
    mov = assoc(unit, intarr(resx, resy))
ENDIF ELSE $
  mov=intarr(resx, resy, n)

FOR i=0, n-1 DO BEGIN
    rdfits, p, f(i)
    
    IF keyword_set(ang) THEN BEGIN
        p = rot(reform(p(*, *, img)), ang(0, i), ang(1, i), cubic=-0.5)
        p = bestarea(p, ang(i))
    ENDIF ELSE BEGIN
        p = reform(p(*, *, img))
    ENDELSE
    IF bin GT 1 THEN BEGIN
        ts = (size(p))(1:2)/bin
        p = rebin(p(0:bin*ts(0)-1, 0:bin*ts(1)-1), ts(0), ts(1))
    ENDIF
    tmp = p(sx(i):sx(i)+resx-1, $
            sy(i):sy(i)+resy-1)
    
    IF keyword_set(file) THEN $
      mov(i)=fix((scl*tmp/avg(tmp)) < 32767) $
    ELSE $
      mov(*, *, i)=fix((scl*tmp/avg(tmp)) < 32767)
    
    if (plotout eq 1) then tvscl, tmp   ;RR
ENDFOR

IF keyword_set(file) THEN BEGIN
    free_lun, unit
    mov = [resx, resy, n]
ENDIF

END
